package cn.elwy.common.util;

import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

import org.dom4j.Attribute;
import org.dom4j.Document;
import org.dom4j.DocumentException;
import org.dom4j.DocumentHelper;
import org.dom4j.Element;
import org.dom4j.io.OutputFormat;
import org.dom4j.io.SAXReader;
import org.dom4j.io.XMLWriter;

import cn.elwy.common.util.io.XmlBean;

/**
 * Dom4j操作XML文件工具类
 * @author huangsq
 * @version 1.0, 2018-02-19
 */
@SuppressWarnings("unchecked")
public final class XmlUtil {

	private XmlUtil() {
	}

	/**
	 * 读取XML文件
	 * @param filePath 文件路径
	 * @return
	 * @throws DocumentException
	 */
	public static Document loadXmlFile(String filePath) throws DocumentException {
		return loadXmlFile(new File(filePath));
	}

	/**
	 * 读取XML文件
	 * @param file 文件
	 * @return
	 * @throws DocumentException
	 */
	public static Document loadXmlFile(File file) throws DocumentException {
		SAXReader reader = new SAXReader();
		return reader.read(file);
	}

	/**
	 * 读取XML文件
	 * @param in 输入流
	 * @return
	 * @throws DocumentException
	 */
	public static Document loadXmlFile(InputStream in) throws DocumentException {
		SAXReader reader = new SAXReader();
		return reader.read(in);
	}

	/**
	 * 写XML文件
	 * @param filePath 文件路径
	 * @param document XML对象
	 * @return
	 * @throws IOException
	 */
	public static boolean saveXml(String filePath, Document document) throws IOException {
		return saveXml(new File(filePath), document);
	}

	/**
	 * 写XML文件
	 * @param file 文件
	 * @param document XML对象
	 * @return
	 * @throws IOException
	 */
	public static boolean saveXml(File file, Document document) throws IOException {
		/* 格式化输出 */
		OutputFormat format = OutputFormat.createPrettyPrint(); // 紧缩
		format.setEncoding("UTF-8"); // 设置UTF-8编码
		// format.setLineSeparator("\r\n");
		return saveXml(file, document, format);
	}

	/**
	 * 写XML文件
	 * @param filePath 文件路径
	 * @param document XML对象
	 * @param format XML输出格式
	 * @return
	 * @throws IOException
	 */
	public static boolean saveXml(String filePath, Document document, OutputFormat format) throws IOException {
		return saveXml(new File(filePath), document, format);
	}

	/**
	 * 写XML文件
	 * @param file 文件
	 * @param document XML对象
	 * @param format XML输出格式
	 * @return
	 * @throws IOException
	 */
	public static boolean saveXml(File file, Document document, OutputFormat format) throws IOException {
		FileOutputStream fos = null;
		XMLWriter xmlWriter = null;
		try {
			file.getParentFile().mkdirs();
			fos = new FileOutputStream(file);
			xmlWriter = new XMLWriter(fos, format);
			xmlWriter.setEscapeText(false);
			xmlWriter.write(document);
			return true;
		} finally {
			close(xmlWriter);
			CloseUtil.close(fos);
		}
	}

	/**
	 * 关闭资源
	 * @param object 要关闭的对象
	 */
	public static void close(XMLWriter object) {
		try {
			if (object != null) {
				object.close();
			}
		} catch (Exception e) {
			e.printStackTrace();
		}
	}

	/**
	 * 创建一个空的XML文档
	 * @return
	 */
	public static Document createDocument() {
		// 使用DocumentHelper.createDocument方法建立一个文档实例
		return DocumentHelper.createDocument();
	}

	/**
	 * 获取子元素的字符串值
	 * @param parent 父元素节点
	 * @param elementName 元素名称
	 * @return
	 */
	public static String getTextStringValue(Element parent, String elementName) {
		return getTextStringValue(parent, elementName, "");
	}

	/**
	 * 获取子元素的字符串值
	 * @param parent 父元素节点
	 * @param elementName 元素名称
	 * @param defaultValue 缺省值
	 * @return
	 */
	public static String getTextStringValue(Element parent, String elementName, String defaultValue) {
		Element element = parent.element(elementName);
		if (element == null) {
			return defaultValue;
		} else {
			return element.getTextTrim();
		}
	}

	/**
	 * 获取子元素的布尔值
	 * @param parent 父元素节点
	 * @param elementName 元素名称
	 * @return
	 */
	public static boolean getTextBooleanValue(Element parent, String elementName) {
		return getTextBooleanValue(parent, elementName, false);
	}

	/**
	 * 获取子元素的布尔值
	 * @param parent 父元素节点
	 * @param elementName 元素名称
	 * @param defaultValue 缺省值
	 * @return
	 */
	public static boolean getTextBooleanValue(Element parent, String elementName, boolean defaultValue) {
		String value = getTextStringValue(parent, elementName, null);
		if (value == null) {
			return defaultValue;
		} else {
			return ConvertTypeUtil.toBoolean(value);
		}
	}

	/**
	 * 获取子元素的整型值
	 * @param parent 父元素节点
	 * @param elementName 元素名称
	 * @return
	 */
	public static int getTextIntValue(Element parent, String elementName) {
		return getTextIntValue(parent, elementName, 0);
	}

	/**
	 * 获取子元素的整型值
	 * @param parent 父元素节点
	 * @param elementName 元素名称
	 * @param defaultValue 缺省值
	 * @return
	 */
	public static int getTextIntValue(Element parent, String elementName, int defaultValue) {
		String value = getTextStringValue(parent, elementName, null);
		if (value == null) {
			return defaultValue;
		} else {
			return ConvertTypeUtil.toInteger(value);
		}
	}

	/**
	 * 设置或添加属性
	 * @param element
	 * @param name
	 * @param value
	 */
	public static void setAttribute(Element element, String name, Object value) {
		setAttribute(element, name, ConvertTypeUtil.toString(value));
	}

	/**
	 * 设置或添加属性
	 * @param element
	 * @param name
	 * @param value
	 */
	public static void setAttribute(Element element, String name, String value) {
		Attribute attribute = element.attribute(name);
		if (attribute != null) {
			attribute.setValue(value);
		} else {
			element.addAttribute(name, value);
		}
	}

	/**
	 * 设置元素的内容
	 * @param element
	 * @param value
	 */
	public static void setText(Element element, Object value) {
		setText(element, ConvertTypeUtil.toString(value));
	}

	/**
	 * 设置元素的内容
	 * @param element
	 * @param value
	 */
	public static void setText(Element element, String value) {
		if (element != null && value != null) {
			element.setText(value);
		}
	}

	/**
	 * 设置或添加子元素
	 * @param element
	 * @param childName
	 */
	public static Element setElementChild(Element element, String childName) {
		Element child = element.element(childName);
		if (child == null) {
			child = element.addElement(childName);
		}
		return child;
	}

	/**
	 * 设置或添加子元素
	 * @param element
	 * @param childName
	 * @param childValue
	 */
	public static Element setElementChild(Element element, String childName, String childValue) {
		Element child = element.element(childName);
		if (child != null) {
			child.setText(childValue);
		} else {
			child = element.addElement(childName, childValue);
		}
		return child;
	}

	public static List<XmlBean> getBean(Document document) {
		List<Element> elementList = document.getRootElement().elements();
		List<XmlBean> nodeList = new ArrayList<XmlBean>();
		return getBean(elementList, nodeList);
	}

	public static List<XmlBean> getBean(List<Element> elementList, List<XmlBean> nodeList) {
		for (Element element : elementList) {
			XmlBean node = getXmlBean(element);
			nodeList.add(node);
		}
		return nodeList;
	}

	public static XmlBean getXmlBean(Element element) {
		XmlBean xmlBean = new XmlBean(element.getName(), element.getText());

		List<Attribute> attributeList = element.attributes();
		if (attributeList.size() > 0) {
			Map<String, Object> attributeMap = new HashMap<String, Object>();
			for (Attribute attribute : attributeList) {
				attributeMap.put(attribute.getName(), attribute.getValue());
			}
			xmlBean.setAttribute(attributeMap);
		}

		List<Element> childList = element.elements();
		if (childList.size() > 0) {
			List<XmlBean> children = new ArrayList<XmlBean>();
			children = getBean(childList, children);
			xmlBean.setChildren(children);
		}

		return xmlBean;
	}

	/**
	 * 创建XML文件
	 * @return
	 */
	private static Document testCreateXMLFile() {
		// 使用DocumentHelper.createDocument方法建立一个文档实例
		Document document = DocumentHelper.createDocument();
		// 使用addElement方法方法创建根元素
		Element catalogElement = document.addElement("catalog");
		// 使用addComment方法方法向catalog元素添加注释
		catalogElement.addComment("使用addComment方法方法向catalog元素添加注释");
		// 使用addProcessInstruction向catalog元素增加处理指令
		catalogElement.addProcessingInstruction("target", "text");

		// 使用addElement方法向catalog元素添加journal子元素
		Element journalElement = catalogElement.addElement("journal");
		// 使用addAttribute方法向journal元素添加title和publisher属性

		journalElement.addAttribute("title", "Dom4j Create XML Schema");
		journalElement.addAttribute("publisher", "Willpower Co");

		// 使用addElement方法向journal元素添加article子元素
		Element articleElement = journalElement.addElement("article");
		// 使用addAttribute方法向article元素添加level和date属性
		articleElement.addAttribute("level", "Intermediate");
		articleElement.addAttribute("date", "July-2006");

		// 使用addElement方法向article元素添加title子元素
		Element urlElement = articleElement.addElement("url");
		urlElement.addCDATA(
				"?useUnicode=true&characterEncoding=utf8&autoReconnect=true&failOverReadOnly=false&useServerPrepStmts=false&emulateUnsupportedPstmts=false&useOldAliasMetadataBehavior=true");
		Element urlElement1 = articleElement.addElement("url1");
		urlElement1.addCDATA(
				"jdbc:mysql://192.168.103.153:3306/sofa?useUnicode=true&amp;characterEncoding=utf8&amp;autoReconnect=true&amp;failOverReadOnly=false&amp;useServerPrepStmts=false&amp;emulateUnsupportedPstmts=false&amp;useOldAliasMetadataBehavior=true");
		Element urlElement2 = articleElement.addElement("url2");
		urlElement2.addAttribute("url2",
				"?useUnicode=true&characterEncoding=utf8&autoReconnect=true&failOverReadOnly=false&useServerPrepStmts=false&emulateUnsupportedPstmts=false&useOldAliasMetadataBehavior=true");
		Element urlElement3 = articleElement.addElement("url3");
		urlElement3.addAttribute("url3",
				"jdbc:mysql://192.168.103.153:3306/sofa?useUnicode=true&amp;characterEncoding=utf8&amp;autoReconnect=true&amp;failOverReadOnly=false&amp;useServerPrepStmts=false&amp;emulateUnsupportedPstmts=false&amp;useOldAliasMetadataBehavior=true");

		Element titleElement = articleElement.addElement("title");
		// 使用setText方法设置title子元素的值
		titleElement.setText("Dom4j Create XML Schema");

		// 使用addElement方法向article元素添加authorElement子元素
		Element authorElement = articleElement.addElement("author");

		// 使用addElement方法向author元素添加firstName子元素
		Element firstName = authorElement.addElement("fistname");
		// 使用setText方法设置firstName子元素的值
		firstName.setText("Yi");

		// 使用addElement方法向author元素添加lastname子元素
		Element lastName = authorElement.addElement("lastname");
		// 使用setText方法设置lastName子元素的值
		lastName.setText("Qiao");

		return document;
	}

	/**
	 * 修改XML文件
	 * @param fileName
	 * @param newFileName
	 * @return
	 */
	private static Document testModifyXMLFile(String fileName) {
		SAXReader reader = new SAXReader();
		Document document = null;
		try {
			document = reader.read(new File(fileName));
			// 用xpath查找对象
			List<?> list = document.selectNodes("/catalog/journal/@title");
			Iterator<?> itr = list.iterator();
			while (itr.hasNext()) {
				Attribute attribute = (Attribute) itr.next();
				if (attribute.getValue().equals("XML Zone")) {
					attribute.setText("Modi XML");
				}
			}
			// 在journal元素中增加date元素
			list = document.selectNodes("/catalog/journal");
			itr = list.iterator();
			if (itr.hasNext()) {
				Element journalElement = (Element) itr.next();
				Element dateElement = journalElement.addElement("date");
				dateElement.setText("2006-07-10");
				dateElement.addAttribute("type", "Gregorian calendar");
			}
			// 删除title接点
			list = document.selectNodes("/catalog/journal/article");
			itr = list.iterator();
			while (itr.hasNext()) {
				Element articleElement = (Element) itr.next();

				// 使用addElement方法向article元素添加title子元素
				Element urlElement = (Element) articleElement.selectSingleNode("url");
				Element urlElement1 = (Element) articleElement.selectSingleNode("url1");
				Element urlElement2 = (Element) articleElement.selectSingleNode("url2");
				Element urlElement3 = (Element) articleElement.selectSingleNode("url3");

				String url = urlElement.getText();
				String url1 = urlElement1.getText();
				String url2 = urlElement2.attributeValue("url2");
				String url3 = urlElement3.attributeValue("url3");

				urlElement.clearContent();
				urlElement1.clearContent();

				urlElement.addCDATA("edit:" + url);
				urlElement1.addCDATA("edit:" + url1);
				urlElement2.addAttribute("url2", "edit:" + url2);
				urlElement3.addAttribute("url3", "edit:" + url3);

				Iterator<Element> iter = articleElement.elementIterator("title");
				while (iter.hasNext()) {
					Element titleElement = (Element) iter.next();
					if (titleElement.getText().equals("Dom4j Create XML Schema")) {
						articleElement.remove(titleElement);
					}
				}
			}

		} catch (DocumentException e) {
			e.printStackTrace();
		}

		return document;

	}

	/**
	 * 递归显示文档内容
	 * @param els elements数组
	 */
	private static void print(List<Element> els) {
		for (Element el : els) {
			System.out.println("##" + el.getName() + "--value==" + el.getTextTrim());
			if (el.hasContent()) {
				print(el.elements());
			}
		}
	}

	public static void main(String[] args) throws IOException {
		dom4jTest("d:/temp");
	}

	private static void dom4jTest(String path) throws IOException {
		Document document = testCreateXMLFile();
		String filePath = path + "/dom4j_create.xml";
		saveXml(filePath, document);

		document = testModifyXMLFile(filePath);
		String modifyPath = path + "/dom4j_modify.xml";
		saveXml(modifyPath, document);

		// 读取xml文档
		Document doc;
		try {
			doc = loadXmlFile(filePath);
			List<Element> listEl = doc.getRootElement().elements();

			print(listEl);
		} catch (DocumentException e) {
			e.printStackTrace();
		}
	}

}
