package cn.elwy.common.util;

import java.util.Comparator;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

/**
 * 比较版本号是否相等工具类
 * @author huangsq
 * @version 1.0, 2018-02-19
 */
public final class VersionUtil {

	private static final String SPLIT_REGEX = "-|_";

	private VersionUtil() {
	}

	/**
	 * 获取文件的名称，不包含版本号信息
	 * @param fileName 包含扩展名的文件名
	 * @return
	 */
	public static String getFileName(String fileName) {
		int index = fileName.lastIndexOf(".");
		String extName = (index == -1) ? "" : fileName.substring(index + 1);
		String text = (index < 0) ? fileName : (index == 0) ? "" : fileName.substring(0, index);

		String name = getName(text);
		return name + "." + extName;
	}

	/**
	 * 获取文件的版本号，不包含扩展名信息
	 * @param fileName 包含扩展名的文件名
	 * @return
	 */
	public static String getFileVersion(String fileName) {
		return getFileVersion(fileName, SPLIT_REGEX);
	}

	/**
	 * 获取文件的版本号，不包含扩展名信息
	 * @param fileName 包含扩展名的文件名
	 * @param splitRegex 分割版本号的正则表达式
	 * @return
	 */
	public static String getFileVersion(String fileName, String splitRegex) {
		int index = fileName.lastIndexOf(".");
		String text = (index < 0) ? fileName : (index == 0) ? "" : fileName.substring(0, index);
		return getVersion(text, splitRegex);
	}

	/**
	 * 获取文件的名称，不包含版本号和扩展名信息
	 * @param text 带版本号的字符串
	 * @return
	 */
	public static String getName(String text) {
		return getName(text, SPLIT_REGEX);
	}

	/**
	 * 获取文件的名称，不包含版本号和扩展名信息
	 * @param text 带版本号的字符串
	 * @param splitRegex 分割版本号的正则表达式
	 * @return
	 */
	public static String getName(String text, String splitRegex) {
		String version = getVersion(text, splitRegex);
		if (text.length() > version.length()) {
			String name = text.substring(0, text.length() - version.length());
			return name.replaceAll("[-_]+$", "");
		}
		return "";
	}

	/**
	 * 解析版本号，过滤掉版本号前缀
	 * @param text 带版本号的字符串
	 * @return
	 */
	public static String getVersion(String text) {
		return getVersion(text, SPLIT_REGEX);
	}

	/**
	 * 解析版本号，过滤掉版本号前缀
	 * @param text 带版本号的字符串
	 * @param splitRegex 分割版本号的正则表达式
	 * @return
	 */
	public static String getVersion(String text, String splitRegex) {
		String[] split = text.split(splitRegex);
		String v = split[split.length - 1];
		boolean startsWith = v.substring(0, 1).matches("^[0-9]");
		String version = v;
		if (!startsWith) {
			version = v.replaceFirst("[^0-9]+", "");
		}
		return version;
	}

	/**
	 * 版本号自增，如果自增失败返回null
	 * @param text 带版本号的字符串
	 * @return
	 */
	public static String versionAdd(String text) {
		return versionAdd(text, -1, 1);
	}

	/**
	 * 版本号自增，如果自增失败返回null
	 * @param text 带版本号的字符串
	 * @param index 版本级别,-1 表示尾部版本号,0表示头部版本号
	 * @param step 自增步长
	 * @return
	 */
	public static String versionAdd(String text, int index, int step) {
		Pattern pattern = Pattern.compile("(\\d+)", Pattern.CASE_INSENSITIVE);
		Matcher matcher = pattern.matcher(text);
		if (matcher.find()) {
			int start = matcher.start();
			String prefix = text.substring(0, start);
			String versionStr = text.substring(start, text.length());

			String[] numberStr = versionStr.split("\\.");
			if (index < 0 || index > numberStr.length) {
				index = numberStr.length - 1;
			}
			numberStr[index] = "" + (ConvertTypeUtil.toInteger(numberStr[index]) + step);
			for (int i = index + 1; i < numberStr.length; i++) {
				numberStr[i] = "0";
			}

			return prefix + ConvertTypeUtil.join(".", (Object[]) numberStr);
		}
		return null;
	}

	/**
	 * 比较版本号的大小,前者大则返回一个正数,后者大返回一个负数,相等则返回0
	 * @param version1
	 * @param version2
	 * @return
	 */
	public static int compare(String version1, String version2) {
		return compare(version1, version2, null);
	}

	/**
	 * 比较版本号的大小是否在指定的范围内
	 * @param version
	 * @param minVersion
	 * @param maxVersion
	 * @return
	 */
	public static boolean within(String version, String minVersion, String maxVersion) {
		return compare(version, minVersion, null) > 0 && compare(version, maxVersion, null) < 0;
	}

	/**
	 * 比较版本号的大小,前者大则返回一个正数,后者大返回一个负数,相等则返回0
	 * @param version1
	 * @param version2
	 * @param comparator 版本比较器
	 * @return
	 */
	public static int compare(String version1, String version2, Comparator<String> comparator) {
		if (version1 == null && version2 == null) {
			return 0;
		} else if (version1 == null) {
			return -1;
		} else if (version2 == null) {
			return 1;
		}
		String[] v1 = version1.split("\\.|-|_");
		String[] v2 = version2.split("\\.|-|_");
		int minLength = Math.min(v1.length, v2.length);// 取最小长度值
		int diff = 0;
		int i = 0;
		while (i < minLength && (diff = compareTo(v1[i], v2[i], comparator)) == 0) {// 再比较字符
			++i;
		}
		// 如果已经分出大小，则直接返回，如果未分出大小，则再比较位数，有子版本的为大；
		diff = (diff != 0) ? diff : v1.length - v2.length;
		return diff;
	}

	/**
	 * 比较版本号是否相等
	 * @param v1
	 * @param v2
	 * @param comparator
	 * @return
	 */
	protected static int compareTo(String v1, String v2, Comparator<String> comparator) {
		int diff = 0;
		if ((diff = v1.length() - v2.length()) != 0) {
			return diff;
		} else if (comparator != null) {
			return comparator.compare(v1, v2);
		} else {
			return v1.compareTo(v2);
		}
	}

}
