package cn.elwy.common.util;

import java.util.regex.Matcher;
import java.util.regex.Pattern;

/**
 * @author huangsq
 * @version 1.0, 2018-02-19
 */
public final class ValidateUtil {

	/** 匹配纯小写字母字符串 */
	public static final String LOWER_CASE = "^[a-z]+$";

	/** 匹配纯大写字母字符串 */
	public static final String UPPER_CASE = "^[A-Z]+$";

	/** 匹配纯字母字符串 */
	// public static final String CHAR = "[a-zA-Z]\\d*";
	public static final String CHAR = "^[A-Za-z]+$";

	/** 匹配纯数字字符串 */
	public static final String NUMBER = "[0-9]\\d*";

	/** 包含数字和字母 （最少输入一个数字和一个字母， 更改{2,20}中的值调整验证位数，试用于账户和密码验证） */
	public static final String NUMBER_AND_CHAR = "^(?![0-9]+$)(?![a-zA-Z]+$)[0-9A-Za-z]{2,20}$";

	/** 匹配由数字、26个英文字母或者下划线组成的字符串 ^\w+$ */
	public static final String COMPLEX = "^\\w+$";

	/** Integer正则表达式 ^-?(([1-9]\d*$)|0) */
	public static final String INTEGER = "^-?(([1-9]\\d*$)|0)";

	/** 正整数正则表达式 >=0 ^[1-9]\d*|0$ */
	public static final String INTEGER_POSITIVE = "^[1-9]\\d*|0$";

	/** 负整数正则表达式 <=0 ^-[1-9]\d*|0$ */
	public static final String INTEGER_NEGATIVE = "^-[1-9]\\d*|0$";

	/** Double正则表达式 ^-?([1-9]\d*\.\d*|0\.\d*[1-9]\d*|0?\.0+|0)$ */
	public static final String DOUBLE = "^-?([1-9]\\d*\\.\\d*|0\\.\\d*[1-9]\\d*|0?\\.0+|0)$";

	/** 正Double正则表达式 >=0 ^[1-9]\d*\.\d*|0\.\d*[1-9]\d*|0?\.0+|0$ */
	public static final String DOUBLE_POSITIVE = "^[1-9]\\d*\\.\\d*|0\\.\\d*[1-9]\\d*|0?\\.0+|0$";

	/** 负Double正则表达式 <= 0 ^(-([1-9]\d*\.\d*|0\.\d*[1-9]\d*))|0?\.0+|0$ */
	public static final String DOUBLE_NEGATIVE = "^(-([1-9]\\d*\\.\\d*|0\\.\\d*[1-9]\\d*))|0?\\.0+|0$";

	/**
	 * 电话号码正则表达式=
	 * (^(\d{2,4}[-_－—]?)?\d{3,8}([-_－—]?\d{3,8})?([-_－—]?\d{1,7})?$)|(^0?1[35]\d{9}$)
	 */
	public static final String PHONE = "(^(\\d{2,4}[-_－—]?)?\\d{3,8}([-_－—]?\\d{3,8})?([-_－—]?\\d{1,7})?$)|(^0?1[35]\\d{9}$)";

	/** 手机号码正则表达式=^(13[0-9]|15[0|3|6|7|8|9]|18[8|9])\d{8}$ */
	public static final String MOBILE = "^(13[0-9]|15[0|3|6|7|8|9]|18[8|9])\\d{8}$";

	/** 邮编正则表达式 [1-9]\d{5}(?!\d) 国内6位邮编 */
	public static final String CODE = "[1-9]\\d{5}(?!\\d)";

	/** Email正则表达式=^\w+([-+.]\w+)*@\w+([-.]\w+)*\.\w+([-.]\w+)*$ */
	public static final String EMAIL = "^([a-zA-Z0-9_\\-\\.]+)@((\\[[0-9]{1,3}\\.[0-9]{1,3}\\.[0-9]{1,3}\\.)|(([a-zA-Z0-9\\-]+\\.)+))([a-zA-Z]{2,4}|[0-9]{1,3})(\\]?)$";

	/**
	 * IP地址正则表达式
	 * ((?:(?:25[0-5]|2[0-4]\\d|[01]?\\d?\\d)\\.){3}(?:25[0-5]|2[0-4]\\d|[01]?\\d?\\d))
	 */
	public static final String IPADDRESS = "((?:(?:25[0-5]|2[0-4]\\d|[01]?\\d?\\d)\\.){3}(?:25[0-5]|2[0-4]\\d|[01]?\\d?\\d))";

	/** IP地址模糊匹配正则表达式 */
	public static final String IPADDRESS_BLUR = "^([01]?[0-9][0-9]|[01]?[0-9]?[0-9]|2[0-4][0-9]|25[0-5])\\.([01]?[0-9][0-9]|[01]?[0-9]?[0-9]|2[0-4][0-9]|25[0-5])\\.([01]?[0-9][0-9]|[01]?[0-9]?[0-9]|2[0-4][0-9]|25[0-5]|\\*)\\.([01]?[0-9][0-9]|[01]?[0-9]?[0-9]|2[0-4][0-9]|25[0-5]|\\*)$";

	/** 日期正则支持日期和时间 */
	public static final String DATE_TIME = "^((\\d{2}(([02468][048])|([13579][26]))[\\-\\/\\s]?((((0?[13578])|(1[02]))[\\-\\/\\s]?((0?[1-9])|([1-2][0-9])|(3[01])))|(((0?[469])|(11))[\\-\\/\\s]?((0?[1-9])|([1-2][0-9])|(30)))|(0?2[\\-\\/\\s]?((0?[1-9])|([1-2][0-9])))))|(\\d{2}(([02468][1235679])|([13579][01345789]))[\\-\\/\\s]?((((0?[13578])|(1[02]))[\\-\\/\\s]?((0?[1-9])|([1-2][0-9])|(3[01])))|(((0?[469])|(11))[\\-\\/\\s]?((0?[1-9])|([1-2][0-9])|(30)))|(0?2[\\-\\/\\s]?((0?[1-9])|(1[0-9])|(2[0-8]))))))(\\s(((0?[0-9])|([1-2][0-3]))\\:([0-5]?[0-9])((\\s)|(\\:([0-5]?[0-9])))))?$";

	/*** 日期正则 支持： YYYY-MM-DD YYYY/MM/DD YYYY_MM_DD YYYYMMDD YYYY.MM.DD的形式 */
	public static final String DATE_ALL = "((^((1[8-9]\\d{2})|([2-9]\\d{3}))([-\\/\\._]?)(10|12|0?[13578])([-\\/\\._]?)(3[01]|[12][0-9]|0?[1-9])$)"
			+ "|(^((1[8-9]\\d{2})|([2-9]\\d{3}))([-\\/\\._]?)(11|0?[469])([-\\/\\._]?)(30|[12][0-9]|0?[1-9])$)"
			+ "|(^((1[8-9]\\d{2})|([2-9]\\d{3}))([-\\/\\._]?)(0?2)([-\\/\\._]?)(2[0-8]|1[0-9]|0?[1-9])$)|(^([2468][048]00)([-\\/\\._]?)(0?2)([-\\/\\._]?)(29)$)|(^([3579][26]00)"
			+ "([-\\/\\._]?)(0?2)([-\\/\\._]?)(29)$)"
			+ "|(^([1][89][0][48])([-\\/\\._]?)(0?2)([-\\/\\._]?)(29)$)|(^([2-9][0-9][0][48])([-\\/\\._]?)"
			+ "(0?2)([-\\/\\._]?)(29)$)"
			+ "|(^([1][89][2468][048])([-\\/\\._]?)(0?2)([-\\/\\._]?)(29)$)|(^([2-9][0-9][2468][048])([-\\/\\._]?)(0?2)"
			+ "([-\\/\\._]?)(29)$)|(^([1][89][13579][26])([-\\/\\._]?)(0?2)([-\\/\\._]?)(29)$)|"
			+ "(^([2-9][0-9][13579][26])([-\\/\\._]?)(0?2)([-\\/\\._]?)(29)$))";

	/** 日期正则 支持： YYYY-MM-DD */
	public static final String DATE_FORMAT1 = "(([0-9]{3}[1-9]|[0-9]{2}[1-9][0-9]{1}|[0-9]{1}[1-9][0-9]{2}|[1-9][0-9]{3})-(((0[13578]|1[02])-(0[1-9]|[12][0-9]|3[01]))|((0[469]|11)-(0[1-9]|[12][0-9]|30))|(02-(0[1-9]|[1][0-9]|2[0-8]))))|((([0-9]{2})(0[48]|[2468][048]|[13579][26])|((0[48]|[2468][048]|[3579][26])00))-02-29)";

	/**
	 * 过滤特殊字符串正则
	 * regEx="[`~!@#$%^&*()+=|{}':;',\\[\\].<>/?~！@#￥%……&*（）——+|{}【】‘；：”“’。，、？]";
	 */
	public static final String STR_SPECIAL = "[`~!@#$%^&*()+=|{}':;',\\[\\].<>/?~！@#￥%……&*（）——+|{}【】‘；：”“’。，、？]";

	/** URL正则表达式 匹配 http www ftp */
	public static final String URL = "^(http|www|ftp|)?(://)?(\\w+(-\\w+)*)(\\.(\\w+(-\\w+)*))*((:\\d+)?)(/(\\w+(-\\w+)*))*(\\.?(\\w)*)(\\?)?"
			+ "(((\\w*%)*(\\w*\\?)*(\\w*:)*(\\w*\\+)*(\\w*\\.)*(\\w*&)*(\\w*-)*(\\w*=)*(\\w*%)*(\\w*\\?)*"
			+ "(\\w*:)*(\\w*\\+)*(\\w*\\.)*" + "(\\w*&)*(\\w*-)*(\\w*=)*)*(\\w*)*)$";

	private ValidateUtil() {
	}

	/**
	 * 匹配字符是否满足条件
	 * @param data 要匹配的字符
	 * @param condition 正则表达式
	 * @return
	 */
	public static boolean checkMatcher(String text, String condition) {
		Pattern pattern = Pattern.compile(condition);
		Matcher matcher = pattern.matcher(text);
		return matcher.matches();
	}

	/** 匹配纯小写字母字符串 */
	public static boolean checkLowerCase(String text) {
		return checkMatcher(text, LOWER_CASE);
	}

	/** 匹配纯大写字母字符串 */
	public static boolean checkUpperCase(String text) {
		return checkMatcher(text, UPPER_CASE);
	}

	/** 匹配纯字母字符串 */
	public static boolean checkLetter(String text) {
		return checkMatcher(text, CHAR);
	}

	/** 匹配纯数字字符串 */
	public static boolean checkNumber(String text) {
		return checkMatcher(text, NUMBER);
	}

	/** 匹配必须同时有数字和字母的字符串 */
	public static boolean checkNumberAndChar(String text) {
		return checkMatcher(text, NUMBER_AND_CHAR);
	}

	/** 匹配由数字，字母，下划线组成的字符串 */
	public static boolean checkComplex(String text) {
		return checkMatcher(text, COMPLEX);
	}

	/** 验证IP地址格式 */
	public static boolean checkIP(String text) {
		return checkIP(text, "default");
	}

	/**
	 * 验证IP地址，分为模糊匹配和完全匹配
	 * @param data
	 * @param type blur为模糊匹配，否则为完全匹配
	 * @return
	 */
	public static boolean checkIP(String text, String type) {
		String condition = null;
		if ("blur".equals(type)) { // 模糊匹配
			condition = IPADDRESS_BLUR;
		} else { // 完全匹配
			condition = IPADDRESS;
		}
		return checkMatcher(text, condition);
	}

	/** 验证Email格式 */
	public static boolean checkEmail(String text) {
		return checkMatcher(text, EMAIL);
	}

	/** 验证日期时间 */
	public static boolean checkDateTime(String text) {
		Pattern pattern = Pattern.compile(DATE_TIME);
		Matcher matcher = pattern.matcher(text);
		return matcher.matches();
	}

	/**
	 * 判断字段是否为日期
	 * @param data
	 * @return boolean
	 */
	public static boolean checkDate(String text) {
		Pattern pattern = Pattern.compile(DATE_ALL);
		Matcher matcher = pattern.matcher(text);
		return matcher.matches();
	}

	/**
	 * 验证日期格式
	 * @param data
	 * @param format
	 *          format为"YYYY-DD-MM"时，只匹配此格式日期。format为"datetime"时，匹配时间和日期。否则为匹配所有的日期格式（不包含时间）
	 * @return
	 */
	public static boolean isDate(String text, String format) {
		String condition = null;
		if ("YYYY-DD-MM".equalsIgnoreCase(format)) {
			condition = DATE_FORMAT1;
		} else if ("datetime".equalsIgnoreCase(format)) {
			condition = DATE_TIME;
		} else {
			condition = DATE_ALL;
		}
		return checkMatcher(text, condition);
	}

	/** 验证电话号码 */
	public static boolean checkPhone(String text) {
		return checkMatcher(text, PHONE);
	}

	/** 验证手机号码 */
	public static boolean checkMobile(String text) {
		return checkMatcher(text, MOBILE);
	}

	/** 验证邮政编码 */
	public static boolean checkCode(String text) {
		return checkMatcher(text, CODE);
	}

	/** 验证整型 */
	public static boolean checkInteger(String text) {
		return checkInteger(text, null);
	}

	/**
	 * 验证整型
	 * @param data
	 * @param type +为正Integer类型， -为负Integer类型，无符号则为Integer类型
	 * @return
	 */
	public static boolean checkInteger(String text, String type) {
		String condition = null;
		if ("+".equals(type)) { // 正整数正则表达式
			condition = INTEGER_POSITIVE;
		} else if ("-".equals(type)) { // 负整数正则表达式
			condition = INTEGER_NEGATIVE;
		} else {
			condition = INTEGER;
		}
		return checkMatcher(text, condition);
	}

	/** 验证Double类型 */
	public static boolean checkDouble(String text) {
		return checkDouble(text, null);
	}

	/**
	 * 验证Double类型
	 * @param data
	 * @param type "+" 为正Double类型，"-"为负Double类型，无符号则为Double类型
	 * @return
	 */
	public static boolean checkDouble(String text, String type) {
		String condition = null;
		if ("+".equals(type)) { // 正Double正则表达式
			condition = DOUBLE_POSITIVE;
		} else if ("-".equals(type)) { // 负Double正则表达式
			condition = DOUBLE_NEGATIVE;
		} else { // Double正则表达式
			condition = DOUBLE;
		}
		return checkMatcher(text, condition);
	}

	/** 验证URL匹配 http www ftp */
	public static boolean checkUrl(String text) {
		return checkMatcher(text, URL);
	}

	/**
	 * 过滤特殊字符串 返回过滤后的字符串
	 * @param data
	 * @return boolean
	 */
	public static String filterStr(String text) {
		return filterStr(text, STR_SPECIAL);
	}

	/**
	 * 过滤特殊字符串 返回过滤后的字符串
	 * @param data
	 * @param filterText 过滤的字符串
	 * @return boolean
	 */
	public static String filterStr(String text, String filterText) {
		Pattern pattern = Pattern.compile(filterText);
		Matcher matcher = pattern.matcher(text);
		return matcher.replaceAll("").trim();
	}

	public static void main(String[] args) {
		String con = "[(^insert) | (\\sinsert) | (insert\\s) ]";
		System.out.println(checkMatcher("insert", con));
		System.out.println(checkMatcher("insert ss", con));
		System.out.println(checkMatcher("t insert", con));
		System.out.println(checkMatcher("t insert x", con));
		System.out.println(checkMatcher("tinsert", con));

		String a = "aaa";
		System.out.println("小写字母:" + checkLowerCase(a));
		String A = "SSDFSFD";
		System.out.println("小写字母:" + checkUpperCase(A));
		String letter = "abcABC";
		System.out.println("只有字母:" + checkLetter(letter));
		String num = "012340";
		System.out.println("只有数字:" + checkNumber(num));
		String numAndChar = "1abaaaabaaaabaaa";
		System.out.println("必须由字母和数字两种字符构成:" + checkNumberAndChar(numAndChar));
		String complex = "1bc_";
		System.out.println("允许有数字字母下划线:" + checkComplex(complex));
		String ip = "192.168.1.1";
		System.out.println("验证是否为IP:" + checkIP(ip, "blur"));
		System.out.println("checkIP:" + checkIP(ip));
		String email = "sdafasf";
		System.out.println("验证Email:" + checkEmail(email));
		String phone = "12345678901234";
		System.out.println("验证电话号码:" + checkPhone(phone));
		String mobile = "13810530117";
		System.out.println("验证手机号码:" + checkMobile(mobile));
		String code = "";
		System.out.println("验证邮编号码:" + checkCode(code));
		String integer = "-1";
		System.out.println("验证是否为整型:" + checkInteger(integer));
		String Double = "-1.1";
		System.out.println("验证小数：" + checkDouble(Double));
		String dateTime = "2012-12/3";
		System.out.println("验证日期时间:" + checkDateTime(dateTime));
		String date = "2012-12/03";
		System.out.println("验证日期:" + isDate(date, "YYYY-DD-MM"));
		String url = "http://182.50.4.68:8080/browse/JIANGS-271";
		System.out.println("验证URL:" + checkUrl(url));
	}
}
