package cn.elwy.common.util;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.ObjectInputStream;
import java.io.ObjectOutputStream;
import java.io.ObjectStreamClass;
import java.io.OutputStream;
import java.io.Serializable;
import java.util.Collection;
import java.util.HashMap;

import cn.elwy.common.exception.RunException;

/**
 * Java序列化工具类
 * @author huangsq
 * @version 1.0, 2018-02-19
 */
@SuppressWarnings("unchecked")
public final class SerializeUtil {

	private SerializeUtil() {
	}

	/**
	 * 清除本地的序列化文件
	 * @param file 序列化文件
	 * @return
	 */
	public static boolean clear(File file) {
		boolean cleared = true;
		if (file.exists()) {
			cleared = file.delete();
		}
		return cleared;
	}

	/**
	 * 克隆继承Serializable的对象
	 * @param object 序列化对象
	 */
	public static <T> T clone(Serializable object) {
		return (T) unSerialize(serialize(object), object.getClass());
	}

	/**
	 * 克隆继承Serializable的对象
	 * @param <T>
	 * @param object 序列化对象
	 */
	public static Collection<Serializable> clone(Collection<? extends Serializable> collections) {
		try {
			Collection<Serializable> newInstance = collections.getClass().newInstance();
			for (Serializable t : newInstance) {
				Serializable unSerialize = unSerialize(serialize(t), t.getClass());
				newInstance.add(unSerialize);
			}
			return newInstance;
		} catch (Exception e) {
			throw new RunException(e);
		}
	}

	/**
	 * 将对象序列化到字节数组
	 * @param object 序列化对象
	 */
	public static byte[] serialize(Serializable object) {
		ByteArrayOutputStream baos = null;
		try {
			baos = new ByteArrayOutputStream(512);
			serialize(object, baos);
			byte[] byteArray = baos.toByteArray();
			return byteArray;
		} finally {
			CloseUtil.close(baos);
		}
	}

	/**
	 * 将对象序列化到输出流
	 * @param object 序列化对象
	 * @param outputStream 输出流
	 */
	public static void serialize(Serializable object, OutputStream outputStream) {
		if (outputStream == null) {
			throw new IllegalArgumentException("The OutputStream must not be null");
		}
		ObjectOutputStream out = null;
		try {
			out = new ObjectOutputStream(outputStream);
			out.writeObject(object);
		} catch (IOException e) {
			throw new RuntimeException(e);
		} finally {
			CloseUtil.close(out);
		}
	}

	/**
	 * 将对象序列化到本地文件
	 * @param object 序列化对象
	 * @param file 序列化文件
	 */
	public static boolean serialize(final Serializable object, final File file) {
		boolean result = false;
		synchronized (object) {
			FileOutputStream out = null;
			try {
				file.getParentFile().mkdirs();
				out = new FileOutputStream(file.getAbsoluteFile());
				serialize(object, out);
				result = true;
			} catch (IOException e) {
				throw new RuntimeException(e);
			} finally {
				CloseUtil.close(out);
			}
		}
		return result;
	}

	/**
	 * 输入字节数反序列化对象
	 * @param objectData 输入字节数
	 * @param clazz 对象类型
	 * @return
	 */
	public static <T> T unSerialize(byte objectData[], Class<T> clazz) {
		if (objectData == null) {
			throw new IllegalArgumentException("The byte[] must not be null");
		}
		ByteArrayInputStream bais = null;
		try {
			bais = new ByteArrayInputStream(objectData);
			return unSerialize(bais, clazz);
		} finally {
			CloseUtil.close(bais);
		}
	}

	/**
	 * 输入流反序列化对象
	 * @param inputStream 输入流
	 * @param clazz 对象类型
	 * @return
	 */
	public static <T> T unSerialize(InputStream inputStream, Class<T> clazz) {
		if (inputStream == null) {
			throw new IllegalArgumentException("The InputStream must not be null");
		}
		SerializeObjectInputStream in = null;
		try {
			in = new SerializeObjectInputStream(inputStream, clazz.getClassLoader());
			Object readObject = in.readObject();
			return (T) readObject;
		} catch (Exception e) {
			throw new RuntimeException(e);
		} finally {
			CloseUtil.close(in);
		}
	}

	/**
	 * 从本地文件中反序列化对象
	 * @param file 序列化文件
	 * @param clazz 对象类型
	 * @return
	 */
	public static <T> T unSerialize(File file, Class<T> clazz) {
		FileInputStream out = null;
		Object serializedObject = null;
		try {
			if (file.exists()) {
				out = new FileInputStream(file.getAbsoluteFile());
				serializedObject = unSerialize(out, clazz);
			}
			return (T) serializedObject;
		} catch (Exception e) {
			throw new RuntimeException(e);
		} finally {
			CloseUtil.close(out);
		}
	}

}

/**
 * 反序列化对象流，解决找不到类问题
 * @author huangsq
 * @version 1.0
 */
class SerializeObjectInputStream extends ObjectInputStream {

	private static final HashMap<String, Class<?>> primClasses = new HashMap<String, Class<?>>(8, 1.0F);
	private ClassLoader classLoader;

	static {
		primClasses.put("boolean", boolean.class);
		primClasses.put("byte", byte.class);
		primClasses.put("char", char.class);
		primClasses.put("short", short.class);
		primClasses.put("int", int.class);
		primClasses.put("long", long.class);
		primClasses.put("float", float.class);
		primClasses.put("double", double.class);
		primClasses.put("void", void.class);
	}

	public SerializeObjectInputStream(InputStream in, ClassLoader classLoader) throws IOException {
		super(in);
		this.classLoader = classLoader;
	}

	@Override
	protected Class<?> resolveClass(ObjectStreamClass desc) throws IOException, ClassNotFoundException {
		String name = desc.getName();
		try {
			if (classLoader == null) {
				classLoader = Thread.currentThread().getContextClassLoader();
			}
			return Class.forName(name, false, classLoader);
		} catch (ClassNotFoundException ex) {
			Class<?> cl = (Class<?>) primClasses.get(name);
			if (cl != null) {
				return cl;
			} else {
				throw ex;
			}
		}
	}

}
