package cn.elwy.common.util;

import java.math.BigDecimal;
import java.util.HashMap;
import java.util.Map;
import java.util.Random;
import java.util.UUID;

/**
 * 进制转换工具类
 * @author huangsq
 * @version 1.0, 2018-02-19
 */
public final class NumberUtil {

	final static String CHAR62 = "0123456789ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz";
	final static char[] digits = CHAR62.toCharArray();

	final static Map<Character, Integer> digitMap = new HashMap<Character, Integer>();

	static {
		for (int i = 0; i < digits.length; i++) {
			digitMap.put(digits[i], (int) i);
		}
	}

	private NumberUtil() {
	}

	/**
	 * 支持的最大进制数
	 */
	public static final int MAX_RADIX = digits.length;

	/**
	 * 支持的最小进制数
	 */
	public static final int MIN_RADIX = 2;

	/**
	 * 将长整型数值转换为62进制数
	 * @param num
	 * @return
	 */
	public static String toString(long num) {
		return toString(num, MAX_RADIX);
	}

	/**
	 * 将长整型数值转换为指定的进制数（最大支持62进制，字母数字已经用尽）
	 * @param num
	 * @param radix
	 * @return
	 */
	public static String toString(long num, int radix) {
		if (radix < MIN_RADIX || radix > MAX_RADIX)
			radix = 10;
		if (radix == 10)
			return Long.toString(num);

		final int size = 65;
		int charPos = 64;

		char[] buf = new char[size];
		boolean negative = (num < 0);

		if (!negative) {
			num = -num;
		}

		while (num <= -radix) {
			buf[charPos--] = digits[(int) (-(num % radix))];
			num = num / radix;
		}
		buf[charPos] = digits[(int) (-num)];

		if (negative) {
			buf[--charPos] = '-';
		}

		return new String(buf, charPos, (size - charPos));
	}

	/**
	 * 将62进制字符串转换为长整型数字
	 * @param str 数字字符串
	 * @param radix 进制数
	 * @return
	 */
	public static long toNumber(String str) {
		return toNumber(str, MAX_RADIX);
	}

	/**
	 * 将字符串转换为长整型数字
	 * @param str 数字字符串
	 * @param radix 进制数
	 * @return
	 */
	public static long toNumber(String str, int radix) {
		if (str == null) {
			throw new NumberFormatException("null");
		}

		if (radix < MIN_RADIX) {
			throw new NumberFormatException("radix " + radix + " less than " + MIN_RADIX);
		}
		if (radix > MAX_RADIX) {
			throw new NumberFormatException("radix " + radix + " greater than " + MAX_RADIX);
		}

		long result = 0;
		boolean negative = false;
		int i = 0, len = str.length();
		long limit = -Long.MAX_VALUE;
		long multmin;
		Integer digit;

		if (len > 0) {
			char firstChar = str.charAt(0);
			if (firstChar < '0') {
				if (firstChar == '-') {
					negative = true;
					limit = Long.MIN_VALUE;
				} else if (firstChar != '+')
					throw forInputString(str);

				if (len == 1) {
					throw forInputString(str);
				}
				i++;
			}
			multmin = limit / radix;
			while (i < len) {
				digit = digitMap.get(str.charAt(i++));
				if (digit == null) {
					throw forInputString(str);
				}
				if (digit < 0) {
					throw forInputString(str);
				}
				if (result < multmin) {
					throw forInputString(str);
				}
				result *= radix;
				if (result < limit + digit) {
					throw forInputString(str);
				}
				result -= digit;
			}
		} else {
			throw forInputString(str);
		}
		return negative ? result : -result;
	}

	// 默认除法运算精度
	private static final int DEF_DIV_SCALE = 10;

	/**
	 * 提供精确的加法运算。
	 * @param v1 被加数
	 * @param v2 加数
	 * @return 两个参数的和
	 */

	public static double add(double v1, double v2) {
		BigDecimal b1 = new BigDecimal(Double.toString(v1));
		BigDecimal b2 = new BigDecimal(Double.toString(v2));
		return b1.add(b2).doubleValue();
	}

	/**
	 * 提供精确的减法运算。
	 * @param v1 被减数
	 * @param v2 减数
	 * @return 两个参数的差
	 */

	public static double sub(double v1, double v2) {
		BigDecimal b1 = new BigDecimal(Double.toString(v1));
		BigDecimal b2 = new BigDecimal(Double.toString(v2));
		return b1.subtract(b2).doubleValue();
	}

	/**
	 * 提供精确的乘法运算。
	 * @param v1 被乘数
	 * @param v2 乘数
	 * @return 两个参数的积
	 */
	public static double mul(double v1, double v2) {
		BigDecimal b1 = new BigDecimal(Double.toString(v1));
		BigDecimal b2 = new BigDecimal(Double.toString(v2));
		return b1.multiply(b2).doubleValue();
	}

	/**
	 * 提供（相对）精确的除法运算，当发生除不尽的情况时，精确到 小数点以后10位，以后的数字四舍五入。
	 * @param v1 被除数
	 * @param v2 除数
	 * @return 两个参数的商
	 */

	public static double div(double v1, double v2) {
		return div(v1, v2, DEF_DIV_SCALE);
	}

	/**
	 * 提供（相对）精确的除法运算。当发生除不尽的情况时，由scale参数指 定精度，以后的数字四舍五入。
	 * @param v1 被除数
	 * @param v2 除数
	 * @param scale 表示表示需要精确到小数点以后几位。
	 * @return 两个参数的商
	 */
	public static double div(double v1, double v2, int scale) {
		if (scale < 0) {
			throw new IllegalArgumentException("The scale must be a positive integer or zero");
		}
		BigDecimal b1 = new BigDecimal(Double.toString(v1));
		BigDecimal b2 = new BigDecimal(Double.toString(v2));
		return b1.divide(b2, scale, BigDecimal.ROUND_HALF_UP).doubleValue();
	}

	/**
	 * 提供精确的小数位四舍五入处理。
	 * @param v 需要四舍五入的数字
	 * @param scale 小数点后保留几位
	 * @return 四舍五入后的结果
	 */
	public static double round(double v, int scale) {
		if (scale < 0) {
			throw new IllegalArgumentException("The scale must be a positive integer or zero");
		}
		BigDecimal b = new BigDecimal(Double.toString(v));
		BigDecimal one = new BigDecimal("1");
		return b.divide(one, scale, BigDecimal.ROUND_HALF_UP).doubleValue();
	}

	/**
	 * 随机生成指定长度的字符串，由数字、小写字母和大写字母组成
	 * @param length 字符串长度
	 * @return
	 */
	public static String random(int length) {
		return random(length, MAX_RADIX);
	}

	/**
	 * 随机生成指定长度的字符串，由数字、小写字母和大写字母组成
	 * @param length 字符串长度
	 * @param radix 进制数
	 * @return
	 */
	public static String random(int length, int radix) {
		Random random = new Random();
		char[] cs = new char[length];
		for (int i = 0; i < length; i++) {
			cs[i] = digits[random.nextInt(radix)];
		}
		return new String(cs);
	}

	/**
	 * 以62进制（字母加数字）生成19位UUID，最短的UUID
	 * @return
	 */
	public static String uuid() {
		UUID uuid = UUID.randomUUID();
		StringBuilder sb = new StringBuilder();
		sb.append(digits(uuid.getMostSignificantBits() >> 32, 8));
		sb.append(digits(uuid.getMostSignificantBits() >> 16, 4));
		sb.append(digits(uuid.getMostSignificantBits(), 4));
		sb.append(digits(uuid.getLeastSignificantBits() >> 48, 4));
		sb.append(digits(uuid.getLeastSignificantBits(), 12));
		return sb.toString();
	}

	private static String digits(long val, int digits) {
		long hi = 1L << (digits * 4);
		return NumberUtil.toString(hi | (val & (hi - 1)), NumberUtil.MAX_RADIX).substring(1);
	}

	static NumberFormatException forInputString(String s) {
		return new NumberFormatException("For input string: \"" + s + "\"");
	}

}