package cn.elwy.common.util;

import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.Set;
import java.util.TimeZone;
import java.util.TreeSet;

import com.fasterxml.jackson.annotation.JsonInclude.Include;
import com.fasterxml.jackson.core.JsonParser;
import com.fasterxml.jackson.databind.DeserializationFeature;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.SerializationFeature;
import com.fasterxml.jackson.databind.ser.impl.SimpleBeanPropertyFilter;
import com.fasterxml.jackson.databind.ser.impl.SimpleFilterProvider;
import com.fasterxml.jackson.databind.type.TypeFactory;

import cn.elwy.common.exception.RunException;

/**
 * JSON 工具类(jackson2.x)
 * @author huangsq
 * @version 1.0, 2018-02-19
 */
public final class JsonUtil {

	private static final ObjectMapper objectMapper = new ObjectMapper();
	private static final ObjectMapper objectMapperFileter = new ObjectMapper();

	static {
		initJsonConfig(objectMapper);
		initJsonConfig(objectMapperFileter);
	}

	private static void initJsonConfig(ObjectMapper objectMapper) {
		// 去掉默认的时间戳格式
		objectMapper.configure(SerializationFeature.WRITE_DATES_AS_TIMESTAMPS, false);
		// 设置时区
		objectMapper.setTimeZone(TimeZone.getTimeZone("GMT+08:00"));
		objectMapper.configure(SerializationFeature.WRITE_NULL_MAP_VALUES, false);
		// 空值不序列化
		objectMapper.setSerializationInclusion(Include.NON_NULL);
		// 反序列化时，属性不存在的兼容处理
		objectMapper.getDeserializationConfig().withoutFeatures(DeserializationFeature.FAIL_ON_UNKNOWN_PROPERTIES);
		// 序列化时，日期的统一格式
		objectMapper.setDateFormat(new SimpleDateFormat("yyyy-MM-dd HH:mm:ss"));

		objectMapper.configure(SerializationFeature.FAIL_ON_EMPTY_BEANS, false);
		objectMapper.configure(DeserializationFeature.FAIL_ON_UNKNOWN_PROPERTIES, false);
		// 单引号处理
		objectMapper.configure(JsonParser.Feature.ALLOW_SINGLE_QUOTES, true);
	}

	/**
	 * 将JAVA对象包装成JSON格式.
	 * @param object 要转换为json格式的对象
	 * @return 转换后的JSON字符串
	 * @throws RunException 异常
	 */
	public static String toJson(Object object) {
		try {
			return objectMapper.writeValueAsString(object);
		} catch (Exception e) {
			throw new RunException(e.getMessage(), e);
		}
	}

	/**
	 * 将JAVA对象包装成JSON格式.
	 * @param object 要转换为json格式的对象
	 * @param isExecute 是否排除字段，false表示包含字段
	 * @param object 要转换为json格式的对象
	 * @return propertyNames 过滤的属性
	 * @throws RunException 异常
	 */
	public static String toJson(Object object, boolean isExecute, String... propertyNames) {
		SimpleFilterProvider fileter = new SimpleFilterProvider();
		if (isExecute) {
			fileter.addFilter("executeFilter", SimpleBeanPropertyFilter.serializeAllExcept(propertyNames));
		} else {
			fileter.addFilter("includeFilter", SimpleBeanPropertyFilter.serializeAllExcept(propertyNames));
		}
		objectMapperFileter.setFilterProvider(fileter);
		try {
			return objectMapperFileter.writeValueAsString(object);
		} catch (Exception e) {
			throw new RunException(e.getMessage(), e);
		}
	}

	/**
	 * 将JAVA对象包装成JSON格式.
	 * @param json Json格式字符串
	 * @param clazz 要转换的Java对象类型
	 * @return 转换后的Java对象
	 * @throws RunException 异常
	 */
	public static <T> T toObject(String json, Class<T> clazz) {
		try {
			return objectMapper.readValue(json, clazz);
		} catch (Exception e) {
			throw new RunException(e.getMessage(), e);
		}
	}

	/**
	 * 将JAVA对象包装成JSON格式.
	 * @param json Json格式字符串
	 * @param clazz 要转换的Java对象类型
	 * @return 转换后的Java对象
	 * @throws RunException 异常
	 */
	public static <T> T toObject(String json, Class<T> clazz, boolean isExecute, String... propertyNames) {
		SimpleFilterProvider fileter = new SimpleFilterProvider();
		if (isExecute) {
			fileter.addFilter("executeFilter", SimpleBeanPropertyFilter.serializeAllExcept(propertyNames));
		} else {
			fileter.addFilter("includeFilter", SimpleBeanPropertyFilter.serializeAllExcept(propertyNames));
		}
		objectMapperFileter.setFilterProvider(fileter);
		try {
			return objectMapperFileter.readValue(json, clazz);
		} catch (Exception e) {
			throw new RunException(e.getMessage(), e);
		}
	}

	/**
	 * 将 JSON 数组转换为List
	 * @param <T>
	 * @param json JSON字符串
	 * @param clazz 实体类型
	 * @return 储存类型为 clazz的对象List
	 * @throws RunException 异常
	 */
	public static <T> List<T> toList(String json, Class<T> clazz) throws RunException {
		try {
			TypeFactory factory = TypeFactory.defaultInstance();
			return objectMapper.readValue(json, factory.constructCollectionType(ArrayList.class, clazz));
		} catch (Exception e) {
			throw new RunException(e.getMessage(), e);
		}
	}

	/**
	 * 将 JSON 数组转换为List
	 * @param <T>
	 * @param json JSON字符串
	 * @param clazz 实体类型
	 * @return 储存类型为 clazz的对象List
	 * @throws RunException 异常
	 */
	public static <T> Set<T> toSet(String json, Class<T> clazz) throws RunException {
		try {
			TypeFactory factory = TypeFactory.defaultInstance();
			return objectMapper.readValue(json, factory.constructCollectionType(TreeSet.class, clazz));
		} catch (Exception e) {
			throw new RunException(e.getMessage(), e);
		}
	}

	/**
	 * 将 JSON 数组转换为Collection
	 * @param <T>
	 * @param <C>
	 * @param json JSON字符串
	 * @param clazz 实体类型
	 * @param collection Collection类型
	 * @return 储存类型为 clazz的对象List
	 * @throws RunException 异常
	 */
	@SuppressWarnings("rawtypes")
	public static <T, C> Set<T> toCollection(String json, Class<T> clazz, Class<Collection> collection)
			throws RunException {
		try {
			TypeFactory factory = TypeFactory.defaultInstance();
			return objectMapper.readValue(json, factory.constructCollectionType(collection, clazz));
		} catch (Exception e) {
			throw new RunException(e.getMessage(), e);
		}
	}

}