package cn.elwy.common.util;

import java.io.File;
import java.io.StringReader;
import java.io.StringWriter;
import java.util.Collection;

import javax.xml.bind.JAXBContext;
import javax.xml.bind.JAXBElement;
import javax.xml.bind.Marshaller;
import javax.xml.bind.Unmarshaller;
import javax.xml.bind.annotation.XmlAnyElement;
import javax.xml.namespace.QName;

/**
 * 使用JAXB进行对象和XML互相转换工具类
 * @author huangsq
 * @version 1.0, 2018-02-19
 */
public class JaxbUtil {

	/**
	 * 将JavaBean转换成XML字符串
	 * @param object 需要转换成xml的对象
	 * @return
	 * @throws Exception
	 */
	public static String toXml(Object object) {
		Marshaller marshaller = getMarshaller(object);
		StringWriter writer = new StringWriter();
		try {
			marshaller.marshal(object, writer);
		} catch (Exception e) {
			throw new RuntimeException(e);
		}
		return writer.toString().replace(" standalone=\"yes\"", "");
	}

	/**
	 * 将JavaBean转换成XML字符串
	 * @param root 需要转换成xml的对象
	 * @param rootName 根元素名称
	 * @return
	 * @throws Exception
	 */
	public static String toXml(Collection<?> root, String rootName) {
		Marshaller marshaller = getMarshaller(root);
		StringWriter writer = new StringWriter();

		CollectionWrapper wrapper = new CollectionWrapper();
		wrapper.collection = root;
		JAXBElement<CollectionWrapper> wrapperElement = new JAXBElement<CollectionWrapper>(new QName(rootName),
				CollectionWrapper.class, wrapper);
		try {
			marshaller.marshal(wrapperElement, writer);
		} catch (Exception e) {
			throw new RuntimeException(e);
		}
		return writer.toString().replace(" standalone=\"yes\"", "");
	}

	/**
	 * 将JavaBean转换成XML字符串
	 * @param object 需要转换成xml的对象
	 * @param file 文件路径
	 * @return
	 * @throws Exception
	 */
	public static void toXml(Object object, File file) {
		Marshaller marshaller = getMarshaller(object);
		try {
			marshaller.marshal(object, file);
		} catch (Exception e) {
			throw new RuntimeException(e);
		}
	}

	/**
	 * 将JavaBean转换成XML字符串
	 * @param root 需要转换成xml的对象
	 * @param rootName 根元素名称
	 * @param file 文件路径
	 * @return
	 * @throws Exception
	 */
	public static void toXml(Collection<?> root, String rootName, File file) {
		Marshaller marshaller = getMarshaller(root);

		CollectionWrapper wrapper = new CollectionWrapper();
		wrapper.collection = root;
		JAXBElement<CollectionWrapper> wrapperElement = new JAXBElement<CollectionWrapper>(new QName(rootName),
				CollectionWrapper.class, wrapper);
		try {
			marshaller.marshal(wrapperElement, file);
		} catch (Exception e) {
			throw new RuntimeException(e);
		}
	}

	@SuppressWarnings("unchecked")
	public static <T> T toBean(String xml, Class<T> beanClass) {
		Unmarshaller unmarshaller = getUnmarshaller(beanClass);
		try {
			StringReader reader = new StringReader(xml);
			return (T) unmarshaller.unmarshal(reader);
		} catch (Exception e) {
			throw new RuntimeException(e);
		}
	}

	@SuppressWarnings("unchecked")
	public static <T> T toBean(File file, Class<T> beanClass) {
		Unmarshaller unmarshaller = getUnmarshaller(beanClass);
		try {
			return (T) unmarshaller.unmarshal(file);
		} catch (Exception e) {
			throw new RuntimeException(e);
		}
	}

	private static <T> Unmarshaller getUnmarshaller(Class<T> beanClass) {
		try {
			JAXBContext context = JAXBContext.newInstance(beanClass);
			return context.createUnmarshaller();
		} catch (Exception e) {
			throw new RuntimeException(e);
		}
	}

	private static Marshaller getMarshaller(Object object) {
		return getMarshaller(object, "UTF-8");
	}

	private static Marshaller getMarshaller(Object object, String encoding) {
		try {
			JAXBContext context = JAXBContext.newInstance(object.getClass());
			// 根据上下文获取marshaller对象
			Marshaller marshaller = context.createMarshaller();
			// 设置编码字符集
			marshaller.setProperty(Marshaller.JAXB_ENCODING, "UTF-8");
			// 格式化XML输出，有分行和缩进
			marshaller.setProperty(Marshaller.JAXB_FORMATTED_OUTPUT, true);
			// 是否省略XML头声明信息
			marshaller.setProperty(Marshaller.JAXB_FRAGMENT, false);
			return marshaller;
		} catch (Exception e) {
			throw new RuntimeException(e);
		}
	}

	/**
	 * 封装Root Element 是 Collection的情况.
	 */
	public static class CollectionWrapper {
		@XmlAnyElement
		protected Collection<?> collection;
	}

}