package cn.elwy.common.util;

import java.io.BufferedReader;
import java.io.Closeable;
import java.io.DataInputStream;
import java.io.DataOutputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.OutputStream;
import java.io.Serializable;
import java.io.StringWriter;
import java.io.UnsupportedEncodingException;
import java.net.HttpURLConnection;
import java.net.MalformedURLException;
import java.net.URL;
import java.net.URLEncoder;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Set;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import javax.activation.MimetypesFileTypeMap;

import cn.elwy.common.log.Logger;
import cn.elwy.common.log.LoggerFactory;

/**
 * HTTP请求工具类
 * @author huangsq
 * @version 1.0, 2018-02-19
 */
public final class HttpUtil {

	private static Logger logger = LoggerFactory.getLogger(HttpUtil.class);

	public static final String DEFAULT_CHARSET = "UTF-8";
	public static final String METHOD_POST = "POST";
	public static final String METHOD_GET = "GET";
	public static final int BUFFER_SIZE = 8192; // 8K
	public static final int CONNECT_TIMEOUT = 3000; // 3s
	public static final int READ_TIMEOUT = 60000; // 6s
	public static final int READ_FILE_TIMEOUT = 3600000; // 3600s

	public static final String KEY_X_REQUESTED_WITH = "X-Requested-With";
	public static final String KEY_USER_AGENT = "User-Agent";
	public static final String KEY_ACCEPT_ENCODING = "Accept-Encoding";
	public static final String KEY_CONTENT_TYPE = "Content-Type";
	public static final String KEY_ACCEPT_LANGUAGE = "Accept-Language";
	public static final String KEY_CONNECTION = "Connection";
	public static final String KEY_CACHE_CONTROL = "Cache-Control";
	public static final String KEY_ACCEPT = "Accept";
	public static final String KEY_CHARSERT = "Charsert";
	public static final String KEY_COOKIE = "Cookie";

	// 定义数据分隔线
	private static String BOUNDARY = "---------------------------71a816d53b6o219";

	private HttpUtil() {
	}

	/**
	 * 测试url是否可连接，默认3秒超时
	 * @param url 请求地址
	 * @return
	 */
	public static boolean isConnected(String url) {
		try {
			return isConnected(new URL(url), CONNECT_TIMEOUT);
		} catch (MalformedURLException e) {
			logger.debug(e.getMessage(), e);
			return false;
		}
	}

	/**
	 * 测试url是否可连接
	 * @param url 请求地址
	 * @param timeout 超时时间
	 * @return
	 */
	public static boolean isConnected(String url, int timeout) {
		try {
			return isConnected(new URL(url), timeout);
		} catch (MalformedURLException e) {
			logger.debug(e.getMessage(), e);
			return false;
		}
	}

	/**
	 * 测试url是否可连接
	 * @param url 请求地址
	 * @return
	 */
	public static boolean isConnected(URL url) {
		return isConnected(url, CONNECT_TIMEOUT);
	}

	/**
	 * 测试url是否可连接
	 * @param url 请求地址
	 * @param timeout 超时时间
	 * @return
	 */
	public static boolean isConnected(URL url, int timeout) {
		HttpURLConnection httpConn = null;
		try {
			httpConn = (HttpURLConnection) url.openConnection();
			httpConn.setConnectTimeout(timeout);
			httpConn.connect();
			int responseCode = httpConn.getResponseCode();
			return isSuccess(responseCode);
		} catch (Exception e) {
			logger.debug(e.getMessage(), e);
			return false;
		} finally {
			close(httpConn);
		}
	}

	/**
	 * 发送异步请求，返回响应的结果
	 * @param httpParameter 请求参数
	 * @return
	 * @throws IOException
	 */
	public static HttpResult ajaxRequest(HttpParameter httpParameter) throws IOException {
		// Ajax 异步请求方式
		httpParameter.putDefaultProperty(KEY_X_REQUESTED_WITH, "XMLHttpRequest");
		return httpRequest(httpParameter);
	}

	/**
	 * 发送http请求，返回响应的结果
	 * @param httpParameter 请求参数
	 * @return
	 * @throws IOException
	 */
	public static HttpResult httpRequest(HttpParameter httpParameter) throws IOException {
		HttpURLConnection httpConn = null;
		try {
			httpConn = getHttpConn(httpParameter);
			return getResponseAsString(httpConn);
		} finally {
			close(httpConn);
		}
	}

	/**
	 * 发送POST请求，下载远程文件
	 * @param url 请求地址
	 * @param connectTimeout 连接超时时间
	 * @param readTimeout 读取超时时间
	 * @param file 文件保存路径
	 * @param params 请求时传递的参数
	 * @param charset 字符集
	 * @param property HttpURLConnection属性配置,对象不允许为空
	 * @return File 文件保存路径
	 * @throws IOException
	 */
	public static File ajaxDownload(HttpParameter httpParameter, File file) throws IOException {
		// Ajax 异步请求方式
		httpParameter.putDefaultProperty(KEY_X_REQUESTED_WITH, "XMLHttpRequest");
		return httpDownload(httpParameter, file);
	}

	/**
	 * 下载远程文件
	 * @param url 请求地址
	 * @param connectTimeout 连接超时时间
	 * @param readTimeout 读取超时时间
	 * @param file 文件保存路径
	 * @param params 请求时传递的参数
	 * @param charset 字符集
	 * @param property HttpURLConnection属性配置,对象不允许为空
	 * @return File 文件保存路径
	 * @throws IOException
	 */
	public static File httpDownload(HttpParameter httpParameter, File file) throws IOException {
		HttpURLConnection httpConn = null;
		try {
			httpConn = getHttpConn(httpParameter);
			return getResponseAsFile(httpConn, file);
		} finally {
			close(httpConn);
		}
	}

	/**
	 * 发送POST异步请求，上传文件
	 * @param url 请求地址
	 * @param connectTimeout 连接超时时间
	 * @param readTimeout 读取超时时间
	 * @param fileMap 上传的文件和对应的字段信息
	 * @param params 请求时传递的参数
	 * @return
	 * @throws IOException
	 */
	public static HttpResult ajaxUpload(HttpParameter httpParameter, List<File> fileList) throws IOException {
		// Ajax 异步请求方式
		httpParameter.putDefaultProperty(KEY_X_REQUESTED_WITH, "XMLHttpRequest");
		return httpUpload(httpParameter, fileList);
	}

	/**
	 * 发送POST异步请求，上传文件
	 * @param url 请求地址
	 * @param connectTimeout 连接超时时间
	 * @param readTimeout 读取超时时间
	 * @param fileMap 上传的文件和对应的字段信息
	 * @param params 请求时传递的参数
	 * @return
	 * @throws IOException
	 */
	public static HttpResult ajaxUpload(HttpParameter httpParameter, Map<String, File> fileMap) throws IOException {
		// Ajax 异步请求方式
		httpParameter.putDefaultProperty(KEY_X_REQUESTED_WITH, "XMLHttpRequest");
		return httpUpload(httpParameter, fileMap);
	}

	/**
	 * 发送POST请求，返回响应的结果
	 * @param httpParameter 请求地址
	 * @param fileList 上传的文件列表
	 * @return
	 * @throws IOException
	 */
	public static HttpResult httpUpload(HttpParameter httpParameter, List<File> fileList) throws IOException {
		Map<String, File> fileMap = new HashMap<String, File>();
		for (File file : fileList) {
			fileMap.put(file.getName(), file);
		}
		return httpUpload(httpParameter, fileMap);
	}

	/**
	 * 发送http请求，返回响应的结果
	 * @param url 请求地址
	 * @param connectTimeout 连接超时时间
	 * @param readTimeout 读取超时时间
	 * @param fileMap 上传的文件和对应的字段信息
	 * @param params 请求时传递的参数
	 * @param charset 请求的编码
	 * @param property HttpURLConnection属性配置,对象不允许为空
	 * @return
	 * @throws IOException
	 */
	public static HttpResult httpUpload(HttpParameter httpParameter, Map<String, File> fileMap) throws IOException {
		String url = httpParameter.getUrl();
		Map<String, String> params = httpParameter.getParams();
		Map<String, String> property = httpParameter.getProperty();
		int connectTimeout = httpParameter.getConnectTimeout();
		int readTimeout = httpParameter.getReadTimeout();

		HttpURLConnection httpConn = null;
		OutputStream out = null;
		try {
			String contentType = "multipart/form-data; boundary=" + BOUNDARY;
			property.put(KEY_CONTENT_TYPE, contentType);
			httpConn = openHttpConn(new URL(url), connectTimeout, readTimeout, METHOD_POST, property);

			out = new DataOutputStream(httpConn.getOutputStream());
			appendContent(params, out);
			appendFile(fileMap, out);
			byte[] endData = ("\r\n--" + BOUNDARY + "--\r\n").getBytes();
			out.write(endData);
			out.flush();
			return getResponseAsString(httpConn);
		} finally {
			close(out);
			close(httpConn);
		}
	}

	/**
	 * 发送http请求，返回响应的结果
	 * @param httpParameter 请求地址
	 * @return
	 * @throws IOException
	 */
	public static HttpURLConnection getHttpConn(HttpParameter httpParameter) throws IOException {
		String charset = httpParameter.getCharset();
		if (isEmpty(charset)) {
			charset = DEFAULT_CHARSET;
		}
		String url = httpParameter.getUrl();
		Map<String, String> params = httpParameter.getParams();
		params = buildParams(url, params);
		byte[] data = httpParameter.getData();
		if (data == null) {
			String content = getParamsContext(params, charset);
			if (!isEmpty(content)) {
				data = content.getBytes(charset);
			}
		}
		String method = httpParameter.getMethod();
		if (METHOD_POST.equalsIgnoreCase(method)) {
			url = tranferUrl(url);
		}
		Map<String, String> property = httpParameter.getProperty();
		// 设置请求的 HTTP内容类型
		String contentType = "application/x-www-form-urlencoded;charset=" + charset;
		httpParameter.putDefaultProperty(KEY_CONTENT_TYPE, contentType);
		httpParameter.putDefaultProperty(KEY_ACCEPT, "*/*");
		httpParameter.putDefaultProperty(KEY_CHARSERT, charset);
		httpParameter.putDefaultProperty(KEY_USER_AGENT, "Mozilla/5.0 (Windows; U; Windows NT 6.1; zh-CN; rv:1.9.2.6)");
		httpParameter.putDefaultProperty(KEY_ACCEPT_ENCODING, "gzip, deflate");
		httpParameter.putDefaultProperty(KEY_ACCEPT_LANGUAGE, "zh-cn");
		httpParameter.putDefaultProperty(KEY_CONNECTION, "Keep-Alive");
		httpParameter.putDefaultProperty(KEY_CACHE_CONTROL, "no-cache");

		int connectTimeout = httpParameter.getConnectTimeout();
		int readTimeout = httpParameter.getReadTimeout();

		HttpURLConnection httpConn = null;
		OutputStream out = null;
		try {
			httpConn = openHttpConn(new URL(url), connectTimeout, readTimeout, method, property);
			if (data != null) {
				out = httpConn.getOutputStream();
				out.write(data);
			}
			return httpConn;
		} finally {
			close(out);
		}
	}

	/**
	 * 获取一个HttpURLConnection连接，并设置相应的参数
	 * @param url 请求地址 url地址
	 * @param connectTimeout 连接超时时间
	 * @param readTimeout 读取超时时间
	 * @param method 请求的方法，可以是POST,GET
	 * @param property HttpURLConnection属性配置,对象不允许为空
	 * @return
	 * @throws IOException
	 */
	private static HttpURLConnection openHttpConn(URL url, int connectTimeout, int readTimeout, String method,
			Map<String, String> property) throws IOException {

		HttpURLConnection httpConn = (HttpURLConnection) url.openConnection();
		if (isEmpty(method)) {
			method = METHOD_POST;
		}
		// 设定请求的方法，默认是GET
		httpConn.setRequestMethod(method);
		if ("POST".equalsIgnoreCase(method)) {
			// 设置是否向httpUrlConnection输出，因为这个是post请求，参数要放在
			// http正文内，因此需要设为true, 默认情况下是false;
			httpConn.setDoOutput(true);
			// 设置是否从httpUrlConnection读入，默认情况下是true;
			httpConn.setDoInput(true);
			// Post 请求不能使用缓存
			httpConn.setUseCaches(false);
		}

		if (connectTimeout != -1) {
			// 设置连接超时时间
			httpConn.setConnectTimeout(connectTimeout > 0 ? connectTimeout : CONNECT_TIMEOUT);
		}
		if (readTimeout != -1) {
			// 设置读取超时时间
			httpConn.setReadTimeout(readTimeout > 0 ? readTimeout : READ_TIMEOUT);
		}

		for (Entry<String, String> entry : property.entrySet()) {
			httpConn.setRequestProperty(entry.getKey(), entry.getValue());
		}
		return httpConn;
	}

	/**
	 * 添加表单内容
	 * @param textMap
	 * @param out
	 * @throws IOException
	 */
	private static void appendContent(Map<String, String> textMap, OutputStream out) throws IOException {
		if (textMap == null) {
			return;
		}
		StringBuffer contentBuffer = new StringBuffer();
		for (Entry<String, String> entry : textMap.entrySet()) {
			String inputName = entry.getKey();
			String inputValue = entry.getValue();
			if (inputValue == null) {
				continue;
			}
			contentBuffer.append("\r\n").append("--").append(BOUNDARY).append("\r\n");
			contentBuffer.append("Content-Disposition: form-data; name=\"" + inputName + "\"\r\n\r\n");
			contentBuffer.append(inputValue);
		}
		out.write(contentBuffer.toString().getBytes());
	}

	/**
	 * 添加表单文件
	 * @param fileMap
	 * @param out
	 * @throws IOException
	 */
	private static void appendFile(Map<String, File> fileMap, OutputStream out) throws IOException {
		if (fileMap == null) {
			return;
		}
		for (Entry<String, File> entry : fileMap.entrySet()) {
			String inputName = entry.getKey();
			File file = entry.getValue();
			if (file == null) {
				continue;
			}
			upFile(inputName, file, out);
		}
	}

	private static void upFile(String inputName, File file, OutputStream out) throws IOException {
		if (file.isDirectory()) {
			File[] listFiles = file.listFiles();
			if (listFiles != null) {
				for (int i = 0; i < listFiles.length; i++) {
					upFile(inputName + "_" + i, listFiles[i], out);
				}
			}
		} else {
			String filename = file.getName();
			String contentType = new MimetypesFileTypeMap().getContentType(file);

			if (contentType == null || contentType.equals("")) {
				contentType = "application/octet-stream";
			}

			StringBuffer strBuf = new StringBuffer();
			strBuf.append("\r\n").append("--").append(BOUNDARY).append("\r\n");
			strBuf.append("Content-Disposition: form-data;name=\"" + inputName + "\"; filename=\"" + filename + "\"\r\n");
			strBuf.append("Content-Type:" + contentType + "\r\n\r\n");

			out.write(strBuf.toString().getBytes());

			int length = 0;
			byte[] buffer = new byte[1024];
			DataInputStream in = null;
			try {
				in = new DataInputStream(new FileInputStream(file));
				while ((length = in.read(buffer)) != -1) {
					out.write(buffer, 0, length);
				}
			} finally {
				close(in);
			}
		}
	}

	/**
	 * 判断http请求返回状态是否正确
	 * @param responseCode
	 * @return
	 */
	private static boolean isSuccess(int responseCode) {
		return (responseCode >= 200 && responseCode < 300);// || responseCode == 404;
	}

	/**
	 * 过滤掉Post请求url中的参数
	 * @param url 请求地址
	 * @return
	 * @throws MalformedURLException
	 * @throws UnsupportedEncodingException
	 */
	private static String tranferUrl(String url) throws MalformedURLException, UnsupportedEncodingException {
		if (url != null) {
			URL _url = new URL(url);
			String path = _url.getPath();
			String DOT_RE = "/\\./";
			String DOUBLE_DOT_RE = "/[^/]+/\\.\\./";

			path = path.replaceAll(DOT_RE, "/");
			Pattern p = Pattern.compile(DOUBLE_DOT_RE);
			Matcher m = p.matcher(path);
			while (m.find()) {
				String a = path.substring(m.start(), m.end());
				path = path.replaceFirst(a, "/");
				m = p.matcher(path);
			}
			path = path.replaceAll("/\\.\\./", "/");
			url = _url.getProtocol() + "://" + _url.getHost() + (_url.getPort() > 0 ? ":" + _url.getPort() : "") + path;
		}
		// 加入对url中包括非URI格式的处理
		if (url != null && url.indexOf("&") > -1) {
			int idx = -1;
			String prefiex = null;
			StringBuilder sb = new StringBuilder();
			while ((idx = url.indexOf("&")) > -1) {
				if (prefiex == null) {
					prefiex = url.substring(0, idx);
				}
				url = url.substring(idx + 1, url.length());
				int split = url.indexOf("=");
				String key = url.substring(0, split);
				int last = url.indexOf("&");
				String value;
				if (last > -1) {
					value = url.substring(split + 1, last);
				} else {
					value = url.substring(split + 1, url.length());
				}
				value = URLEncoder.encode(value, "UTF-8");
				sb.append("&" + key + "=" + value);
			}
			url = prefiex + sb.toString();
		}
		int mark = url.indexOf("?");
		// 已经包含参数
		if (mark > -1) {
			url = url.substring(0, mark);
		}
		return url;
	}

	private static Map<String, String> buildParams(String url, Map<String, String> params) {
		int mark = url.indexOf("?");
		// 已经包含参数
		if (mark > -1) {
			String _params = url.substring(mark + 1, url.length());
			url = url.substring(0, mark);
			String[] paramArray = _params.split("&");
			if (params == null) {
				params = new HashMap<String, String>();
			}
			for (int i = 0; i < paramArray.length; i++) {
				String string = paramArray[i];
				int index = string.indexOf("=");
				if (index > 0) {
					String key = string.substring(0, index);
					String value = string.substring(index + 1, string.length());
					params.put(key, value);
				}
			}
		}
		return params;
	}

	private static String getParamsContext(Map<String, String> params, String charset)
			throws UnsupportedEncodingException {
		if ((params == null) || (params.isEmpty())) {
			return null;
		}

		StringBuilder query = new StringBuilder();
		Set<Entry<String, String>> entries = params.entrySet();
		boolean hasParam = false;
		for (Map.Entry<String, String> entry : entries) {
			String name = entry.getKey();
			String value = entry.getValue();
			if (hasParam) {
				query.append("&");
			} else {
				hasParam = true;
			}

			if (!isEmpty(name) && !isEmpty(value)) {
				query.append(name).append("=").append(URLEncoder.encode(value, charset));
			} else if (!isEmpty(name)) {
				query.append(name).append("=");
			}
		}
		return query.toString();
	}

	/**
	 * 获取Http请求返回的内容
	 * @param httpConn
	 * @return
	 * @throws IOException
	 */
	private static HttpResult getResponseAsString(HttpURLConnection httpConn) throws IOException {
		String charset = getResponseCharset(httpConn.getContentType());
		InputStream errorStream = httpConn.getErrorStream();
		HttpResult httpResult = new HttpResult();
		httpResult.setResponseCode(httpConn.getResponseCode());
		if (errorStream == null) {
			httpResult.setSuccess(true);
			httpResult.setMessage(getStreamAsString(httpConn.getInputStream(), charset));
		} else {
			httpResult.setSuccess(false);
			httpResult.setMessage(getStreamAsString(errorStream, charset));
		}
		return httpResult;
	}

	/**
	 * 获取Http请求返回的内容
	 * @param httpConn
	 * @return
	 * @throws IOException
	 */
	private static File getResponseAsFile(HttpURLConnection httpConn, File file) throws IOException {
		String charset = getResponseCharset(httpConn.getContentType());
		InputStream errorStream = httpConn.getErrorStream();
		if (errorStream != null) {
			String streamAsString = getStreamAsString(errorStream, charset);
			throw new RuntimeException(streamAsString);
		}
		writeFileByInputStream(file, httpConn.getInputStream());
		return file;
	}

	/**
	 * 保存数据流到指定的文件
	 * @param file 输出文件
	 * @param inputStream 输入流
	 * @throws IOException
	 */
	private static void writeFileByInputStream(File file, InputStream inputStream) throws IOException {
		OutputStream out = null;
		if (!file.getParentFile().exists()) { // 如果目录不存在则创建目录
			file.getParentFile().mkdirs();
		}
		try {
			out = new FileOutputStream(file);
			byte[] buffer = new byte[BUFFER_SIZE];
			int read;
			while ((read = inputStream.read(buffer)) != -1) {
				out.write(buffer, 0, read);
				out.flush();
			}
		} finally {
			close(inputStream);
			close(out);
		}
	}

	/**
	 * 读取输入流的内容，转换成字符串返回
	 * @param stream
	 * @param charset
	 * @return
	 * @throws IOException
	 */
	private static String getStreamAsString(InputStream stream, String charset) throws IOException {
		try {
			BufferedReader reader = new BufferedReader(new InputStreamReader(stream, charset));
			StringWriter writer = new StringWriter();
			char[] buffer = new char[BUFFER_SIZE];
			int length = 0;
			while ((length = reader.read(buffer)) > 0) {
				writer.write(buffer, 0, length);
			}
			return writer.toString();
		} finally {
			close(stream);
		}
	}

	/**
	 * 获取请求响应的字符编码
	 * @param contentType 内容类型
	 * @return
	 */
	private static String getResponseCharset(String contentType) {
		String charset = DEFAULT_CHARSET;
		if (!isEmpty(contentType)) {
			String[] params = contentType.split(";");
			for (String param : params) {
				param = param.trim();
				if (param.startsWith("charset")) {
					String[] pair = param.split("=", 2);
					if ((pair.length != 2) || (isEmpty(pair[1])))
						break;
					charset = pair[1].trim();
					break;
				}
			}
		}
		return charset;
	}

	/**
	 * 判断字符串是否为空
	 * @param msg
	 * @return
	 */
	private static boolean isEmpty(Object msg) {
		return msg == null || msg.toString().trim().length() == 0;
	}

	/**
	 * 关闭资源
	 * @param object 要关闭的对象
	 */
	public static void close(Closeable object) {
		try {
			if (object != null) {
				object.close();
			}
		} catch (Exception e) {
			logger.warn(e.getMessage(), e);
		}
	}

	/**
	 * 关闭资源
	 * @param conn 要关闭的对象
	 */
	public static void close(HttpURLConnection conn) {
		try {
			if (conn != null) {
				close(conn.getInputStream());
				close(conn.getErrorStream());
				conn.disconnect();
			}
		} catch (Exception e) {
			logger.warn(e.getMessage(), e);
		}
	}

	/**
	 * HTTP请求响应结果
	 * @author huangsq
	 * @version 1.0, 2018-02-19
	 */
	public static class HttpResult implements Serializable {

		private static final long serialVersionUID = 1L;

		private boolean success;
		private String message;
		private int responseCode;

		public boolean isSuccess() {
			return success;
		}

		public void setSuccess(boolean success) {
			this.success = success;
		}

		public String getMessage() {
			return message;
		}

		public void setMessage(String message) {
			this.message = message;
		}

		public int getResponseCode() {
			return responseCode;
		}

		public void setResponseCode(int responseCode) {
			this.responseCode = responseCode;
		}

	}

	/**
	 * HTTP请求参数
	 * @author huangsq
	 * @version 1.0, 2018-02-19
	 */
	public static class HttpParameter implements Serializable {

		private static final long serialVersionUID = 1L;

		private String url;
		private int connectTimeout;
		private int readTimeout;
		private String method;
		private byte[] data;
		private Map<String, String> params;
		private String charset;
		private Map<String, String> property;

		public HttpParameter(String url) {
			this(url, CONNECT_TIMEOUT, READ_TIMEOUT, METHOD_POST, null, DEFAULT_CHARSET, new HashMap<String, String>());
		}

		public HttpParameter(String url, int connectTimeout, int readTimeout) {
			this(url, connectTimeout, readTimeout, METHOD_POST, null, DEFAULT_CHARSET, new HashMap<String, String>());
		}

		public HttpParameter(String url, int connectTimeout, int readTimeout, String method) {
			this(url, connectTimeout, readTimeout, method, null, DEFAULT_CHARSET, new HashMap<String, String>());
		}

		public HttpParameter(String url, int connectTimeout, int readTimeout, String method, Map<String, String> params) {
			this(url, connectTimeout, readTimeout, method, params, DEFAULT_CHARSET, new HashMap<String, String>());
		}

		public HttpParameter(String url, int connectTimeout, int readTimeout, String method, byte[] data) {
			this(url, connectTimeout, readTimeout, method, null, data, DEFAULT_CHARSET, new HashMap<String, String>());
		}

		public HttpParameter(String url, int connectTimeout, int readTimeout, String method, Map<String, String> params,
				String charset) {
			this(url, connectTimeout, readTimeout, method, params, charset, new HashMap<String, String>());
		}

		public HttpParameter(String url, int connectTimeout, int readTimeout, String method, Map<String, String> params,
				String charset, Map<String, String> property) {
			this(url, connectTimeout, readTimeout, method, params, null, charset, property);
		}

		public HttpParameter(String url, int connectTimeout, int readTimeout, String method, Map<String, String> params,
				byte[] data, String charset, Map<String, String> property) {
			super();
			this.url = url;
			this.connectTimeout = connectTimeout;
			this.readTimeout = readTimeout;
			this.method = method;
			this.params = params;
			this.data = data;
			this.charset = charset;
			this.property = property;
		}

		public String getUrl() {
			return url;
		}

		public void setUrl(String url) {
			this.url = url;
		}

		public int getConnectTimeout() {
			return connectTimeout;
		}

		public void setConnectTimeout(int connectTimeout) {
			this.connectTimeout = connectTimeout;
		}

		public int getReadTimeout() {
			return readTimeout;
		}

		public void setReadTimeout(int readTimeout) {
			this.readTimeout = readTimeout;
		}

		public String getMethod() {
			return method;
		}

		public void setMethod(String method) {
			this.method = method;
		}

		public byte[] getData() {
			return data;
		}

		public void setData(byte[] content) {
			this.data = content;
		}

		public Map<String, String> getParams() {
			return params;
		}

		public void setParams(Map<String, String> params) {
			this.params = params;
		}

		public String getCharset() {
			return charset;
		}

		public void setCharset(String charset) {
			this.charset = charset;
		}

		public Map<String, String> getProperty() {
			if (property == null) {
				property = new HashMap<String, String>();
			}
			return property;
		}

		public void setProperty(Map<String, String> property) {
			this.property = property;
		}

		public void putDefaultProperty(String key, String value) {
			Map<String, String> property = getProperty();
			String result = property.get(key);
			if (result == null) {
				property.put(key, value);
			}
		}

	}

}
