package cn.elwy.common.util;

import java.lang.reflect.Type;
import java.text.ParseException;
import java.util.Calendar;
import java.util.Date;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;

import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONException;
import com.alibaba.fastjson.JSONObject;
import com.alibaba.fastjson.TypeReference;
import com.alibaba.fastjson.parser.DefaultJSONParser;
import com.alibaba.fastjson.parser.JSONScanner;
import com.alibaba.fastjson.parser.JSONToken;
import com.alibaba.fastjson.parser.ParserConfig;
import com.alibaba.fastjson.parser.deserializer.AbstractDateDeserializer;
import com.alibaba.fastjson.serializer.JSONSerializer;
import com.alibaba.fastjson.serializer.PropertyPreFilter;
import com.alibaba.fastjson.serializer.SerializeConfig;
import com.alibaba.fastjson.serializer.SerializerFeature;

/**
 * FastJson 工具类
 * @author huangsq
 * @version 1.0, 2018-02-19
 */
public final class FastJsonUtil {
	private static SerializeConfig mapping = new SerializeConfig();
	private static ParserConfig config = ParserConfig.getGlobalInstance();

	static {
		MyDateDeserializer deserializer = new MyDateDeserializer();
		config.putDeserializer(Date.class, deserializer);
		config.putDeserializer(Calendar.class, deserializer);
		// mapping.put(Date.class, new MyDateDeserializer(dateFormat));
	}

	/**
	 * 将JAVA对象包装成JSON格式.
	 * @param object 要转换为json格式的对象
	 * @return 转换后的JSON字符串
	 */
	public static String toJson(Object object) {
		JSONObject.toJSONString(object, SerializerFeature.SortField);
		return JSON.toJSONString(object, mapping);
	}

	/**
	 * 将JAVA对象包装成JSON格式.
	 * @param object 要转换为json格式的对象
	 * @param isExecute 是否排除字段，false表示包含字段
	 * @return propertyNames 过滤的属性
	 */
	public static String toJson(Object object, boolean isExecute, String... propertyNames) {
		PropertyFilter propertyFilter = new PropertyFilter(isExecute, propertyNames);
		return JSON.toJSONString(object, propertyFilter);
	}

	/**
	 * 根据json字符串和对象类型，转换成相应的对象
	 * @param json json字符串
	 * @param clazz 对象类型
	 * @return
	 */
	public static <T> T toObject(String json, Class<T> clazz) {
		return JSON.parseObject(json, clazz);
	}

	/**
	 * 根据json字符串和对象类型，转换成相应的对象
	 * @param json json字符串
	 * @param clazz 对象类型
	 * @return
	 */
	public static <T> T toObject(String json, Class<T> clazz, String... propertyNames) {
		return JSON.parseObject(json, clazz);
	}

	/**
	 * 根据json字符串，转换成一个String型的List
	 * @param json json字符串
	 * @return
	 */
	public static List<String> toList(String json) {
		return JSON.parseObject(json, new TypeReference<List<String>>() {
		});
	}

	/**
	 * 根据json字符串和对象类型，转换成一个相应型的List
	 * @param json json字符串
	 * @param clazz 对象类型
	 * @return
	 */
	public static <T> List<T> toList(String json, Class<T> clazz) {
		return JSON.parseArray(json, clazz);
	}

	/**
	 * 根据json字符串，转换成一个Map类型的List
	 * @param json json字符串
	 * @return
	 */
	public static List<Map<String, Object>> toListMap(String json) {
		return JSON.parseObject(json, new TypeReference<List<Map<String, Object>>>() {
		});
	}

}

class MyDateDeserializer extends AbstractDateDeserializer {

	@SuppressWarnings("unchecked")
	protected <T> T cast(DefaultJSONParser parser, Type clazz, Object fieldName, Object val) {
		if (val == null) {
			return null;
		}

		if (val instanceof Date) {
			return (T) val;
		} else if (val instanceof Number) {
			return (T) new Date(((Number) val).longValue());
		} else if (val instanceof String) {
			String strVal = ((String) val).trim();
			int length = strVal.length();
			if (length == 0) {
				return null;
			}
			JSONScanner dateLexer = new JSONScanner(strVal);
			try {
				if (dateLexer.scanISO8601DateIfMatch(false)) {
					Calendar calendar = dateLexer.getCalendar();
					if (clazz == Calendar.class) {
						return (T) calendar;
					}
					return (T) calendar.getTime();
				}
			} finally {
				dateLexer.close();
			}
			try {
				Date date = DateUtil.format(strVal);
				if (date != null) {
					return (T) date;
				}
			} catch (ParseException err) {
			}
			long longVal = Long.parseLong(strVal);
			return (T) new java.util.Date(longVal);
		}
		throw new JSONException("parse error");
	}

	public int getFastMatchToken() {
		return JSONToken.LITERAL_INT;
	}

}

class PropertyFilter implements PropertyPreFilter {

	private final Class<?> clazz;
	private final Set<String> includes = new HashSet<String>();
	private final Set<String> excludes = new HashSet<String>();

	public PropertyFilter(boolean isExecute, String... properties) {
		this(null, isExecute, properties);
	}

	public PropertyFilter(Class<?> clazz, boolean isExecute, String... properties) {
		super();
		this.clazz = clazz;
		if (properties != null) {
			Set<String> filter = excludes;
			if (!isExecute) {
				filter = includes;
			}
			for (String item : properties) {
				if (item != null) {
					filter.add(item);
				}
			}
		}
	}

	public Class<?> getClazz() {
		return clazz;
	}

	public Set<String> getIncludes() {
		return includes;
	}

	public Set<String> getExcludes() {
		return excludes;
	}

	public boolean apply(JSONSerializer serializer, Object source, String name) {
		if (source == null) {
			return true;
		}

		if (clazz != null && !clazz.isInstance(source)) {
			return true;
		}

		if (this.excludes.contains(name)) {
			return false;
		}

		if (includes.size() == 0 || includes.contains(name)) {
			return true;
		}

		return false;
	}

}
