package cn.elwy.common.util;

import java.text.DateFormat;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.Calendar;
import java.util.Date;
import java.util.Map;
import java.util.TimeZone;
import java.util.concurrent.ConcurrentHashMap;

import cn.elwy.common.log.Logger;
import cn.elwy.common.log.LoggerFactory;

/**
 * 时间工具类
 * @author huangsq
 * @version 1.0, 2018-02-19
 */
public final class DateUtil {

	private static final Logger logger = LoggerFactory.getLogger(DateUtil.class);

	private static TimeZone timeZone;
	private static final Class<SimpleDateFormat> sdf = SimpleDateFormat.class;

	/** 一秒毫秒数 */
	public static final long MILLIS_PER_SECOND = 1000;
	/** 一分钟毫秒数 */
	public static final long MILLIS_PER_MINUTE = 60 * MILLIS_PER_SECOND;
	/** 一小时毫秒数 */
	public static final long MILLIS_PER_HOUR = 60 * MILLIS_PER_MINUTE;
	/** 一天毫秒数 */
	public static final long MILLIS_PER_DAY = 24 * MILLIS_PER_HOUR;

	public static final String Y_M_D_H_M_S = "yyyy-MM-dd HH:mm:ss";
	public static final String Y_M_D_H_M_S_ = "yyyy/MM/dd HH:mm:ss";
	public static final String Y_M_D_H_M_S_S = "yyyy-MM-dd HH:mm:ss.SSS";
	public static final String Y_M_D_H_M_S_S_ = "yyyy/MM/dd HH:mm:ss.SSS";
	public static final String Y_M_D_H_M = "yyyy-MM-dd HH:mm";
	public static final String Y_M_D_H_M_ = "yyyy/MM/dd HH:mm";
	public static final String Y_M_D = "yyyy-MM-dd";
	public static final String Y_M_D_ = "yyyy/MM/dd";
	public static final String YEAR = "yyyy";
	public static final String MONTH = "MM";
	public static final String DAY = "dd";
	public static final String WEEK = "E";
	public static final String YMD = "yyyyMMdd";
	public static final String YMDHMS = "yyyyMMddHHmmss";
	public static final String YMDHMSS = "yyyyMMddHHmmssSSS";
	public static final String H_M_S = "HH:mm:ss";
	public static final String HMS = "HHmmss";
	public static final String Y年M月 = "yyyy年MM月";
	public static final String Y年M月D日 = "yyyy年MM月dd日 ";
	public static final String Y年M月D日H时M分 = "yyyy年MM月dd日 HH时mm分";
	public static final String Y年M月D日H时M分S秒 = "yyyy年MM月dd日 HH时mm分ss秒";

	public static final ThreadLocal<DateFormat> Y_M_D_H_M_S_F = new ThreadLocalType<DateFormat>(sdf, Y_M_D_H_M_S);
	public static final ThreadLocal<DateFormat> Y_M_D_H_M_S__F = new ThreadLocalType<DateFormat>(sdf, Y_M_D_H_M_S_);
	public static final ThreadLocal<DateFormat> Y_M_D_H_M_S_S_F = new ThreadLocalType<DateFormat>(sdf, Y_M_D_H_M_S_S);
	public static final ThreadLocal<DateFormat> Y_M_D_H_M_S_S__F = new ThreadLocalType<DateFormat>(sdf, Y_M_D_H_M_S_S_);
	public static final ThreadLocal<DateFormat> Y_M_D_H_M_F = new ThreadLocalType<DateFormat>(sdf, Y_M_D_H_M);
	public static final ThreadLocal<DateFormat> Y_M_D_H_M__F = new ThreadLocalType<DateFormat>(sdf, Y_M_D_H_M_);
	public static final ThreadLocal<DateFormat> Y_M_D_F = new ThreadLocalType<DateFormat>(sdf, Y_M_D);
	public static final ThreadLocal<DateFormat> Y_M_D__F = new ThreadLocalType<DateFormat>(sdf, Y_M_D_);
	public static final ThreadLocal<DateFormat> YEAR_F = new ThreadLocalType<DateFormat>(sdf, YEAR);
	public static final ThreadLocal<DateFormat> MONTH_F = new ThreadLocalType<DateFormat>(sdf, MONTH);
	public static final ThreadLocal<DateFormat> DAY_F = new ThreadLocalType<DateFormat>(sdf, DAY);
	public static final ThreadLocal<DateFormat> WEEK_F = new ThreadLocalType<DateFormat>(sdf, WEEK);
	public static final ThreadLocal<DateFormat> YMD_F = new ThreadLocalType<DateFormat>(sdf, YMD);
	public static final ThreadLocal<DateFormat> YMDHMS_F = new ThreadLocalType<DateFormat>(sdf, YMDHMS);
	public static final ThreadLocal<DateFormat> YMDHMSS_F = new ThreadLocalType<DateFormat>(sdf, YMDHMSS);
	public static final ThreadLocal<DateFormat> H_M_S_F = new ThreadLocalType<DateFormat>(sdf, H_M_S);
	public static final ThreadLocal<DateFormat> HMS_F = new ThreadLocalType<DateFormat>(sdf, HMS);
	public static final ThreadLocal<DateFormat> Y年M月_F = new ThreadLocalType<DateFormat>(sdf, Y年M月);
	public static final ThreadLocal<DateFormat> Y年M月D日_F = new ThreadLocalType<DateFormat>(sdf, Y年M月D日);
	public static final ThreadLocal<DateFormat> Y年M月D日H时M分_F = new ThreadLocalType<DateFormat>(sdf, Y年M月D日H时M分);
	public static final ThreadLocal<DateFormat> Y年M月D日H时M分S秒_F = new ThreadLocalType<DateFormat>(sdf, Y年M月D日H时M分S秒);
	/** 线程ID和数据源连接池的hashCode组成的当前连接Map */
	private static final Map<String, ThreadLocal<DateFormat>> currentDF = new ConcurrentHashMap<String, ThreadLocal<DateFormat>>();

	public static final String MIN_DATE_STRING = "1970-01-01 00:00:00";
	public static final String MAX_DATE_STRING = "9999-12-31 00:00:00";
	public static final Date MIN_DATE;
	public static final Date MAX_DATE;

	static {
		setTimeZone(TimeZone.getTimeZone("GMT+08:00"));

		currentDF.put(Y_M_D_H_M_S, Y_M_D_H_M_S_F);
		currentDF.put(Y_M_D_H_M_S_, Y_M_D_H_M_S__F);
		currentDF.put(Y_M_D_H_M_S_S, Y_M_D_H_M_S_S_F);
		currentDF.put(Y_M_D_H_M_S_S_, Y_M_D_H_M_S_S__F);
		currentDF.put(Y_M_D_H_M, Y_M_D_H_M_F);
		currentDF.put(Y_M_D_H_M_, Y_M_D_H_M__F);
		currentDF.put(Y_M_D, Y_M_D_F);
		currentDF.put(Y_M_D_, Y_M_D__F);
		currentDF.put(YMD, YMD_F);
		currentDF.put(YMDHMS, YMDHMS_F);
		currentDF.put(YMDHMSS, YMDHMSS_F);
		currentDF.put(H_M_S, H_M_S_F);
		currentDF.put(HMS, HMS_F);
		currentDF.put(Y年M月, Y年M月_F);
		currentDF.put(Y年M月D日, Y年M月D日_F);
		currentDF.put(Y年M月D日H时M分, Y年M月D日H时M分_F);
		currentDF.put(Y年M月D日H时M分S秒, Y年M月D日H时M分S秒_F);

		MIN_DATE = parse(MIN_DATE_STRING, new SimpleDateFormat(Y_M_D_H_M_S));
		MAX_DATE = parse(MAX_DATE_STRING, new SimpleDateFormat(Y_M_D_H_M_S));
	}

	private DateUtil() {
	}

	/**
	 * 获取时区
	 * @return the timeZone
	 */
	public static TimeZone getTimeZone() {
		return timeZone;
	}

	/**
	 * 设置显示时区
	 * @param timeZone
	 */
	public static void setTimeZone(TimeZone timeZone) {
		if (timeZone == null || DateUtil.timeZone == timeZone) {
			return;
		}
		DateUtil.timeZone = timeZone;
	}

	/**
	 * 获取当前时间，以yyyyMMddHHmmssSSS格式返回
	 * @return
	 */
	public static String getCurrentDateTimes() {
		return YMDHMSS_F.get().format(new Date());
	}

	/**
	 * 获取当前时间，以yyyy-MM-dd HH:mm:ss格式返回
	 * @return String
	 * @return
	 */
	public static String getCurrentDateTime() {
		return Y_M_D_H_M_S_F.get().format(new Date());
	}

	/**
	 * 获取当前时间，以yyyy-MM-dd格式返回
	 * @return String
	 */
	public static String getCurrentDate() {
		return Y_M_D_F.get().format(new Date());
	}

	/**
	 * 获得当前时间(字符串类型) 格式：2014-12-02 10:38:53
	 * @return String
	 */
	public static String getCurrentTime() {
		return Y_M_D_H_M_S_F.get().format(new Date());
	}

	/**
	 * 得到当前年份字符串 格式（yyyy）
	 */
	public static String getCurrentYear() {
		return YEAR_F.get().format(new Date());
	}

	/**
	 * 得到当前月份字符串 格式（MM）
	 */
	public static String getCurrentMonth() {
		return MONTH_F.get().format(new Date());
	}

	/**
	 * 得到当天字符串 格式（dd）
	 */
	public static String getCurrentDay() {
		return DAY_F.get().format(new Date());
	}

	/**
	 * 得到当前星期字符串 格式（E）星期几
	 */
	public static String getCurrentWeek() {
		return WEEK_F.get().format(new Date());
	}

	/**
	 * 获取过去的天数
	 * @param date
	 * @return
	 */
	public static long pastDays(Date date) {
		long t = new Date().getTime() - date.getTime();
		return t / MILLIS_PER_DAY;
	}

	/**
	 * 获取过去的小时
	 * @param date
	 * @return
	 */
	public static long pastHour(Date date) {
		long t = new Date().getTime() - date.getTime();
		return t / MILLIS_PER_HOUR;
	}

	/**
	 * 获取过去的分钟
	 * @param date
	 * @return
	 */
	public static long pastMinutes(Date date) {
		long t = new Date().getTime() - date.getTime();
		return t / MILLIS_PER_MINUTE;
	}

	/**
	 * 获取两个日期之间的天数
	 * @param before
	 * @param after
	 * @return
	 */
	public static double getDistanceOfTwoDate(Date before, Date after) {
		long beforeTime = before.getTime();
		long afterTime = after.getTime();
		return (afterTime - beforeTime) / MILLIS_PER_DAY;
	}

	/**
	 * 增加N月
	 * @param date
	 * @param n
	 * @return
	 */
	public static Date addMonth(Date date, int n) {
		Calendar calendar = Calendar.getInstance();
		calendar.setTime(date);
		calendar.add(Calendar.MONTH, n);
		return calendar.getTime();
	}

	/**
	 * 增加N天
	 * @param date
	 * @param n
	 * @return
	 */
	public static Date addDay(Date date, int n) {
		Calendar calendar = Calendar.getInstance();
		calendar.setTime(date);
		calendar.add(Calendar.DAY_OF_MONTH, n);
		return calendar.getTime();
	}

	/**
	 * 增加N小时
	 * @param date
	 * @param interval
	 * @return
	 */
	public static Date addHour(Date date, int interval) {
		Calendar calendar = Calendar.getInstance();
		calendar.setTime(date);
		calendar.add(Calendar.HOUR_OF_DAY, interval);
		return calendar.getTime();
	}

	public static Date addMinute(Date date, int interval) {
		Calendar calendar = Calendar.getInstance();
		calendar.setTime(date);
		calendar.add(Calendar.MINUTE, interval);
		return calendar.getTime();
	}

	/**
	 * 是否是同一天
	 */
	public static boolean isSameDay(Date date, Date date2) {
		if (date == null || date2 == null) {
			return false;
		}
		DateFormat df = YMD_F.get();
		return df.format(date).equals(df.format(date2));
	}

	/**
	 * 获取某年某月的天数
	 * @param year int
	 * @param month int 月份[1-12]
	 * @return int
	 */
	public static int getDaysOfMonth(int year, int month) {
		Calendar calendar = Calendar.getInstance();
		calendar.set(year, month - 1, 1);
		return calendar.getActualMaximum(Calendar.DAY_OF_MONTH);
	}

	/**
	 * 获得当前日期
	 * @return int
	 */
	public static int getToday() {
		Calendar calendar = Calendar.getInstance();
		return calendar.get(Calendar.DATE);
	}

	/**
	 * 获得当前月份
	 * @return int
	 */
	public static int getToMonth() {
		Calendar calendar = Calendar.getInstance();
		return calendar.get(Calendar.MONTH) + 1;
	}

	/**
	 * 获得当前年份
	 * @return int
	 */
	public static int getToYear() {
		Calendar calendar = Calendar.getInstance();
		return calendar.get(Calendar.YEAR);
	}

	/**
	 * 返回日期的天
	 * @param date Date
	 * @return int
	 */
	public static int getDay(Date date) {
		Calendar calendar = Calendar.getInstance();
		calendar.setTime(date);
		return calendar.get(Calendar.DATE);
	}

	/**
	 * 返回日期的年
	 * @param date Date
	 * @return int
	 */
	public static int getYear(Date date) {
		Calendar calendar = Calendar.getInstance();
		calendar.setTime(date);
		return calendar.get(Calendar.YEAR);
	}

	/**
	 * 返回日期的月份，1-12
	 * @param date Date
	 * @return int
	 */
	public static int getMonth(Date date) {
		Calendar calendar = Calendar.getInstance();
		calendar.setTime(date);
		return calendar.get(Calendar.MONTH) + 1;
	}

	/**
	 * 计算两个日期相差的天数，如果date2 > date1 返回正数，否则返回负数
	 * @param date1 Date
	 * @param date2 Date
	 * @return long
	 */
	public static long dayDiff(Date date1, Date date2) {
		return (date2.getTime() - date1.getTime()) / 86400000;
	}

	/**
	 * 比较两个日期的年差
	 * @param befor
	 * @param after
	 * @return
	 */
	public static int yearDiff(String before, String after) {
		Date beforeDay = parse(before, Y_M_D);
		Date afterDay = parse(after, Y_M_D);
		return getYear(afterDay) - getYear(beforeDay);
	}

	/**
	 * 比较指定日期与当前日期的差
	 * @param befor
	 * @param after
	 * @return
	 */
	public static int yearDiffCurr(String after) {
		Date beforeDay = new Date();
		Date afterDay = parse(after, Y_M_D);
		return getYear(beforeDay) - getYear(afterDay);
	}

	/**
	 * 比较指定日期与当前日期的差
	 * @param before
	 * @return
	 * @author chenyz
	 */
	public static long dayDiffCurr(String before) {
		Date currDate = DateUtil.parse(getCurrentDate(), Y_M_D);
		Date beforeDate = parse(before, Y_M_D);
		return (currDate.getTime() - beforeDate.getTime()) / 00;

	}

	/**
	 * 获取每月的第一周
	 * @param year
	 * @param month
	 * @return
	 * @author chenyz
	 */
	public static int getFirstWeekdayOfMonth(int year, int month) {
		Calendar c = Calendar.getInstance();
		c.setFirstDayOfWeek(Calendar.SATURDAY); // 鏄熸湡澶╀负绗竴澶?
		c.set(year, month - 1, 1);
		return c.get(Calendar.DAY_OF_WEEK);
	}

	/**
	 * 获取每月的最后一周
	 * @param year
	 * @param month
	 * @return
	 * @author chenyz
	 */
	public static int getLastWeekdayOfMonth(int year, int month) {
		Calendar c = Calendar.getInstance();
		c.setFirstDayOfWeek(Calendar.SATURDAY); // 鏄熸湡澶╀负绗竴澶?
		c.set(year, month - 1, getDaysOfMonth(year, month));
		return c.get(Calendar.DAY_OF_WEEK);
	}

	/**
	 * 获取指定日期所属年份的第一天，即当年的1月1日
	 * @param baseDate
	 * @return
	 */
	public static Date getFirstDay(Date baseDate) {
		Date firstDate = null;
		Calendar calendar = null;
		if (null != baseDate) {
			calendar = convertToCalendar(baseDate);
			calendar.set(calendar.get(Calendar.YEAR), 0, 1);
			firstDate = calendar.getTime();
		}
		return firstDate;
	}

	/**
	 * 转换Date类型为Caleadar类型
	 * @param date
	 * @return calendar
	 */

	public static Calendar convertToCalendar(Date date) {
		Calendar calendar = null;
		if (null != date) {
			calendar = Calendar.getInstance();
			calendar.setTime(date);
		}
		return calendar;
	}

	/**
	 * 根据指定日期获取当月的第一个自然日，即当月第一天
	 * @param date 指定日期
	 * @return 当月第一天
	 */
	public static Date getFirstNatureDayOnMonth(Date date) {
		Date lastDate = null;
		Calendar calendar = null;
		if (null != date) {
			// 转换日期格式
			calendar = convertToCalendar(date);
			// 获取当期月份的最小天数
			int days = calendar.getActualMinimum(Calendar.DAY_OF_MONTH);
			// 按yyyy-MM-dd格式返回所需日期
			calendar.set(Calendar.DAY_OF_MONTH, days);

			lastDate = calendar.getTime();
		}
		return lastDate;

	}

	/**
	 * 根据指定日期获取当月的最后一个自然日，即当月最后一天
	 * @param date 需要判断的日期
	 * @return 返回当月最后一个自然日
	 */
	public static Date getLastNatureDayOnMonth(Date date) {
		Date lastDate = null;
		Calendar calendar = null;
		if (null != date) {
			// 转换日期格式
			calendar = convertToCalendar(date);
			// 获取当期月份的最大天数
			int days = calendar.getActualMaximum(Calendar.DAY_OF_MONTH);
			// 按yyyy-MM-dd格式返回所需日期
			calendar.set(Calendar.DAY_OF_MONTH, days);

			lastDate = calendar.getTime();
		}
		return lastDate;
	}

	/**
	 * 获取日期所在周的第一个自然日
	 * @param date 基准日期
	 * @return
	 */
	public static Date getFirstNatureDayOnWeek(Date date) {
		Date firstDate = null;
		Calendar calendar = null;
		if (null != date) {
			calendar = convertToCalendar(date);
			// 获取当期星期的最小天数
			int days = calendar.getActualMinimum(Calendar.DAY_OF_WEEK);
			// 按yyyy-MM-dd格式返回所需日期
			calendar.set(Calendar.DAY_OF_WEEK, days);
			firstDate = calendar.getTime();
		}
		return firstDate;
	}

	/**
	 * 获取日期所在周的最后一个自然日
	 * @param date 基准日期
	 * @return
	 */
	public static Date getLastNatureDayOnWeek(Date date) {
		Date lastDate = null;
		Calendar calendar = null;
		if (null != date) {
			calendar = convertToCalendar(date);
			// 获取当期星期的最小天数
			int days = calendar.getActualMaximum(Calendar.DAY_OF_WEEK);
			// 按yyyy-MM-dd格式返回所需日期
			calendar.set(Calendar.DAY_OF_WEEK, days);
			lastDate = calendar.getTime();
		}
		return lastDate;
	}

	/**
	 * 获取日期所在半年的第一个自然日
	 * @param date 基准日期
	 * @return
	 */
	public static Date getFirstNatureDayOnHalfYear(Date date) {
		Date lastDate = null;
		Calendar calendar = null;
		if (null != date) {
			calendar = convertToCalendar(date);
			int currentMonth = calendar.get(Calendar.MONTH) + 1;

			if (currentMonth >= 1 && currentMonth <= 6) {
				calendar.set(Calendar.MONTH, 0);
			} else {
				calendar.set(Calendar.MONTH, 6);
			}
			calendar.set(Calendar.DATE, 1);

			lastDate = calendar.getTime();
		}
		return lastDate;
	}

	/**
	 * 获取日期所在周的第一个自然日
	 * @param date 基准日期
	 * @return
	 */
	public static Date getFirstNatureDayOnYear(Date date) {
		Date firstDate = null;
		Calendar calendar = null;
		if (null != date) {
			calendar = convertToCalendar(date);
			// 获取当前年的最小天数
			int days = calendar.getActualMinimum(Calendar.DAY_OF_YEAR);
			// 按yyyy-MM-dd格式返回所需日期
			calendar.set(Calendar.DAY_OF_YEAR, days);
			firstDate = calendar.getTime();
		}
		return firstDate;
	}

	/**
	 * 根据指定日期获取当前的最后一个自然日
	 * @param date
	 * @return 返回当年最后一个自然日
	 */
	public static Date getLastNatureDayOnYear(Date date) {
		Date lastDate = null;
		Calendar calendar = null;
		if (null != date) {
			calendar = convertToCalendar(date);
			// 获取当前年的最大天数
			int days = calendar.getActualMaximum(Calendar.DAY_OF_YEAR);
			// 按yyyy-MM-dd格式返回所需日期
			calendar.set(Calendar.DAY_OF_YEAR, days);
			lastDate = calendar.getTime();
		}
		return lastDate;
	}

	/**
	 * 根据指定日期获取该日期所在季度的最后一个自然日
	 * @param date 需要判断的日期
	 * @return 返回当前季度最后一个自然日
	 */
	public static Date getLastNatureDayOnQuarter(Date date) {
		Date lastDate = null;
		Calendar calendar = null;
		if (null != date) {
			calendar = convertToCalendar(date);
			int currentMonth = calendar.get(Calendar.MONTH) + 1;

			if (currentMonth >= 1 && currentMonth <= 3) {
				calendar.set(Calendar.MONTH, 2);
				calendar.set(Calendar.DATE, 31);
			} else if (currentMonth >= 4 && currentMonth <= 6) {
				calendar.set(Calendar.MONTH, 5);
				calendar.set(Calendar.DATE, 30);
			} else if (currentMonth >= 7 && currentMonth <= 9) {
				calendar.set(Calendar.MONTH, 8);
				calendar.set(Calendar.DATE, 30);
			} else if (currentMonth >= 10 && currentMonth <= 12) {
				calendar.set(Calendar.MONTH, 11);
				calendar.set(Calendar.DATE, 31);
			}

			lastDate = calendar.getTime();
		}
		return lastDate;
	}

	/**
	 * CL 20121101 根据指定日期获取该日期所在季度的第一个自然日
	 * @param date
	 * @return
	 */
	public static Date getFirstNatureDayOnQuarter(Date date) {
		Date lastDate = null;
		Calendar calendar = null;
		if (null != date) {
			calendar = convertToCalendar(date);
			int currentMonth = calendar.get(Calendar.MONTH) + 1;

			if (currentMonth >= 1 && currentMonth <= 3) {
				calendar.set(Calendar.MONTH, 0);
			} else if (currentMonth >= 4 && currentMonth <= 6) {
				calendar.set(Calendar.MONTH, 3);
			} else if (currentMonth >= 7 && currentMonth <= 9) {
				calendar.set(Calendar.MONTH, 6);
			} else if (currentMonth >= 10 && currentMonth <= 12) {
				calendar.set(Calendar.MONTH, 9);
			}
			calendar.set(Calendar.DATE, 1);

			lastDate = calendar.getTime();
		}
		return lastDate;
	}

	/**
	 * fj 20130726 根据指定日期获取该日期所在季度最后一个月的第一个自然日
	 * @param date
	 * @return
	 */
	public static Date getFirstNatureDayOnQuarterEnd(Date date) {
		Date lastDate = null;
		Calendar calendar = null;
		if (null != date) {
			calendar = convertToCalendar(date);
			int currentMonth = calendar.get(Calendar.MONTH) + 1;

			if (currentMonth >= 1 && currentMonth <= 3) {
				calendar.set(Calendar.MONTH, 2);
			} else if (currentMonth >= 4 && currentMonth <= 6) {
				calendar.set(Calendar.MONTH, 5);
			} else if (currentMonth >= 7 && currentMonth <= 9) {
				calendar.set(Calendar.MONTH, 8);
			} else if (currentMonth >= 10 && currentMonth <= 12) {
				calendar.set(Calendar.MONTH, 11);
			}
			calendar.set(Calendar.DATE, 1);

			lastDate = calendar.getTime();
		}
		return lastDate;
	}

	////
	//
	///////////////
	/**
	 * 可以获取昨天的日期 格式：yyyy-MM-dd
	 * @return String
	 */
	// public static String getYesterdayYMD() {
	// Date date = new Date(System.currentTimeMillis() - DATEMM * 1000L);
	// String str = Y_M_D_F.get().format(date);
	// try {
	// date = yyyyMddHmmss.parse(str + " 00:00:00");
	// return yyyyMdd.format(date);
	// } catch (ParseException e) {
	// e.printStackTrace();
	// }
	// return "";
	// }

	/**
	 * 将字符串按对应格式转换为日期，根据日期字符串的长度自动识别日期格式
	 * @param date
	 * @return
	 * @throws ParseException
	 */
	public static Date format(String date) throws ParseException {
		if (AssertUtil.isEmpty(date)) {
			return null;
		}
		int length = date.length();
		if (length == 19) {
			try {
				return Y_M_D_H_M_S_F.get().parse(date);
			} catch (ParseException e) {
				return Y_M_D_H_M_S__F.get().parse(date);
			}
		} else if (length == 23) {
			try {
				return Y_M_D_H_M_S_S_F.get().parse(date);
			} catch (ParseException e) {
				return Y_M_D_H_M_S_S__F.get().parse(date);
			}
		} else if (length == 16) {
			try {
				return Y_M_D_H_M_F.get().parse(date);
			} catch (ParseException e) {
				return Y_M_D_H_M__F.get().parse(date);
			}
		} else if (length == 10) {
			try {
				return Y_M_D_F.get().parse(date);
			} catch (ParseException e) {
				return Y_M_D__F.get().parse(date);
			}
		} else if (length == 11) {
			return Y年M月D日_F.get().parse(date);
		} else if (length == 18) {
			return Y_M_D_F.get().parse(date);
		} else if (length == 21) {
			return Y年M月D日H时M分S秒_F.get().parse(date);
		} else if (length == 17) {
			return YMDHMSS_F.get().parse(date);
		} else if (length == 14) {
			return YMDHMS_F.get().parse(date);
		} else if (length == 8) {
			try {
				return YMD_F.get().parse(date);
			} catch (ParseException e) {
				try {
					return H_M_S_F.get().parse(date);
				} catch (ParseException e1) {
					return Y年M月_F.get().parse(date);
				}
			}
		}
		return null;
	}

	/**
	 * 将字符串按对应格式转换为日期
	 * @param dateString 日期字符串
	 * @param pattern 日期格式
	 * @return 日期型对象
	 */
	public static Date parse(String dateString, String pattern) {
		DateFormat format = getDateFormat(pattern);
		return parse(dateString, format);
	}

	/**
	 * 将字符串按对应格式转换为日期，格式化失败返回缺省值
	 * @param dateString 日期字符串
	 * @param pattern 日期格式
	 * @param defaultValue 缺省值
	 * @return 日期型对象
	 */
	public static Date parse(String dateString, String pattern, Date defaultValue) {
		try {
			return parse(dateString, pattern);
		} catch (Exception e) {
			logger.warn(e.getMessage(), e);
			return defaultValue;
		}
	}

	/**
	 * 将字符串按对应格式转换为日期
	 * @param dateString 日期字符串
	 * @param format 字符串对应的日期格式
	 * @return 日期型对象
	 */
	public static Date parse(String dateString, DateFormat format) {
		try {
			return format.parse(dateString);
		} catch (Exception e) {
			throw new RuntimeException(dateString, e);
		}
	}

	/**
	 * 将字符串按对应格式转换为日期，格式化失败返回缺省值
	 * @param dateString 日期字符串
	 * @param format 字符串对应的日期格式
	 * @param defaultValue 缺省值
	 * @return 日期型对象
	 */
	public static Date parse(String dateString, DateFormat format, Date defaultValue) {
		try {
			return parse(dateString, format);
		} catch (Exception e) {
			logger.warn(e.getMessage(), e);
			return defaultValue;
		}
	}

	/**
	 * 根据提供的数值转换成相应的日期对象
	 * @param year 年
	 * @param month 月(0-11)
	 * @param date 日
	 * @return 转换好的日期对象
	 */
	public static Date parse(int year, int month, int date) {
		return parse(year, month, date, 0, 0, 0);
	}

	/**
	 * 根据提供的数值转换成相应的日期对象
	 * @param year 年
	 * @param month 月(0-11)
	 * @param date 日
	 * @param hour 时(24小时制)
	 * @param minute 分
	 * @param second 秒
	 * @return 转换好的日期对象
	 */
	public static Date parse(int year, int month, int day, int hours, int minutes, int seconds) {
		Calendar calendar = Calendar.getInstance();
		calendar.clear();
		calendar.set(year, month, day, hours, minutes, seconds);
		return calendar.getTime();
	}

	/**
	 * 对日期进行格式化
	 * @param date 需格式化的日期
	 * @param pattern 日期格式
	 * @return 格式化后的字符串
	 */
	public static String format(Date date, String pattern) {
		DateFormat format = getDateFormat(pattern);
		return format.format(date);
	}

	/**
	 * 对日期进行格式化，格式化失败返回缺省值
	 * @param date 需格式化的日期
	 * @param pattern 日期格式
	 * @param defaultValue 缺省值
	 * @return 格式化后的字符串
	 */
	public static String format(Date date, String pattern, String defaultValue) {
		try {
			return format(date, pattern);
		} catch (Exception e) {
			logger.warn(e.getMessage(), e);
			return defaultValue;
		}
	}

	/**
	 * 根据格式，对字符串进行格式化
	 * @param dateString 日期字符串
	 * @param pattern 日期格式
	 * @return 格式化后的字符串
	 */
	public static String format(String dateString, String pattern) {
		DateFormat format = getDateFormat(pattern);
		return format.format(dateString);
	}

	/**
	 * 根据格式，对字符串进行格式化，格式化失败返回缺省值
	 * @param dateString 日期字符串
	 * @param pattern 日期格式
	 * @param defaultValue 缺省值
	 * @return 格式化后的字符串
	 */
	public static String format(String dateString, String pattern, String defaultValue) {
		try {
			return format(dateString, pattern);
		} catch (Exception e) {
			logger.warn(e.getMessage(), e);
			return defaultValue;
		}
	}

	/**
	 * 对日期进行格式化 格式为：yyyy-MM-dd
	 * @param date 需格式化的日期
	 * @return 格式化后的字符串
	 */
	public static String formatDate(Date date, String defaultValue) {
		return format(date, Y_M_D, defaultValue);
	}

	/**
	 * 对日期进行格式化 格式为：HH:mm:ss
	 * @param date 需格式化的日期
	 * @param defaultValue 缺省值
	 * @return 格式化后的字符串
	 */
	public static String formatTime(Date date, String defaultValue) {
		return format(date, H_M_S, defaultValue);
	}

	/**
	 * 对日期进行格式化 格式为：yyyy-MM-dd HH:mm:ss
	 * @param date 需格式化的日期
	 * @return 格式化后的字符串
	 */
	public static String formatDateTime(Date date, String defaultValue) {
		return format(date, Y_M_D_H_M_S, defaultValue);
	}

	/**
	 * 将字符串根据"yyyy-MM-dd"格式化成日期，格式化失败返回缺省值
	 * @param dateString 日期字符串
	 * @param defaultValue 缺省值
	 * @return
	 */
	public static Date toDate(String dateString, Date defaultValue) {
		return parse(dateString, Y_M_D, defaultValue);
	}

	/**
	 * 将字符串根据"HH:mm:ss"格式化成时间，格式化失败返回缺省值
	 * @param dateString 时间字符串
	 * @param defaultValue 缺省值
	 * @return
	 */
	public static Date toTime(String dateString, Date defaultValue) {
		try {
			return parse(dateString, H_M_S);
		} catch (Exception e) {
			logger.warn(e.getMessage(), e);
			return defaultValue;
		}
	}

	/**
	 * 将字符串根据"yyyy-MM-dd HH:mm:ss"格式化成日期，格式化失败返回缺省值
	 * @param dateString 日期字符串
	 * @param defaultValue 缺省值
	 * @return
	 */
	public static Date toDateTime(String dateString, Date defaultValue) {
		try {
			return parse(dateString, Y_M_D_H_M_S);
		} catch (Exception e) {
			logger.warn(e.getMessage(), e);
			return defaultValue;
		}
	}

	/**
	 * 获得昨天零点
	 * @return Date
	 */
	public static Date getYesterdayZeroHour() {
		Calendar cal = Calendar.getInstance();
		cal.add(Calendar.DATE, -1);
		cal.set(Calendar.SECOND, 0);
		cal.set(Calendar.MINUTE, 0);
		cal.set(Calendar.HOUR, 0);
		return cal.getTime();
	}

	/**
	 * 获得今天零点
	 * @return Date
	 */
	public static Date getTodayZeroHour() {
		Calendar cal = Calendar.getInstance();
		cal.set(Calendar.SECOND, 0);
		cal.set(Calendar.MINUTE, 0);
		cal.set(Calendar.HOUR, 0);
		return cal.getTime();
	}

	/**
	 * 获得昨天23时59分59秒
	 * @return
	 */
	public static Date getYesterday24Hour() {
		Calendar cal = Calendar.getInstance();
		cal.add(Calendar.DATE, -1);
		cal.set(Calendar.SECOND, 59);
		cal.set(Calendar.MINUTE, 59);
		cal.set(Calendar.HOUR, 23);
		return cal.getTime();
	}

	/**
	 * 获得指定日期所在的自然周的第一天，即周日
	 * @param date 日期
	 * @return 自然周的第一天
	 */
	public static Date getStartDayOfWeek(Date date) {
		Calendar c = Calendar.getInstance();
		c.setTime(date);
		c.set(Calendar.DAY_OF_WEEK, 1);
		date = c.getTime();
		return date;
	}

	/**
	 * 获得指定日期所在的自然周的最后一天，即周六
	 * @param date
	 * @return
	 */
	public static Date getLastDayOfWeek(Date date) {
		Calendar c = Calendar.getInstance();
		c.setTime(date);
		c.set(Calendar.DAY_OF_WEEK, 7);
		date = c.getTime();
		return date;
	}

	/**
	 * 获得指定日期所在当月第一天
	 * @param date
	 * @return
	 */
	public static Date getStartDayOfMonth(Date date) {
		Calendar c = Calendar.getInstance();
		c.setTime(date);
		c.set(Calendar.DAY_OF_MONTH, 1);
		date = c.getTime();
		return date;
	}

	/**
	 * 获得指定日期所在当月最后一天
	 * @param date
	 * @return
	 */
	public static Date getLastDayOfMonth(Date date) {
		Calendar c = Calendar.getInstance();
		c.setTime(date);
		c.set(Calendar.DATE, 1);
		c.add(Calendar.MONTH, 1);
		c.add(Calendar.DATE, -1);
		date = c.getTime();
		return date;
	}

	/**
	 * 获得指定日期的下一个月的第一天
	 * @param date
	 * @return
	 */
	public static Date getStartDayOfNextMonth(Date date) {
		Calendar c = Calendar.getInstance();
		c.setTime(date);
		c.add(Calendar.MONTH, 1);
		c.set(Calendar.DAY_OF_MONTH, 1);
		date = c.getTime();
		return date;
	}

	/**
	 * 获得指定日期的下一个月的最后一天
	 * @param date
	 * @return
	 */
	public static Date getLastDayOfNextMonth(Date date) {
		Calendar c = Calendar.getInstance();
		c.setTime(date);
		c.set(Calendar.DATE, 1);
		c.add(Calendar.MONTH, 2);
		c.add(Calendar.DATE, -1);
		date = c.getTime();
		return date;
	}

	/**
	 * 根据格式，对字符串进行格式化
	 * @param dateString 日期字符串
	 * @param pattern 日期格式
	 * @return 格式化后的Calendar
	 */
	public static Calendar getCalendar(String dateString, String pattern) {
		Date parseDate = parse(dateString, pattern);
		Calendar calendar = Calendar.getInstance();
		calendar.setTime(parseDate);
		return calendar;
	}

	public static DateFormat getDateFormat(String pattern) {
		ThreadLocal<DateFormat> threadLocal = currentDF.get(pattern);
		if (threadLocal != null) {
			return threadLocal.get();
		} else {
			DateFormat format = new SimpleDateFormat(pattern);
			format = new SimpleDateFormat(pattern);
			format.setTimeZone(timeZone);
			return format;
		}
	}

}
