package cn.elwy.common.util;

import java.beans.BeanInfo;
import java.beans.Introspector;
import java.beans.PropertyDescriptor;
import java.lang.reflect.Method;
import java.math.BigDecimal;
import java.math.BigInteger;
import java.text.DateFormat;
import java.text.DecimalFormat;
import java.text.NumberFormat;
import java.text.ParseException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;

import cn.elwy.common.log.Logger;
import cn.elwy.common.log.LoggerFactory;

/**
 * 数据类型转换工具
 * @author huangsq
 * @version 1.0, 2018-02-19
 */
public final class ConvertTypeUtil {

	private static final Logger logger = LoggerFactory.getLogger(ConvertTypeUtil.class);

	private static String HEX_STRING = "0123456789ABCDEF";
	private static char[] HEX_CHAR = HEX_STRING.toCharArray();

	private static Map<String, NumberFormat> numberFormats = new HashMap<String, NumberFormat>();

	private ConvertTypeUtil() {
	}

	/**
	 * 根据模式获取相应的数字格式
	 * @param pattern
	 * @return
	 */
	public static NumberFormat getNumberFormat(String pattern) {
		NumberFormat format = numberFormats.get(pattern);
		if (format == null) {
			format = new DecimalFormat(pattern);
			numberFormats.put(pattern, format);
		}
		return format;
	}

	/**
	 * 根据“,”分隔拆分字符串，以字符串数组的形式返回
	 * @param string 字符串
	 * @return
	 */
	public static String[] toArray(String string) {
		return toArray(string, ",");
	}

	/**
	 * 根据分隔符拆分字符串，以字符串数组的形式返回
	 * @param string 字符串
	 * @param separate 分隔符
	 * @return
	 */
	public static String[] toArray(String string, String separate) {
		return string.split(separate);
	}

	/**
	 * 根据“,”分隔拆分字符串，以字符串列表的形式返回
	 * @param string 字符串
	 * @return
	 */
	public static List<String> toList(String string) {
		return toList(string, ",");
	}

	/**
	 * 根据分隔符拆分字符串，以字符串列表的形式返回
	 * @param string 字符串
	 * @param separate 分隔符
	 * @return
	 */
	public static List<String> toList(String string, String separator) {
		List<String> list = null;
		if (!isEmpty(string)) {
			list = new ArrayList<String>();
			String[] args = string.split(separator);
			for (String value : args) {
				list.add(value);
			}
		} else {
			list = Collections.emptyList();
		}
		return list;
	}

	/**
	 * 将字符串转换成Boolean类型，如果转换失败将返回false
	 * @param value 字符串
	 * @return
	 */
	public static Boolean toBoolean(Object value) {
		return toBoolean(value, false);
	}

	/**
	 * 将字符串转换成Boolean类型，如果转换失败将返回defaultValue
	 * @param value 字符串
	 * @param defaultValue 默认值
	 * @return
	 */
	public static Boolean toBoolean(Object value, Boolean defaultValue) {
		if (isEmpty(value)) {
			return defaultValue;
		}
		if (value instanceof Number) {
			return ((Number) value).intValue() > 0;
		}
		value = value.toString().trim().toLowerCase();
		if ("true".equals(value) || "1".equals(value) || "yes".equals(value) || "y".equals(value) || "on".equals(value)
				|| "ok".equals(value) || "是".equals(value) || "真".equals(value)) {
			return true;
		} else {
			return false;
		}
	}

	/**
	 * 将字符串数组转换成Boolean类型数组，如果转换失败将返回defaultValue
	 * @param stringArray 字符串数组
	 * @param defaultValue 默认值
	 * @return
	 */
	public static Boolean[] toBooleanArray(String[] stringArray, Boolean defaultValue) {
		if (isEmpty(stringArray)) {
			return null;
		}
		Boolean[] result = new Boolean[stringArray.length];
		for (int i = 0; i < result.length; i++) {
			result[i] = toBoolean(stringArray[i], defaultValue);
		}
		return result;
	}

	/**
	 * int值转成4字节的byte数组
	 * @param num
	 * @return
	 */
	public static byte[] toByteArray(int num) {
		byte[] result = new byte[4];
		result[0] = (byte) (num >>> 24);// 取最高8位放到0下标
		result[1] = (byte) (num >>> 16);// 取次高8为放到1下标
		result[2] = (byte) (num >>> 8); // 取次低8位放到2下标
		result[3] = (byte) (num); // 取最低8位放到3下标
		return result;
	}

	/**
	 * 将null转换成""
	 * @param string
	 * @return
	 */
	public static String toEmpty(String string) {
		if (string == null) {
			return "";
		}
		return string;
	}

	/**
	 * 将字符串转换成枚举类型
	 * @param enumType 枚举类型的类名
	 * @param enumName 枚举类型名称
	 * @return
	 */
	@SuppressWarnings({ "rawtypes", "unchecked" })
	public static Enum toEnum(Class enumType, String enumName) {
		if (!"".equals(enumName)) {
			return Enum.valueOf(enumType, enumName);
		}
		return null;
	}

	/**
	 * 将字符串转换成枚举类型
	 * @param <T>
	 * @param enumType 枚举类型的类名
	 * @param enumName 枚举类型名称
	 * @param defaultValue 缺省值
	 * @return
	 */
	@SuppressWarnings({ "rawtypes", "unchecked" })
	public static <T> T toEnum(Class enumType, String enumName, Enum defaultValue) {
		try {
			Enum result = toEnum(enumType, enumName);
			if (result != null) {
				return (T) result;
			}
		} catch (Exception e) {
			e.printStackTrace();
		}
		return (T) defaultValue;
	}

	/**
	 * 将4字节的byte数组转成int值
	 * @param b
	 * @return
	 */
	public static int toInt(byte[] b) {
		byte[] a = new byte[4];
		int i = a.length - 1, j = b.length - 1;
		for (; i >= 0; i--, j--) {// 从b的尾部(即int值的低位)开始copy数据
			if (j >= 0)
				a[i] = b[j];
			else
				a[i] = 0;// 如果b.length不足4,则将高位补0
		}
		int v0 = (a[0] & 0xff) << 24;// &0xff将byte值无差异转成int,避免Java自动类型提升后,会保留高位的符号位
		int v1 = (a[1] & 0xff) << 16;
		int v2 = (a[2] & 0xff) << 8;
		int v3 = (a[3] & 0xff);
		return v0 + v1 + v2 + v3;
	}

	/**
	 * 十六进制转换成字节
	 * @param hex 十六进制字符串
	 * @return byte[] 转换后的字节
	 */
	public static byte[] toHexBytes(String hex) {
		char[] hex2char = hex.toCharArray();
		byte[] bytes = new byte[hex.length() / 2];
		int temp;
		for (int i = 0; i < bytes.length; i++) {
			temp = HEX_STRING.indexOf(hex2char[2 * i]) * 16;
			temp += HEX_STRING.indexOf(hex2char[2 * i + 1]);
			bytes[i] = (byte) (temp & 0xff);
		}
		return bytes;
	}

	/**
	 * 将字节码转换成十六进制字符串
	 * @param bytes
	 * @return 十六进制字符串
	 */
	public static String toHex(byte[] bytes) {
		StringBuilder sb = new StringBuilder(bytes.length * 2);
		for (int i = 0; i < bytes.length; i++) {
			sb.append(HEX_CHAR[(bytes[i] & 0xf0) >>> 4]);
			sb.append(HEX_CHAR[bytes[i] & 0x0f]);
		}
		return sb.toString();
	}

	/**
	 * 将字符串转换成Integer类型，如果转换失败将返回0
	 * @param string 字符串
	 * @return
	 */
	public static Integer toInteger(String string) {
		return toInteger(string, 0);
	}

	/**
	 * 将字符串转换成Integer类型，如果转换失败将返回defaultValue
	 * @param string 字符串
	 * @param defaultValue 默认值
	 * @return
	 */
	public static Integer toInteger(String string, Integer defaultValue) {
		Integer result = defaultValue;
		if (isEmpty(string)) {
			return result;
		} else {
			string = string.trim();
		}
		try {
			result = (int) Double.parseDouble(string);
		} catch (Exception e) {
			logger.warn(e.getMessage(), e);
		}
		return result;
	}

	/**
	 * 将字符串数组转换成Integer类型数组，如果转换失败将返回defaultValue
	 * @param stringArray 字符串数组
	 * @param defaultValue 默认值
	 * @return
	 */
	public static Integer[] toIntegerArray(String[] stringArray, Integer defaultValue) {
		if (isEmpty(stringArray)) {
			return null;
		}
		Integer[] result = new Integer[stringArray.length];
		for (int i = 0; i < result.length; i++) {
			result[i] = toInteger(stringArray[i], defaultValue);
		}
		return result;
	}

	public static Long toLong(String string) {
		return toLong(string, Long.valueOf(0L));
	}

	public static Long toLong(String string, Long defaultValue) {
		Long result = defaultValue;
		if (isEmpty(string)) {
			return result;
		}
		string = string.trim();
		try {
			result = Long.valueOf(Long.parseLong(string));
		} catch (Exception e) {
			logger.warn(e.getMessage(), e);
		}
		return result;
	}

	public static Long[] toLongArray(String[] stringArray, Long defaultValue) {
		if (isEmpty(stringArray)) {
			return null;
		}
		Long[] result = new Long[stringArray.length];
		for (int i = 0; i < result.length; i++) {
			result[i] = toLong(stringArray[i], defaultValue);
		}
		return result;
	}

	/**
	 * 将"null"转换成null
	 * @param string
	 * @return
	 */
	public static String toNull(String string) {
		if (string != null && "null".equalsIgnoreCase(string.trim())) {
			return null;
		}
		return string;
	}

	/**
	 * 将字符串转换成Number类型，默认值不能为空，如果默认值为空或转换失败将返回defaultValue
	 * @param string 字符串
	 * @param defaultValue 默认值
	 * @return
	 */
	public static Number toNumber(String string, Number defaultValue) {
		if (defaultValue == null) {
			return defaultValue;
		} else {
			return toNumber(string, defaultValue.getClass(), defaultValue);
		}
	}

	/**
	 * 将字符串转换成Number类型，如果转换失败将返回defaultValue，默认值的类型和targetType的类型必须一致
	 * @param string 字符串
	 * @param targetType 转换的目标数字类型
	 * @param defaultValue 默认值
	 * @return
	 */
	public static Number toNumber(String string, Class<?> targetType, Number defaultValue) {
		Number result = defaultValue;
		if (isEmpty(string)) {
			return result;
		} else {
			string = string.trim();
		}
		try {
			if (Integer.class.equals(targetType)) {
				result = Integer.parseInt(string);
			} else if (Long.class.equals(targetType)) {
				result = Long.parseLong(string);
			} else if (Float.class.equals(targetType)) {
				result = Float.parseFloat(string);
			} else if (Double.class.equals(targetType)) {
				result = Double.parseDouble(string);
			} else if (BigDecimal.class.equals(targetType)) {
				result = new BigDecimal(string);
			} else if (BigInteger.class.equals(targetType)) {
				result = new BigInteger(string);
			} else if (Short.class.equals(targetType)) {
				result = Short.parseShort(string);
			} else if (Byte.class.equals(targetType)) {
				result = Byte.parseByte(string);
			}
		} catch (Exception e) {
			logger.warn(e.getMessage(), e);
		}
		return result;
	}

	/**
	 * 将字符串转换成Number类型，如果转换失败将返回defaultValue
	 * @param string 字符串
	 * @param format 格式
	 * @param defaultValue 默认值
	 * @return
	 */
	public static Number toNumber(String string, NumberFormat format, Number defaultValue) {
		try {
			return format.parse(string);
		} catch (ParseException e) {
			logger.warn(e.getMessage(), e);
			return defaultValue;
		}
	}

	/**
	 * 将字符串根据格式格式化成数值，格式化失败返回defaultValue
	 * @param string 字符串
	 * @param pattern 格式字符串
	 * @param defaultValue 默认值
	 * @return
	 */
	public static Number toNumber(String string, String pattern, Number defaultValue) {
		return toNumber(string, getNumberFormat(pattern), defaultValue);
	}

	/**
	 * 将字符串数组转换成Number类型数组，如果转换失败将返回defaultValue
	 * @param stringArray 字符串数组
	 * @param defaultValue 默认值
	 * @return
	 */
	public static Number[] toNumberArray(String[] stringArray, Number defaultValue) {
		if (isEmpty(stringArray)) {
			return null;
		}
		Number[] result = new Number[stringArray.length];
		for (int i = 0; i < result.length; i++) {
			result[i] = toNumber(stringArray[i], defaultValue);
		}
		return result;
	}

	/**
	 * 将Date类型格式化成字符串，如果格式化失败将返回defaultValue
	 * @param date
	 * @param dateFormat
	 * @param defaultValue
	 * @return
	 */
	public static String toString(Date date, DateFormat dateFormat, String defaultValue) {
		try {
			return dateFormat.format(date);
		} catch (Exception e) {
			logger.warn(e.getMessage(), e);
			return defaultValue;
		}
	}

	public static String toString(Collection<String> object) {
		return toString(object, "\n");
	}

	public static String toString(Collection<String> object, String separate) {
		StringBuffer buffer = new StringBuffer();
		for (String entry : object) {
			buffer.append(entry).append(separate);
		}
		return buffer.toString();
	}

	public static String toString(Map<String, Object> object) {
		return toString(object, "\n");
	}

	public static String toString(Map<String, Object> object, String separate) {
		StringBuffer buffer = new StringBuffer();
		Set<Map.Entry<String, Object>> sets = object.entrySet();
		for (Map.Entry<String, Object> entry : sets) {
			buffer.append(entry.getKey() + entry.getValue());
			buffer.append(separate);
		}
		return buffer.toString();
	}

	/**
	 * 将Number类型格式化成字符串，如果格式化失败将返回""
	 * @param object 对象
	 * @param format 格式
	 * @return
	 */
	public static String toString(Number object, DecimalFormat format) {
		return toString(object, format, "");
	}

	/**
	 * 将Number类型格式化成字符串，如果格式化失败将返回defaultValue
	 * @param object 对象
	 * @param format 格式
	 * @param defaultValue 默认值
	 * @return
	 */
	public static String toString(Number object, NumberFormat format, String defaultValue) {
		try {
			return format.format(object);
		} catch (Exception e) {
			logger.warn(e.getMessage(), e);
			return defaultValue;
		}
	}

	/**
	 * 将Number类型格式化成字符串，如果格式化失败将返回""
	 * @param object 对象
	 * @param pattern 格式字符串
	 * @return
	 */
	public static String toString(Number object, String pattern) {
		return toString(object, getNumberFormat(pattern), "");
	}

	/**
	 * 将Number类型格式化成字符串，如果格式化失败将返回defaultValue
	 * @param object 对象
	 * @param pattern 格式字符串
	 * @param defaultValue 默认值
	 * @return
	 */
	public static String toString(Number object, String pattern, String defaultValue) {
		return toString(object, getNumberFormat(pattern), defaultValue);
	}

	/**
	 * 将对象转换成字符串，object为null默认返回""
	 * @param object
	 * @return
	 */
	public static String toString(Object object) {
		return toString(object, "");
	}

	/**
	 * 将对象转换成字符串，object为null默认返回""
	 * @param object
	 * @param defaultValue
	 * @return
	 */
	public static String toString(Object object, String defaultValue) {
		if (object != null) {
			return object.toString();
		} else {
			return defaultValue;
		}
	}

	/**
	 * Map --> Bean 1: 利用Introspector,PropertyDescriptor实现 Map --> Bean
	 * @param msgMaps
	 * @param object
	 * @return
	 * @throws Exception
	 */
	public static <T> T toBean(Map<String, Object> map, T object) throws Exception {
		BeanInfo beanInfo = Introspector.getBeanInfo(object.getClass());
		PropertyDescriptor[] propertyDescriptors = beanInfo.getPropertyDescriptors();
		for (PropertyDescriptor property : propertyDescriptors) {
			String key = property.getName();
			if (map.containsKey(key)) {
				Object value = map.get(key);
				// 得到property对应的setter方法
				Method setter = property.getWriteMethod();
				setter.invoke(object, value);
			}
		}
		return object;
	}

	/**
	 * Bean --> Map 1: 利用Introspector和PropertyDescriptor 将Bean --> Map
	 * @param <T>
	 * @param object
	 * @return
	 * @throws Exception
	 */
	@SuppressWarnings("unchecked")
	public static <T> Map<String, T> toMap(Object object) throws Exception {
		if (object == null) {
			return null;
		}
		Map<String, Object> map = new HashMap<String, Object>();
		BeanInfo beanInfo = Introspector.getBeanInfo(object.getClass());
		PropertyDescriptor[] propertyDescriptors = beanInfo.getPropertyDescriptors();
		for (PropertyDescriptor property : propertyDescriptors) {
			String key = property.getName();
			// 过滤class属性
			if (!key.equals("class")) {
				// 得到property对应的getter方法
				Method getter = property.getReadMethod();
				Object value = getter.invoke(object);
				map.put(key, value);
			}
		}
		return (Map<String, T>) map;
	}

	/**
	 * 将集合数据根据分隔符连接成字符串
	 * @param arrays 对像列表
	 * @param separator 分隔符，默认为",";
	 * @return
	 * @author huangsq
	 */
	public static String join(String separator, Object... arrays) {
		String result = "";
		if (arrays == null) {
			return result;
		}
		if (separator == null) {
			separator = ",";
		}
		StringBuilder sBuilder = new StringBuilder();
		for (Object object : arrays) {
			if (object instanceof Collection<?>) {
				Collection<?> list = (Collection<?>) object;
				for (Object obj : list) {
					sBuilder.append(obj).append(separator);
				}
			} else {
				sBuilder.append(object).append(separator);
			}
		}
		if (sBuilder.length() >= separator.length()) {
			result = sBuilder.substring(0, sBuilder.length() - separator.length());
		}
		return result;
	}

	private static boolean isEmpty(Object[] object) {
		return object == null || object.length < 1;
	}

	private static boolean isEmpty(Object text) {
		return text == null || text.toString().trim().length() < 1;
	}

}
