package cn.elwy.common.util;

import java.beans.BeanInfo;
import java.beans.Introspector;
import java.beans.PropertyDescriptor;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.math.BigDecimal;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.apache.commons.beanutils.BeanUtils;
import org.apache.commons.beanutils.BeanUtilsBean;
import org.apache.commons.beanutils.ConvertUtils;
import org.apache.commons.beanutils.DynaBean;
import org.apache.commons.beanutils.DynaProperty;
import org.apache.commons.beanutils.MethodUtils;
import org.apache.commons.beanutils.PropertyUtils;
import org.apache.commons.beanutils.converters.BigDecimalConverter;
import org.apache.commons.beanutils.converters.DateConverter;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * 修改自Apache的BeanUtils，更简便操作Bean类
 * @author huangsq
 */
@SuppressWarnings({ "rawtypes", "unchecked" })
public class BeanUtil {

	private static final Logger logger = LoggerFactory.getLogger(BeanUtil.class);

	private static Set<String> notCopyProperties = new HashSet<String>();

	static {
		notCopyProperties.add("class");
		notCopyProperties.add("propName");
		notCopyProperties.add("propValue");
		// 注册日期类型格式化
		DateConverter dateConverter = new DateConverter();
		String[] patterns = new String[] { "yyyy-MM-dd HH:mm:ss", "yyyy-MM-dd", "yyyy-MM-dd HH:mm:ss.S",
				"yyyy/MM/dd HH:mm:ss", "yyyy/MM/dd", "yyyy/MM/dd HH:mm:ss.S" };
		dateConverter.setPatterns(patterns);

		ConvertUtils.register(dateConverter, java.util.Date.class);
		ConvertUtils.register(dateConverter, java.sql.Date.class);
		ConvertUtils.register(new BigDecimalConverter(), BigDecimal.class);
	}

	public static void copyProperty(final Object bean, final String name, final Object value) {
		try {
			BeanUtils.copyProperty(bean, name, value);
		} catch (Exception e) {
			throw new RuntimeException(e.getMessage(), e);
		}
	}

	public static String[] getArrayProperty(final Object bean, final String name) {
		try {
			return BeanUtils.getArrayProperty(bean, name);
		} catch (Exception e) {
			throw new RuntimeException(e.getMessage(), e);
		}
	}

	public static String getIndexedProperty(final Object bean, final String name) {
		try {
			return BeanUtils.getIndexedProperty(bean, name);
		} catch (Exception e) {
			throw new RuntimeException(e.getMessage(), e);
		}
	}

	public static String getIndexedProperty(final Object bean, final String name, final int index) {
		try {
			return BeanUtils.getIndexedProperty(bean, name, index);
		} catch (Exception e) {
			throw new RuntimeException(e.getMessage(), e);
		}
	}

	public static String getMappedProperty(final Object bean, final String name) {
		try {
			return BeanUtils.getMappedProperty(bean, name);
		} catch (Exception e) {
			throw new RuntimeException(e.getMessage(), e);
		}

	}

	public static String getMappedProperty(final Object bean, final String name, final String key) {
		try {
			return BeanUtils.getMappedProperty(bean, name, key);
		} catch (Exception e) {
			throw new RuntimeException(e.getMessage(), e);
		}
	}

	public static String getNestedProperty(final Object bean, final String name) {
		try {
			return BeanUtils.getNestedProperty(bean, name);
		} catch (Exception e) {
			throw new RuntimeException(e.getMessage(), e);
		}
	}

	public static String getProperty(final Object bean, final String name) {
		try {
			return BeanUtils.getProperty(bean, name);
		} catch (Exception e) {
			throw new RuntimeException(e.getMessage(), e);
		}
	}

	public static String getSimpleProperty(final Object bean, final String name) {
		try {
			return BeanUtils.getSimpleProperty(bean, name);
		} catch (Exception e) {
			throw new RuntimeException(e.getMessage(), e);
		}
	}

	public static void populate(final Object bean, final Map<String, ? extends Object> properties) {
		try {
			BeanUtils.populate(bean, properties);
		} catch (Exception e) {
			throw new RuntimeException(e.getMessage(), e);
		}
	}

	public static void setProperty(final Object bean, final String name, final Object value) {
		try {
			BeanUtils.setProperty(bean, name, value);
		} catch (Exception e) {
			throw new RuntimeException(e.getMessage(), e);
		}
	}

	public static boolean initCause(final Throwable throwable, final Throwable cause) {
		try {
			return BeanUtils.initCause(throwable, cause);
		} catch (Exception e) {
			throw new RuntimeException(e.getMessage(), e);
		}
	}

	/**
	 * Map转换为JavaBean对象
	 * @param <T>
	 * @param map
	 * @param clazz
	 * @return
	 */
	public static <T> T map2Bean(Map<?, ?> map, Class<T> clazz) {
		T bean = null;
		try {
			bean = clazz.newInstance();
			BeanUtils.populate(bean, (Map<String, ? extends Object>) map);
		} catch (Exception e) {
			throw new RuntimeException(e.getMessage(), e);
		}
		return bean;
	}

	/**
	 * JavaBean转换为Map
	 * @param bean
	 * @return
	 * @throws Exception
	 */
	public static Map<String, Object> bean2Map(Object bean) throws Exception {
		Map<String, Object> map = new HashMap<String, Object>();
		if (bean == null) {
			return map;
		}
		BeanInfo beanInfo = Introspector.getBeanInfo(bean.getClass(), Object.class);
		// 获取所有的属性描述器
		PropertyDescriptor[] pds = beanInfo.getPropertyDescriptors();
		for (PropertyDescriptor pd : pds) {
			String key = pd.getName();
			Method getter = pd.getReadMethod();
			Object value = getter.invoke(bean);
			map.put(key, value);
		}
		return map;
	}

	public static Map<String, Object> describeAvailableParameter(Object bean)
			throws IllegalAccessException, InvocationTargetException, NoSuchMethodException {
		Map<String, Object> description = new HashMap<String, Object>();
		if (bean == null) {
			return description;
		}
		if (bean instanceof DynaBean) {
			DynaProperty[] descriptors = ((DynaBean) bean).getDynaClass().getDynaProperties();
			for (int i = 0; i < descriptors.length; i++) {
				String name = descriptors[i].getName();
				description.put(name, org.apache.commons.beanutils.BeanUtils.getProperty(bean, name));
			}
		} else {
			PropertyDescriptor[] descriptors = BeanUtilsBean.getInstance().getPropertyUtils().getPropertyDescriptors(bean);
			Class<?> clazz = bean.getClass();
			for (int i = 0; i < descriptors.length; i++) {
				String name = descriptors[i].getName();
				if (name.startsWith("$")) {
					// System.out.println("clazz===" + clazz);
					// System.out.println("descriptors[i].getReadMethod()===" +
					// descriptors[i].getReadMethod());
					if (MethodUtils.getAccessibleMethod(clazz, descriptors[i].getReadMethod()) != null) {
						description.put(name, PropertyUtils.getNestedProperty(bean, name));
					}
				}
			}
		}
		return (description);
	}

	public static Map<String, Object> describe(Object bean)
			throws IllegalAccessException, InvocationTargetException, NoSuchMethodException {
		Map<String, Object> description = new HashMap<String, Object>();
		if (bean == null) {
			return description;
		}
		if (bean instanceof DynaBean) {
			DynaProperty[] descriptors = ((DynaBean) bean).getDynaClass().getDynaProperties();
			for (int i = 0; i < descriptors.length; i++) {
				String name = descriptors[i].getName();
				description.put(name, org.apache.commons.beanutils.BeanUtils.getProperty(bean, name));
			}
		} else {
			PropertyDescriptor[] descriptors = BeanUtilsBean.getInstance().getPropertyUtils().getPropertyDescriptors(bean);
			Class<?> clazz = bean.getClass();
			for (int i = 0; i < descriptors.length; i++) {
				String name = descriptors[i].getName();
				if (MethodUtils.getAccessibleMethod(clazz, descriptors[i].getReadMethod()) != null) {
					description.put(name, PropertyUtils.getNestedProperty(bean, name));
				}
			}
		}
		return (description);
	}

	public static void copyProperties(Object dest, Object orig) throws IllegalAccessException, InvocationTargetException {
		copyProperties(dest, orig, null);
	}

	public static void copyProperties(Object dest, Object orig, List<String> excludedProp)
			throws IllegalAccessException, InvocationTargetException {

		// Validate existence of the specified beans
		if (dest == null) {
			throw new IllegalArgumentException("No destination bean specified");
		}
		if (orig == null) {
			throw new IllegalArgumentException("No origin bean specified");
		}
		if (logger.isDebugEnabled()) {
			logger.debug("BeanUtils.copyProperties(" + dest + ", " + orig + ")");
		}

		// Copy the properties, converting as necessary
		BeanUtilsBean bub = BeanUtilsBean.getInstance();
		if (orig instanceof DynaBean) {
			DynaProperty[] origDescriptors = ((DynaBean) orig).getDynaClass().getDynaProperties();
			for (int i = 0; i < origDescriptors.length; i++) {
				String name = origDescriptors[i].getName();
				// Need to check isReadable() for WrapDynaBean
				// (see Jira issue# BEANUTILS-61)
				if (bub.getPropertyUtils().isReadable(orig, name) && bub.getPropertyUtils().isWriteable(dest, name)) {
					Object value = ((DynaBean) orig).get(name);
					if (value != null) {
						bub.copyProperty(dest, name, value);
					}
				}
			}
		} else if (orig instanceof Map) {
			Iterator entries = ((Map) orig).entrySet().iterator();
			while (entries.hasNext()) {
				Map.Entry entry = (Map.Entry) entries.next();
				String name = (String) entry.getKey();
				if (bub.getPropertyUtils().isWriteable(dest, name)) {
					Object value = entry.getValue();
					if (value != null) {
						bub.copyProperty(dest, name, value);
					}
				}
			}
		} else /* if (orig is a standard JavaBean) */ {
			PropertyDescriptor[] origDescriptors = bub.getPropertyUtils().getPropertyDescriptors(orig);
			for (int i = 0; i < origDescriptors.length; i++) {
				String name = origDescriptors[i].getName();
				if (isIgnoreProperties(name, excludedProp)) {
					continue; // No point in trying to set an object's class
				}
				if (bub.getPropertyUtils().isReadable(orig, name) && bub.getPropertyUtils().isWriteable(dest, name)) {
					try {
						Object value = bub.getPropertyUtils().getSimpleProperty(orig, name);
						if (value != null) {
							bub.copyProperty(dest, name, value);
						}
					} catch (NoSuchMethodException e) {
						logger.warn("复制属性异常：" + e.getMessage(), e);
					}
				}
			}
		}
	}

	/**
	 * 是否为忽略的属性，excludedProp为null根据notCopyProperties判断是否忽略
	 * @author huangsq
	 * @param name
	 * @param excludedProp
	 * @return
	 */
	private static boolean isIgnoreProperties(String name, List<String> excludedProp) {
		if (excludedProp != null) {
			return "class".equals(name) || excludedProp.contains(name);
		} else {
			return notCopyProperties.contains(name) || name.startsWith("$");
		}
	}

}
