package cn.elwy.common.util;

import java.util.Collection;
import java.util.Map;

import cn.elwy.common.exception.AssertException;

/**
 * Java断言类
 * @author huangsq
 * @version 1.0, 2018-02-19
 */
public final class Assert {

	private Assert() {
	}

	public static void notEmpty(Object array[]) {
		notEmpty(array, "[Assertion failed] - this array must not be empty: it must contain at least 1 element"); //$NON-NLS-1$
	}

	public static void notEmpty(Object array[], String message) {
		if (AssertUtil.isEmpty(array)) {
			throw new AssertException(message);
		} else {
			return;
		}
	}

	public static void notEmpty(Collection<?> collection) {
		notEmpty(collection, "[Assertion failed] - this collection must not be empty: it must contain at least 1 element");
	}

	public static void notEmpty(Collection<?> collection, String message) {
		if (AssertUtil.isEmpty(collection)) {
			throw new AssertException(message);
		} else {
			return;
		}
	}

	public static void notEmpty(Map<?, ?> map) {
		notEmpty(map, "[Assertion failed] - this msgMaps must not be empty; it must contain at least one entry");
	}

	public static void notEmpty(Map<?, ?> map, String message) {
		if (AssertUtil.isEmpty(map)) {
			throw new AssertException(message);
		} else {
			return;
		}
	}

	public static void notEmpty(Object object) {
		notEmpty(object, "[Assertion failed] - this argument is required; it must not be empty");
	}

	public static void notEmpty(Object object, String message) {
		if (AssertUtil.isEmpty(object)) {
			throw new AssertException(message);
		} else {
			return;
		}
	}

	public static void isEmpty(Object array[]) {
		isEmpty(array, "[Assertion failed] - this array must not be empty: it must contain at least 1 element"); //$NON-NLS-1$
	}

	public static void isEmpty(Collection<?> collection) {
		isEmpty(collection, "[Assertion failed] - this collection must not be empty: it must contain at least 1 element");
	}

	public static void isEmpty(Map<?, ?> map) {
		isEmpty(map, "[Assertion failed] - this msgMaps must not be empty; it must contain at least one entry");
	}

	public static void isEmpty(Object object) {
		isEmpty(object, "[Assertion failed] - this argument is required; it must not be empty");
	}

	public static void isEmpty(Object object, String message) {
		if (AssertUtil.isNotEmpty(object)) {
			throw new AssertException(message);
		} else {
			return;
		}
	}

	public static void isTrue(boolean condition) {
		isTrue(condition, "[Assertion failed] - this expression must be true");
	}

	public static void isTrue(boolean condition, String message) {
		if (!condition) {
			fail(message);
		}
	}

	public static void isFalse(boolean condition) {
		isFalse("[Assertion failed] - this expression must be false", condition);
	}

	public static void isFalse(String message, boolean condition) {
		if (condition) {
			fail(message);
		}
	}

	public static void isNull(Object object) {
		String message = "[Assertion failed] - Expected: <null> but was: " + String.valueOf(object);
		isNull(object, message);
	}

	public static void isNull(Object object, String message) {
		isTrue(object == null, message);
	}

	public static void notNull(Object object) {
		notNull(object, "[Assertion failed] - this argument is required; it must not be null");
	}

	public static void notNull(Object object, String message) {
		isTrue(object != null, message);
	}

	public static void equals(Object expected, Object actual) {
		equals(expected, actual, null);
	}

	public static void equals(Object expected, Object actual, String message) {
		if (expected == null && actual == null) {
			return;
		}
		if (expected != null && expected.equals(actual)) {
			return;
		}
		failNotEquals(expected, actual, message);
	}

	public static void equals(double expected, double actual, double delta) {
		equals(expected, actual, delta, null);
	}

	public static void equals(double expected, double actual, double delta, String message) {
		if (Double.compare(expected, actual) == 0) {
			return;
		}
		if (!(Math.abs(expected - actual) <= delta)) {
			failNotEquals(new Double(expected), new Double(actual), message);
		}
	}

	public static void equals(float expected, float actual, float delta) {
		equals(expected, actual, delta, null);
	}

	public static void equals(float expected, float actual, float delta, String message) {
		if (Float.compare(expected, actual) == 0) {
			return;
		}
		if (!(Math.abs(expected - actual) <= delta)) {
			failNotEquals(new Float(expected), new Float(actual), message);
		}
	}

	public static void equals(long expected, long actual) {
		equals(expected, actual, null);
	}

	public static void equals(long expected, long actual, String message) {
		equals(new Long(expected), new Long(actual), message);
	}

	public static void equals(boolean expected, boolean actual) {
		equals(expected, actual, null);
	}

	public static void equals(boolean expected, boolean actual, String message) {
		equals(Boolean.valueOf(expected), Boolean.valueOf(actual), message);
	}

	public static void equals(byte expected, byte actual) {
		equals(expected, actual, null);
	}

	public static void equals(byte expected, byte actual, String message) {
		equals(new Byte(expected), new Byte(actual), message);
	}

	public static void equals(char expected, char actual) {
		equals(expected, actual, null);
	}

	public static void equals(char expected, char actual, String message) {
		equals(new Character(expected), new Character(actual), message);
	}

	public static void equals(short expected, short actual) {
		equals(expected, actual, null);
	}

	public static void equals(short expected, short actual, String message) {
		equals(new Short(expected), new Short(actual), message);
	}

	public static void equals(int expected, int actual) {
		equals(expected, actual, null);
	}

	public static void equals(int expected, int actual, String message) {
		equals(new Integer(expected), new Integer(actual), message);
	}

	public static void assertSame(Object expected, Object actual) {
		assertSame(expected, actual, null);
	}

	public static void assertSame(Object expected, Object actual, String message) {
		if (expected == actual) {
			return;
		}
		failNotSame(expected, actual, message);
	}

	public static void assertNotSame(Object expected, Object actual) {
		assertNotSame(expected, actual, null);
	}

	public static void assertNotSame(Object expected, Object actual, String message) {
		if (expected == actual) {
			failSame(message);
		}
	}

	public static void fail(String message) {
		if (message == null) {
			throw new AssertException("");
		}
		throw new AssertException(message);
	}

	public static void fail() {
		fail(null);
	}

	public static void failSame(String message) {
		String formatted = "";
		if (message != null) {
			formatted = message + " ";
		}
		fail(formatted + "expected not same");
	}

	public static void failNotSame(Object expected, Object actual, String message) {
		String formatted = "";
		if (message != null) {
			formatted = message + " ";
		}
		fail(formatted + "expected same:<" + expected + "> was not:<" + actual + ">");
	}

	public static void failNotEquals(Object expected, Object actual, String message) {
		fail(format(expected, actual, message));
	}

	public static String format(Object expected, Object actual, String message) {
		String formatted = "";
		if (message != null && message.length() > 0) {
			formatted = message + " ";
		}
		return formatted + "expected:<" + expected + "> but was:<" + actual + ">";
	}

	public static void hasLength(String text) {
		hasLength(text, "[Assertion failed] - this String argument must have length; it must not be null or empty"); //$NON-NLS-1$
	}

	public static void hasLength(String text, String message) {
		if (text == null || text.length() < 1) {
			throw new AssertException(message);
		} else {
			return;
		}
	}

	public static void hasText(String text) {
		hasText(text, "[Assertion failed] - this String argument must have text; it must not be null, empty, or blank");
	}

	public static void hasText(String text, String message) {
		if (!AssertUtil.isNotBlank(text)) {
			throw new AssertException(message);
		} else {
			return;
		}
	}

	public static void doesNotContain(String textToSearch, String substring) {
		StringBuilder append = new StringBuilder();
		append.append("[Assertion failed] - this String argument must not contain the substring [").append(substring)
				.append("]");
		doesNotContain(textToSearch, substring, append.toString());
	}

	public static void doesNotContain(String textToSearch, String substring, String message) {
		if (AssertUtil.isContain(textToSearch, substring)) {
			throw new AssertException(message);
		} else {
			return;
		}
	}

	public static void noNullElements(Object array[]) {
		noNullElements(array, "[Assertion failed] - this array must not contain any null elements"); //$NON-NLS-1$
	}

	public static void noNullElements(Object array[], String message) {
		if (array != null) {
			int length = array.length;
			for (int j = 0; j < length; j++) {
				Object element = array[j];
				if (element == null) {
					throw new AssertException(message);
				}
			}
		}
	}

	public static void isInstanceOf(Class<?> clazz, Object object) {
		StringBuilder message = new StringBuilder().append("Object of class [")
				.append(object == null ? "null" : object.getClass().getName()).append("] must be an instance of ")
				.append(clazz);
		isInstanceOf(clazz, object, message.toString());
	}

	public static void isInstanceOf(Class<?> type, Object object, String message) {
		notNull(type, "Type to check against must not be null");
		if (!type.isInstance(object)) {
			throw new AssertException(message);
		} else {
			return;
		}
	}

	public static void isAssignable(Class<?> superType, Class<?> subType) {
		StringBuilder message = new StringBuilder().append(subType).append(" is not assignable to ").append(superType); //$NON-NLS-1$
		isAssignable(superType, subType, message.toString());
	}

	public static void isAssignable(Class<?> superType, Class<?> subType, String message) {
		notNull(superType, "Type to check against must not be null"); //$NON-NLS-1$
		if (subType == null || !superType.isAssignableFrom(subType)) {
			throw new AssertException(message);
		} else {
			return;
		}
	}

	public static void state(boolean expression) {
		state(expression, "[Assertion failed] - this state invariant must be true"); //$NON-NLS-1$
	}

	public static void state(boolean expression, String message) {
		if (!expression) {
			throw new IllegalStateException(message);
		} else {
			return;
		}
	}

}
