package cn.elwy.common.log;

import java.io.IOException;
import java.io.OutputStream;
import java.text.DecimalFormat;
import java.text.NumberFormat;
import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.Map;

/**
 * 程序执行时间计时器工具类
 * @author huangsq
 * @date 2018-02-19
 */
public class TimerWatchUtil {

	protected static final double MS = 1000000.0;
	protected static final String FORMAT_DATETIME = "yyyyMMddHHmmssS";

	protected final Logger logger = LoggerFactory.getLogger(TimerWatchUtil.class);

	protected ThreadLocal<Map<String, TimerWatch>> timerMap = new ThreadLocal<Map<String, TimerWatch>>() {
		@Override
		protected Map<String, TimerWatch> initialValue() {
			return new LinkedHashMap<String, TimerWatch>();
		}
	};

	protected static ThreadLocal<NumberFormat> msFormat = new ThreadLocal<NumberFormat>() {
		@Override
		protected NumberFormat initialValue() {
			return new DecimalFormat(",##0.00");
		}
	};

	protected static ThreadLocal<SimpleDateFormat> sdFormat = new ThreadLocal<SimpleDateFormat>() {
		@Override
		protected SimpleDateFormat initialValue() {
			return new SimpleDateFormat(FORMAT_DATETIME);
		}
	};

	protected TimerWatchUtil() {
	}

	private static class LazyHolder {
		private static final TimerWatchUtil INSTANCE = new TimerWatchUtil();
	}

	public static TimerWatchUtil getInstance() {
		return LazyHolder.INSTANCE;
	}

	public TimerWatchUtil start(String id, Object data) {
		TimerWatch timerWatch = timerMap.get().get(id);
		if (timerWatch == null) {
			timerWatch = new TimerWatch();
			timerMap.get().put(id, timerWatch);
		}
		timerWatch.data = data;
		timerWatch.startTime = sdFormat.get().format(new Date());
		timerWatch.start = System.nanoTime() / MS;
		return this;
	}

	public boolean end(String id) {
		TimerWatch timerWatch = timerMap.get().get(id);
		if (timerWatch == null) {
			logger.warn("[TimerWatch] undefined timer: " + id);
			return false;
		}
		timerWatch.end();
		timerWatch.endTime = sdFormat.get().format(new Date());
		return true;
	}

	public boolean end(String id, Object data) {
		TimerWatch timerWatch = timerMap.get().get(id);
		if (timerWatch == null) {
			logger.warn("[TimerWatch] undefined timer: " + id);
			return false;
		}
		timerWatch.data = data;
		return true;
	}

	public boolean endAndDebug(String id) {
		boolean end = end(id);
		if (end) {
			logger.debug(getTimeInfo(id));
		}
		return end;
	}

	public boolean endAndOutput(String id) {
		boolean end = end(id);
		if (end) {
			logger.info(getTimeInfo(id));
		}
		return end;
	}

	/**
	 * 输出计时器信息(DEBUG)
	 * @author huangsq
	 */
	public void debugTimer() {
		logger.debug(getTimerInfo());
	}

	/**
	 * 输出计时器信息(INFO)
	 * @author huangsq
	 */
	public void outputTimer() {
		logger.info(getTimerInfo());
	}

	/**
	 * 输出计时器信息到指定的输出流，输出流为空默认输出INFO信息
	 * @author huangsq
	 * @param os
	 */
	public void outputTimer(OutputStream os) {
		if (os != null) {
			String outputTimer = getTimerInfo();
			try {
				os.write(outputTimer.getBytes());
			} catch (IOException e) {
				logger.error(outputTimer, e);
			}
		} else {
			outputTimer();
		}
	}

	public static String toString(Object source) {
		NumberFormat format = msFormat.get();
		return format.format(source);
	}

	public ThreadLocal<Map<String, TimerWatch>> getTimerMap() {
		return timerMap;
	}

	public static void main(String[] args) {
		TimerWatchUtil instance = TimerWatchUtil.getInstance();
		instance.start("id", "test1");
		for (int i = 0; i < 100000; i++) {
		}
		instance.end("id");
		instance.outputTimer();

		instance.start("id2", "test2");
		for (int i = 0; i < 100000; i++) {
		}
		instance.start("id3", "test3");
		instance.end("id2");
		instance.endAndOutput("id3");

		instance.endAndOutput("id13");
		instance.outputTimer();
		instance.debugTimer();
	}

	private String getTimerInfo() {
		double total = 0;
		StringBuilder result = new StringBuilder();
		Iterator<Map.Entry<String, TimerWatch>> it = timerMap.get().entrySet().iterator();
		while (it.hasNext()) {
			final TimerWatch timerWatch = it.next().getValue();
			total += timerWatch.elapse;
			getTimerInfo(result, timerWatch);
			result.append("\r\n");
			it.remove();
		}
		if (result.length() > 1) {
			result.append("[TimerWatch] -------------------- total used time: ").append(toString(total));
		}
		return result.toString();
	}

	private String getTimeInfo(String id) {
		TimerWatch timerWatch = timerMap.get().get(id);
		StringBuilder result = new StringBuilder();
		if (timerWatch != null) {
			getTimerInfo(result, timerWatch);
			timerMap.get().remove(id);
		}
		return result.toString();
	}

	private void getTimerInfo(StringBuilder result, TimerWatch entry) {
		result.append("[TimerWatch] ").append(entry.data).append(" Start Time: ").append(entry.startTime)
				.append(" End Time: ").append(entry.endTime);
		result.append(" used time(ms): ").append(toString(entry.elapse));
	}

	class TimerWatch {
		private Object data;
		private String startTime;
		private String endTime;
		private double start = 0;
		private double elapse = 0;

		public TimerWatch end() {
			elapse += System.nanoTime() / MS - start;
			return this;
		}
	}

}