package cn.elwy.common.log;

import java.io.InputStream;
import java.io.Serializable;
import java.net.URL;
import java.util.Iterator;
import java.util.List;
import java.util.Map.Entry;
import java.util.Properties;

import org.slf4j.LoggerFactory;

import ch.qos.logback.classic.Logger;
import ch.qos.logback.classic.LoggerContext;
import ch.qos.logback.classic.joran.JoranConfigurator;
import ch.qos.logback.classic.spi.ILoggingEvent;
import ch.qos.logback.core.Appender;
import ch.qos.logback.core.util.StatusPrinter;
import cn.elwy.common.Constant;
import cn.elwy.common.exception.RunException;
import cn.elwy.common.util.CloseUtil;
import cn.elwy.common.util.io.PropertyUtil;

/**
 * 日志配置类
 * @author huangsq
 * @version 1.0, 2018-02-19
 */
public class LoggerConfig implements Constant, Serializable {

	private static final long serialVersionUID = 1L;

	private static volatile LoggerConfig instance;
	private URL logbackFile;
	private LoggerContext loggerContext;

	private LoggerConfig() {
	}

	public static LoggerConfig getInstance() {
		if (instance == null) {
			synchronized (LoggerConfig.class) {
				if (instance == null) {
					instance = new LoggerConfig();
				}
			}
		}
		return instance;
	}

	/**
	 * 配置生效
	 */
	public synchronized void doConfig(URL logbackFile) {
		doConfig(logbackFile, null);
	}

	/**
	 * 配置生效
	 */
	public synchronized void doConfig(URL logbackFile, URL logbackConfigFile) {
		try {
			load(logbackConfigFile);
			loggerContext = (LoggerContext) LoggerFactory.getILoggerFactory();
			JoranConfigurator configurator = new JoranConfigurator();
			configurator.setContext(loggerContext);
			loggerContext.reset();
			configurator.doConfigure(logbackFile);
			StatusPrinter.print(loggerContext);
		} catch (Exception e) {
			throw new RunException(e.getMessage(), e);
		} finally {
			StatusPrinter.printInCaseOfErrorsOrWarnings(loggerContext);
		}
	}

	protected void load(URL logbackConfig) {
		InputStream is = null;
		try {
			if (logbackConfig == null) {
				return;
			}
			PropertyUtil property = new PropertyUtil();
			is = logbackConfig.openStream();
			property.load(is);

			Properties props = property.getProps();
			if (props != null) {
				Iterator<Entry<Object, Object>> it = props.entrySet().iterator();
				while (it.hasNext()) {
					Entry<Object, Object> entry = it.next();
					Object key = entry.getKey();
					Object value = entry.getValue();
					if (key != null && value != null) {
						System.setProperty(key.toString(), value.toString());
					}
				}
			}
		} catch (Exception e) {
			e.printStackTrace();
		} finally {
			CloseUtil.close(is);
		}
	}

	public Appender<ILoggingEvent> getMemoryBufferAppender() {
		return loggerContext.getLogger("root").getAppender("WEB-CONSOLE-OUT");
	}

	public synchronized void stop() {
		try {
			List<Logger> list = loggerContext.getLoggerList();
			for (Logger logger : list) {
				Iterator<Appender<ILoggingEvent>> it = logger.iteratorForAppenders();
				while (it.hasNext()) {
					Appender<ILoggingEvent> appender = it.next();
					appender.clearAllFilters();
					appender.stop();
					appender = null;
				}
			}
		} catch (Throwable e) {
			System.out.println(e.getMessage());
		}
		if (!loggerContext.isStarted())
			loggerContext.stop();

		loggerContext.getStatusManager().clear();
		try {
			loggerContext.getExecutorService().shutdown();
		} catch (Throwable e) {
		}
		loggerContext = null;
	}

	public LoggerContext getLoggerContext() {
		return loggerContext;
	}

	public URL getLogbackFile() {
		return logbackFile;
	}

	public void setLogbackFile(URL logbackFile) {
		this.logbackFile = logbackFile;
	}

}
