package cn.elwy.common.jdbc;

import java.io.File;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;
import java.io.PrintWriter;
import java.io.StringWriter;
import java.sql.Blob;
import java.sql.Connection;
import java.sql.ResultSet;
import java.sql.ResultSetMetaData;
import java.sql.SQLException;
import java.sql.SQLWarning;
import java.sql.Statement;
import java.sql.Types;
import java.util.ArrayList;
import java.util.List;
import java.util.regex.Pattern;

import javax.sql.DataSource;

import cn.elwy.common.log.Logger;
import cn.elwy.common.log.LoggerFactory;
import cn.elwy.common.util.AssertUtil;
import cn.elwy.common.util.CloseUtil;
import cn.elwy.common.util.io.FileParser;

/**
 * 执行SQL文件
 * @author huangsq
 * @version 1.0, 2018-02-19
 */
public class SQLExec {

	private static Logger logger = LoggerFactory.getLogger(SQLExec.class);

	/** 文件编码 */
	private String encoding;
	/** 忽略大小写 */
	private boolean ignoreCase = true;
	/** 保持文件格式 */
	private boolean keepFormat = false;
	/** 命令定界符 */
	private String delimiter = ";";
	/** 忽略的行 */
	private List<String> ignoreLineList;
	/** 忽略的错误号 */
	private List<Integer> ignoreErrorCodeList;
	/** 总行数 */
	private int totalSql = 0;
	/** 成功行数 */
	private int goodSql = 0;

	/** 错误处理方式，默认出错终止 */
	private OnError onError = OnError.ABORT;

	/** 日志输出目标 */
	private PrintWriter printWriter;

	/** 数据源 */
	private DataSource dataSource;
	/** 自动提交事务 */
	private boolean autoCommit = false;
	/** 输出结果 */
	private boolean printResult = false;

	/** 输出执行的SQL语句 */
	private boolean printSQL = false;

	/** 输出详细的错误信息 */
	private boolean printDetailMessage = false;

	/** 输出查询结果的表头 (header columns) */
	private boolean printHeader = true;

	/** 输出SQL语句执行状态 (rows affected) */
	private boolean printRowAffected = false;

	/** 输出BLob数据 */
	private boolean printBlob = false;

	/** 输出SQLWarnings信息 */
	private boolean printWarning = false;

	/** 显示列时使用的分隔符 */
	private String csvColumnSep = ",";

	/** 发生警告，按错误方式处理 */
	private boolean treatWarningsAsErrors = false;

	/**
	 * Argument to Statement.setEscapeProcessing
	 */
	private boolean escapeProcessing = true;

	/**
	 * The character used to quote column values.
	 * <p>
	 * If set, columns that contain either the column separator or the quote
	 * character itself will be surrounded by the quote character. The quote
	 * character itself will be doubled if it appears inside of the column's value.
	 * </p>
	 * <p>
	 * If this value is not set (the default), no column values will be quoted, not
	 * even if they contain the column separator.
	 * </p>
	 * <p>
	 * <b>Note:<b> BLOB values will never be quoted.
	 * </p>
	 * <p>
	 * Defaults to "not set"
	 * </p>
	 */
	private String csvQuoteChar = null;

	private FileParser fileParser;

	public SQLExec() {
		this(null);
	}

	public SQLExec(DataSource dataSource) {
		this.dataSource = dataSource;

		ignoreLineList = new ArrayList<String>();
		ignoreLineList.add("^--|^//|^\\#|^prompt|^spool|^set |^pl/sql|^commit");

		ignoreErrorCodeList = new ArrayList<Integer>();
		// 942: 表或视图不存在
		ignoreErrorCodeList.add(942);
		// 2289: 序列不存在
		ignoreErrorCodeList.add(2289);
		// 2443: 无法删除约束条件 - 不存在的约束条件
		ignoreErrorCodeList.add(2443);
		// // 955: 名称已由现有对象使用
		// ignoreErrorCodeList.add(955);
		// // 2260: 表只能具有一个主键
		// ignoreErrorCodeList.add(2260);
		// // 2275: 此表中已经存在这样的引用约束条件
		// ignoreErrorCodeList.add(2275);
		// // 2261: 表中已存在这样的唯一关键字或主键
		// ignoreErrorCodeList.add(2261);
	}

	public void initFileParser() {
		fileParser = new FileParser(delimiter, true, ignoreLineList);
		fileParser.setKeepFormat(keepFormat);
	}

	/**
	 * 加载并执行SQL文件
	 * @param filePath SQL文件路径
	 * @throws IOException
	 * @throws Exception
	 */
	public boolean execute(String filePath) throws Exception {
		return execute(new File(filePath));
	}

	/**
	 * 加载并执行SQL文件
	 * @param filePath SQL文件路径
	 * @throws IOException
	 * @throws Exception
	 */
	public boolean execute(String filePath, List<String> filterList, boolean isExclude) throws Exception {
		return execute(new File(filePath), filterList, isExclude);
	}

	/**
	 * 加载并执行SQL文件
	 * @param file SQL文件
	 * @throws Exception
	 */
	public boolean execute(File file) throws Exception {
		List<String> commandList = getCommandList(file);
		return execute(commandList);
	}

	/**
	 * 加载并执行SQL文件
	 * @param file SQL文件
	 * @throws IOException
	 * @throws Exception
	 */
	public boolean execute(File file, List<String> filterList, boolean isExclude) throws Exception {
		List<String> commandList = getCommandList(file);
		return execute(commandList, filterList, isExclude);
	}

	/**
	 * 加载并执行SQL文件
	 * @param file 输入流
	 * @throws IOException
	 * @throws Exception
	 */
	public boolean execute(InputStream inputStream) throws Exception {
		List<String> commandList = getCommandList(inputStream);
		return execute(commandList);
	}

	/**
	 * 加载并执行SQL文件
	 * @param inputStream 输入流
	 * @param filterList
	 * @param isExclude
	 * @throws IOException
	 * @throws Exception
	 */
	public boolean execute(InputStream inputStream, List<String> filterList, boolean isExclude) throws Exception {
		List<String> commandList = getCommandList(inputStream);
		return execute(commandList, filterList, isExclude);
	}

	/**
	 * 执行SQL命令
	 * @param commandList SQL命令列表
	 * @throws Exception
	 */
	private boolean execute(List<String> commandList) throws Exception {
		Connection conn = getConnection();
		if (conn == null || conn.isClosed()) {
			throw new SQLException("Connection is null or closed!");
		}
		Statement statement = createStatement(conn);
		try {
			goodSql = 0;
			totalSql = commandList.size();
			int i = 0;
			for (String sqlCommand : commandList) {
				execSQL(sqlCommand, conn, statement);
				i++;
				if (i % 200 == 0 && !autoCommit) { // 如果一个文件提交一次事务，可能提交失败
					conn.commit();
				}
			}
			if (i % 200 != 0 && !autoCommit) {
				conn.commit();
			}
			return goodSql == totalSql;
		} catch (SQLException e) {
			throw e;
		} finally {
			CloseUtil.close(statement);
			CloseUtil.close(conn);
			log(goodSql + " of " + totalSql + " SQL statements executed successfully", MsgLevel.INFO);
		}
	}

	/**
	 * 执行SQL命令
	 * @param commandList SQL命令列表
	 * @param filterList
	 * @param isExclude
	 * @return
	 * @throws Exception
	 */
	private boolean execute(List<String> commandList, List<String> filterList, boolean isExclude) throws Exception {
		Connection conn = getConnection();
		if (conn == null || conn.isClosed()) {
			throw new SQLException("Connection is null or closed!");
		}
		Statement statement = createStatement(conn);
		long startTime = System.currentTimeMillis();
		try {
			goodSql = 0;
			totalSql = commandList.size();
			int i = 0;
			boolean ignore = false;
			for (String sqlCommand : commandList) {
				if (isExclude) {
					ignore = isIgnoreLine(filterList, sqlCommand);
				} else {
					ignore = !isIgnoreLine(filterList, sqlCommand);
				}
				if (ignore) {
					continue;
				}
				execSQL(sqlCommand, conn, statement);
				i++;
				if (i % 200 == 0 && !autoCommit) { // 如果一个文件提交一次事务，可能提交失败
					conn.commit();
				}
			}
			if (i % 200 != 0 && !autoCommit) {
				conn.commit();
			}
			return goodSql == totalSql;
		} catch (SQLException e) {
			throw e;
		} finally {
			CloseUtil.close(statement);
			CloseUtil.close(conn);
			long endTime = System.currentTimeMillis();
			log(goodSql + " of " + totalSql + " SQL executed successfully. use: " + (endTime - startTime), MsgLevel.INFO);
		}
	}

	/**
	 * Exec the sql statement.
	 * @param sql the SQL statement to execute
	 * @param statement the SQL statement
	 * @throws Exception on SQL problems
	 */
	protected void execSQL(String sql, Connection conn, Statement statement) throws Exception {
		if ("".equals(sql.trim())) {
			return;
		}
		try {
			if (printSQL) {
				log("Execute SQL: " + sql, MsgLevel.INFO);
			}
			int updateCount = 0;
			int updateCountTotal = 0;

			sql = processSql(sql);
			long startTime = System.currentTimeMillis();
			boolean result = statement.execute(sql);
			updateCount = statement.getUpdateCount();
			do {
				if (updateCount != -1) {
					updateCountTotal += updateCount;
				}
				if (result) {
					ResultSet resultSet = null;
					try {
						resultSet = statement.getResultSet();
						printWarnings(resultSet.getWarnings(), false);
						resultSet.clearWarnings();
						if (printResult) {
							printResults(resultSet);
						}
					} finally {
						CloseUtil.close(resultSet);
					}
				}
				result = statement.getMoreResults();
				updateCount = statement.getUpdateCount();
			} while (result || updateCount != -1);

			printWarnings(statement.getWarnings(), false);
			statement.clearWarnings();

			if (printRowAffected) {
				log(updateCountTotal + " rows affected. use: " + (System.currentTimeMillis() - startTime), MsgLevel.INFO);
			}

			SQLWarning warning = conn.getWarnings();
			printWarnings(warning, true);
			conn.clearWarnings();
			goodSql++;
		} catch (Exception e) {
			log("Failed to execute: " + sql, MsgLevel.ERROR);
			log("error: " + e.getMessage(), MsgLevel.ERROR);
			if (printDetailMessage) {
				log("error: " + getDetailMessage(e), MsgLevel.ERROR);
			} else {
				logger.debug(e.getMessage(), e);
			}

			if (onError.equals(OnError.ABORT)) {
				closeQuietly(conn);
				throw e;
			} else if (onError.equals(OnError.CONTINUE)) {
				// continue
			} else if (!AssertUtil.isEmpty(ignoreErrorCodeList)) {
				boolean isIgnore = false;
				for (int errorCode : ignoreErrorCodeList) {
					if (e instanceof SQLException) {
						if (((SQLException) e).getErrorCode() == errorCode) {
							isIgnore = true;
							break;
						}
					}
				}
				if (!isIgnore) {
					throw e;
				}
			}
		}

	}

	/**
	 * 对SQL进行处理
	 * @param sql
	 * @return
	 */
	protected String processSql(String sql) {
		return sql;
	}

	/**
	 * 获取文件内容
	 * @param file
	 * @return
	 * @throws IOException
	 */
	private List<String> getCommandList(File file) throws IOException {
		if (file == null || !file.isFile()) {
			throw new IOException("Source file " + file + " is not a file!");
		}
		if (fileParser == null) {
			initFileParser();
		}
		List<String> commandList = fileParser.parseToList(file, encoding);
		return commandList;
	}

	/**
	 * 获取文件内容
	 * @param file
	 * @return
	 * @throws IOException
	 */
	private List<String> getCommandList(InputStream inputStream) throws IOException {
		if (fileParser == null) {
			initFileParser();
		}
		List<String> commandList = fileParser.parseToList(inputStream, encoding);
		return commandList;
	}

	/**
	 * 忽略行
	 * @param content
	 * @param filterList 过滤列表
	 * @return
	 */
	protected boolean isIgnoreLine(List<String> filterList, String content) {
		boolean ignore = false;
		if (AssertUtil.isNotEmpty(filterList)) {
			for (String regex : filterList) {
				if (isMatcher(regex, content)) {
					ignore = true;
					break;
				}
			}
		}
		return ignore;
	}

	/**
	 * 判断内容是否匹配正则表达式
	 * @param content
	 * @return
	 */
	protected boolean isMatcher(String regex, String content) {
		if (ignoreCase) {
			return Pattern.compile(regex, Pattern.CASE_INSENSITIVE).matcher(content).find();
		} else {
			return Pattern.compile(regex).matcher(content).find();
		}
	}

	private String getDetailMessage(Throwable e) {
		StringWriter stringWriter = new StringWriter();
		PrintWriter writer = new PrintWriter(stringWriter);
		e.printStackTrace(writer);
		return stringWriter.toString();
	}

	protected void log(String message, MsgLevel msgLevel) {
		switch (msgLevel) {
		case DEBUG:
			logger.debug(message);
			break;
		case INFO:
			logger.info(message);
			break;
		case WARN:
			logger.warn(message);
			break;
		default:
			logger.error(message);
			break;
		}

		output(message);
	}

	protected void output(String message) {
		if (printWriter != null) {
			printWriter.println(message);
			printWriter.flush();
		}
	}

	public void setOutput(String filePath) throws FileNotFoundException {
		setOutput(new File(filePath));
	}

	public void setOutput(File file) throws FileNotFoundException {
		if (!file.exists()) {
			file.getParentFile().mkdirs();
		}
		printWriter = new PrintWriter(file);
	}

	public void setOutput(PrintWriter printWriter) {
		this.printWriter = printWriter;
	}

	/**
	 * printResult any results in the result set.
	 * @param rs the resultset to printResult information about
	 * @param out the place to printResult results
	 * @throws Exception on SQL problems.
	 * @since Ant 1.6.3
	 */
	protected void printResults(ResultSet rs) throws SQLException {
		if (rs == null) {
			return;
		}
		String lineSeparator = System.getProperty("line.separator");
		StringBuffer outBuffer = new StringBuffer();
		log("Processing new result set.", MsgLevel.DEBUG);
		try {
			ResultSetMetaData md = rs.getMetaData();
			int columnCount = md.getColumnCount();
			if (columnCount > 0) {
				if (printHeader) {
					outBuffer.append(md.getColumnName(1));
					for (int col = 2; col <= columnCount; col++) {
						outBuffer.append(csvColumnSep);
						outBuffer.append(maybeQuote(md.getColumnName(col)));
					}
					outBuffer.append(lineSeparator);
				}
				while (rs.next()) {
					printValue(rs, 1, outBuffer);
					for (int col = 2; col <= columnCount; col++) {
						outBuffer.append(csvColumnSep);
						printValue(rs, col, outBuffer);
					}
					outBuffer.append(lineSeparator);
					printWarnings(rs.getWarnings(), false);
				}
			}
		} catch (SQLException e) {
			log(e.getMessage(), MsgLevel.ERROR);
			if (printDetailMessage) {
				log(getDetailMessage(e), MsgLevel.ERROR);
			} else {
				logger.error("", e);
			}
		}
		outBuffer.append(lineSeparator);
		log(outBuffer.toString(), MsgLevel.DEBUG);
	}

	private void printValue(ResultSet rs, int col, StringBuffer outBuffer) throws SQLException {
		if (printBlob && rs.getMetaData().getColumnType(col) == Types.BLOB) {
			Blob blob = rs.getBlob(col);
			if (blob != null) {
				outBuffer.append(blob);
			}
		} else {
			outBuffer.append(maybeQuote(rs.getString(col)));
		}
	}

	private String maybeQuote(String s) {
		if (csvQuoteChar == null || s == null || (s.indexOf(csvColumnSep) == -1 && s.indexOf(csvQuoteChar) == -1)) {
			return s;
		}
		StringBuffer sb = new StringBuffer(csvQuoteChar);
		int len = s.length();
		char q = csvQuoteChar.charAt(0);
		for (int i = 0; i < len; i++) {
			char c = s.charAt(i);
			if (c == q) {
				sb.append(q);
			}
			sb.append(c);
		}
		return sb.append(csvQuoteChar).toString();
	}

	/**
	 * 当执行SQL失败，并且onError为abort时，事务回滚
	 */
	private void closeQuietly(Connection conn) {
		if (conn != null && OnError.ABORT.equals(onError)) {
			try {
				if (!conn.getAutoCommit()) {
					conn.rollback();
				}
			} catch (SQLException e) {
				// ignore
			}
		}
	}

	/**
	 * Caches the connection returned by the base class's getConnection method.
	 * <p>
	 * Subclasses that need to provide a different connection than the base class
	 * would, should override this method but keep in mind that this class expects
	 * to get the same connection instance on consecutive calls.
	 * </p>
	 * <p>
	 * returns null if the connection does not connect to the expected RDBMS.
	 * </p>
	 * @throws Exception
	 */
	protected Connection getConnection() throws SQLException {
		Connection connection = dataSource.getConnection();
		if (autoCommit != connection.getAutoCommit()) {
			connection.setAutoCommit(autoCommit);
		}
		return connection;
	}

	public DataSource getDataSource() {
		return dataSource;
	}

	public void setDataSource(DataSource dataSource) {
		this.dataSource = dataSource;
	}

	public boolean getAutoCommit() {
		return autoCommit;
	}

	public void setAutoCommit(boolean autoCommit) {
		this.autoCommit = autoCommit;
	}

	/**
	 * Creates and configures a Statement instance which is then cached for
	 * subsequent calls.
	 * <p>
	 * Subclasses that want to provide different Statement instances, should
	 * override this method but keep in mind that this class expects to get the same
	 * connection instance on consecutive calls.
	 * </p>
	 * @param conn Connection
	 */
	protected Statement createStatement(Connection conn) throws SQLException {
		Statement statement = conn.createStatement();
		statement.setEscapeProcessing(escapeProcessing);
		return statement;
	}

	/**
	 * The action a task should perform on an error, one of "CONTINUE", "ABORT" and
	 * "IGNORE_LIST"
	 */
	public static enum OnError {
		CONTINUE, ABORT, IGNORE_LIST;
	}

	public enum MsgLevel {
		DEBUG, ERROR, INFO, TRACE, WARN;

	}

	private void printWarnings(SQLWarning warning, boolean force) throws SQLException {
		if (printWarning || force) {
			while (warning != null) {
				log(warning + " sql warning", MsgLevel.WARN);
				warning = warning.getNextWarning();
			}
		}
		if (treatWarningsAsErrors && warning != null) {
			throw warning;
		}
	}

	/**
	 * @return the encoding
	 */
	public String getEncoding() {
		return encoding;
	}

	/**
	 * @param encoding the encoding to set
	 */
	public void setEncoding(String encoding) {
		this.encoding = encoding;
	}

	public List<String> getIgnoreLineList() {
		return ignoreLineList;
	}

	public void setIgnoreLineList(List<String> ignoreLineList) {
		this.ignoreLineList = ignoreLineList;
	}

	/**
	 * @return the ignoreErrorCodeList
	 */
	public List<Integer> getIgnoreErrorCodeList() {
		return ignoreErrorCodeList;
	}

	/**
	 * @param ignoreErrorCodeList the ignoreErrorCodeList to set
	 */
	public void setIgnoreErrorCodeList(List<Integer> ignoreErrorCodeList) {
		this.ignoreErrorCodeList = ignoreErrorCodeList;
	}

	/**
	 * @return the totalSql
	 */
	public int getTotalSql() {
		return totalSql;
	}

	/**
	 * @param totalSql the totalSql to set
	 */
	public void setTotalSql(int totalSql) {
		this.totalSql = totalSql;
	}

	/**
	 * @return the goodSql
	 */
	public int getGoodSql() {
		return goodSql;
	}

	/**
	 * @param goodSql the goodSql to set
	 */
	public void setGoodSql(int goodSql) {
		this.goodSql = goodSql;
	}

	/**
	 * @return the printResult
	 */
	public boolean isPrintResult() {
		return printResult;
	}

	/**
	 * @param printResult the printResult to set
	 */
	public void setPrintResult(boolean printResult) {
		this.printResult = printResult;
	}

	/**
	 * @return the printSQL
	 */
	public boolean isPrintSQL() {
		return printSQL;
	}

	/**
	 * @param printSQL the printSQL to set
	 */
	public void setPrintSQL(boolean printSQL) {
		this.printSQL = printSQL;
	}

	/**
	 * @return the printDetailMessage
	 */
	public boolean isPrintDetailMessage() {
		return printDetailMessage;
	}

	/**
	 * @param printDetailMessage the printDetailMessage to set
	 */
	public void setPrintDetailMessage(boolean printDetailMessage) {
		this.printDetailMessage = printDetailMessage;
	}

	/**
	 * @return the printHeader
	 */
	public boolean isPrintHeader() {
		return printHeader;
	}

	/**
	 * @param printHeader the printHeader to set
	 */
	public void setPrintHeader(boolean showheaders) {
		this.printHeader = showheaders;
	}

	/**
	 * @return the printRowAffected
	 */
	public boolean isPrintRowAffected() {
		return printRowAffected;
	}

	/**
	 * @param printRowAffected the printRowAffected to set
	 */
	public void setPrintRowAffected(boolean showtrailers) {
		this.printRowAffected = showtrailers;
	}

	/**
	 * @return the onError
	 */
	public OnError getOnError() {
		return onError;
	}

	/**
	 * @param onError the onError to set
	 */
	public void setOnError(OnError onError) {
		this.onError = onError;
	}

	/**
	 * @return the escapeProcessing
	 */
	public boolean isEscapeProcessing() {
		return escapeProcessing;
	}

	/**
	 * @param escapeProcessing the escapeProcessing to set
	 */
	public void setEscapeProcessing(boolean escapeProcessing) {
		this.escapeProcessing = escapeProcessing;
	}

	/**
	 * @return the printBlob
	 */
	public boolean isPrintBlob() {
		return printBlob;
	}

	/**
	 * @param printBlob the printBlob to set
	 */
	public void setPrintBlob(boolean rawBlobs) {
		this.printBlob = rawBlobs;
	}

	/**
	 * @return the printWarning
	 */
	public boolean isPrintWarning() {
		return printWarning;
	}

	/**
	 * @param printWarning the printWarning to set
	 */
	public void setPrintWarning(boolean showWarnings) {
		this.printWarning = showWarnings;
	}

	/**
	 * @return the csvColumnSep
	 */
	public String getCsvColumnSep() {
		return csvColumnSep;
	}

	/**
	 * @param csvColumnSep the csvColumnSep to set
	 */
	public void setCsvColumnSep(String csvColumnSep) {
		this.csvColumnSep = csvColumnSep;
	}

	/**
	 * @return the csvQuoteChar
	 */
	public String getCsvQuoteChar() {
		return csvQuoteChar;
	}

	/**
	 * @param csvQuoteChar the csvQuoteChar to set
	 */
	public void setCsvQuoteChar(String csvQuoteChar) {
		this.csvQuoteChar = csvQuoteChar;
	}

	/**
	 * @return the treatWarningsAsErrors
	 */
	public boolean isTreatWarningsAsErrors() {
		return treatWarningsAsErrors;
	}

	/**
	 * @param treatWarningsAsErrors the treatWarningsAsErrors to set
	 */
	public void setTreatWarningsAsErrors(boolean treatWarningsAsErrors) {
		this.treatWarningsAsErrors = treatWarningsAsErrors;
	}

	public boolean isIgnoreCase() {
		return ignoreCase;
	}

	public void setIgnoreCase(boolean ignoreCase) {
		this.ignoreCase = ignoreCase;
	}

	public boolean isKeepFormat() {
		return keepFormat;
	}

	public void setKeepFormat(boolean keepFormat) {
		this.keepFormat = keepFormat;
	}

	public String getDelimiter() {
		return delimiter;
	}

	public void setDelimiter(String delimiter) {
		this.delimiter = delimiter;
	}

	public PrintWriter getPrintWriter() {
		return printWriter;
	}

	public void setPrintWriter(PrintWriter printWriter) {
		this.printWriter = printWriter;
	}

}
