package cn.elwy.common.jdbc;

import java.lang.reflect.InvocationHandler;
import java.lang.reflect.Method;
import java.lang.reflect.Proxy;
import java.sql.Connection;
import java.sql.DatabaseMetaData;
import java.sql.SQLException;
import java.util.LinkedList;
import java.util.Map;

import cn.elwy.common.exception.ConnException;
import cn.elwy.common.exception.RunException;
import cn.elwy.common.i18n.Msg;

/**
 * 根据配置创建数据库连接池
 * @author huangsq
 * @version 1.0, 2018-02-19
 */
public class PoolDataSource extends MyDataSource {

	// 存储数据库连接的集合,采用先进先出的LinkedList集合
	protected final LinkedList<ConnectionProxy> connPool = new LinkedList<ConnectionProxy>();
	protected int initPoolSize = 2; // 连接池的初始大小
	protected int minPoolSize = 1; // 连接池最小的大小
	protected int maxPoolSize = 10; // 连接池最大的大小
	protected int acquireIncrement = 1; // 连接池自动增加的大小
	protected boolean refreshFlag; // 是否刷新连接池中的连接

	protected int currentCount = 0; // 当前连接数
	private boolean isInitPool; // 是否初始化连接池
	private RunException connError; // 连接异常

	public PoolDataSource(DBInfo dbInfo) {
		super(dbInfo);
		setInitPoolSize(dbInfo.getInitPoolSize());
		setMinPoolSize(dbInfo.getMinPoolSize());
		setMaxPoolSize(dbInfo.getMaxPoolSize());
		setAcquireIncrement(dbInfo.getAcquireIncrement());
	}

	public PoolDataSource(Map<String, Object> dbInfo) {
		super(dbInfo);
		setInitPoolSize(getIntValue((Integer) dbInfo.get("initPoolSize")));
		setMinPoolSize(getIntValue((Integer) dbInfo.get("minPoolSize")));
		setMaxPoolSize(getIntValue((Integer) dbInfo.get("maxPoolSize")));
		setAcquireIncrement(getIntValue((Integer) dbInfo.get("acquireIncrement")));
	}

	public PoolDataSource(String driver, String url, String user, String password) {
		super(driver, url, user, password);
	}

	/** 从数据库连接池中获取一个数据库连接 */
	public synchronized Connection getConnection() {
		Connection conn = getFreeConn(); // 获得一个空闲的数据库连接
		// 如果目前没有可以使用的连接，即所有的连接都在使用中
		for (int i = 0; conn == null && i < 3; i++) {
			wait(1000); // 等一秒再试
			// 重新获得可用的连接，如果getFreeConn()返回的为 null则表明创建一批连接后也不可获得可用连接
			conn = getFreeConn();
			if (conn != null) {
				break;
			}
		}
		if (conn == null) { // 如果conn为null则表明获取数据库连接失败
			if (refreshFlag) {
				refreshConn(); // 刷新连接池中的连接
				conn = getFreeConn(); // 获得一个空闲的数据库连接
				if (conn == null) {
					logger.error(Msg.E_CONN_GET);
					throw connError;
				}
			} else {
				logger.error(Msg.E_CONN_GET);
				if (connError == null) {
					connError = new RunException(Msg.E_CONN_BUSY);
				}
				throw connError;
			}
		}
		currentCount++;
		String message = Msg.format(Msg.E_CONN_CURRENT_COUNT, currentCount, connPool.size());
		logger.debug(message);
		return conn; // 返回一个可用的连接
	}

	/** 初始化数据源 */
	protected synchronized void initDataSource() {
		if (isInitPool) {
			return;
		}
		super.initDataSource();
		isInitPool = true;
		int initSize = connPool.size();
		try {
			for (int i = initSize; i < initPoolSize && i < maxPoolSize; i++) {
				Connection connection = createConn();
				ConnectionProxy connectionProxy = new ConnectionProxy(this);
				connectionProxy.setConn(connection);
				connPool.addLast(connectionProxy);
				if (i == initSize) {
					try {
						DatabaseMetaData metaData = connection.getMetaData();
						// driverMaxConn为返回的一个整数，表示此数据库允许客户连接的数目
						int driverMaxConn = metaData.getMaxConnections();
						// 数据库返回的 driverMaxConn若为0，表示此数据库没有最大连接限制，或数据库的最大连接限制不知道
						// 如果连接池中设置的最大连接数量大于数据库允许的连接数目，则置连接池的最大连接数目为数据库允许的最大数目
						if (driverMaxConn > 0 && this.maxPoolSize > driverMaxConn) {
							this.maxPoolSize = driverMaxConn;
						}
					} catch (SQLException e) {
						logger.warn(Msg.E_CONN_GET_MAX_POOL, e);
					}
				}
			}
			logger.debug(Msg.format(Msg.E_CONN_CREAT_POOL, connPool.size()));
		} catch (ConnException e) {
			if (connPool.size() == 0) {
				throw e;
			}
		}
	}

	/** 从连接池中获得一个可用的数据库连接，如果当前没有可用的数据库连接，则创建一批数据库连接，如果所有连接都不可用，则返回null */
	private synchronized Connection getFreeConn() {
		Connection conn = findFreeConn();
		if (conn == null) {
			if (!isInitPool) {
				initDataSource();
				conn = findFreeConn();
				if (conn != null) {
					return conn;
				}
			}
			// 如果目前连接池中没有可用的连接，根据stepCount设置的值创建几个数据库连接，并放入连接池中。
			// 如果创建后，所有的连接仍都在使用中，则再次创建连接，直到不能创建连接为止，如果依然没有可用的连接，则返回null
			exitFor: for (int i = connPool.size(); i < maxPoolSize; i = connPool.size()) {
				for (int j = i; j < i + acquireIncrement; j++) {
					if (j >= maxPoolSize) {
						break exitFor;
					}
					try {
						Connection connection = createConn();
						ConnectionProxy connectionProxy = new ConnectionProxy(this);
						connectionProxy.setConn(connection);
						connPool.addFirst(connectionProxy);
					} catch (ConnException e) {
						connError = e;
						break exitFor;
					}
				}
				conn = findFreeConn();
				if (conn != null) {
					break exitFor;
				}
			}
		}
		return conn;
	}

	/** 从连接池中查找一个可用的数据库连接， 如果没有可用的连接，返回 null */
	private synchronized Connection findFreeConn() {
		Connection conn = null;
		for (int i = 0; i < connPool.size(); i++) {
			ConnectionProxy connProxy = connPool.removeFirst();
			connPool.addLast(connProxy);
			if (!connProxy.isBusy()) {
				// 如果此对象不忙，则获得它的数据库连接并把它设为忙
				conn = connProxy.getConnProxy();
				connProxy.setBusy(true);
				// 测试此连接是否可用
				if (!isConnected(conn)) {
					// 如果此连接不可再用了，则创建一个新的连接，
					// 并替换此不可用的连接对象，如果创建失败，返回 null
					try {
						closeConn(conn);
						Connection connection = createConn();
						connProxy.setConn(connection);
						conn = connProxy.getConnProxy();
					} catch (ConnException e) {
						connError = e;
						connProxy.setBusy(false);
					}
				}
				break; // 己经找到一个可用的连接，退出
			}
		}
		return conn; // 返回找到到的可用连接
	}

	/** 测试数据库连接是否可用 */
	public boolean ping() {
		try {
			return getFreeConn() != null ? true : false;
		} catch (Exception e) {
			return false;
		}
	}

	/** 释放一个数据库连接到连接池中，并把此连接置为空闲状态 */
	public synchronized void freeConnection(Connection conn) {
		if (connPool == null) {
			logger.warn(Msg.E_CONN_NOT_EXIST);
			return;
		}
		for (int i = connPool.size() - 1; i >= 0; i--) {
			ConnectionProxy connProxy = connPool.get(i);
			if (conn == connProxy.getConnProxy()) {
				if (connProxy.isBusy()) {
					connProxy.setBusy(false);
					currentCount--;
				}
				break;
			}
		}
	}

	/** 刷新连接池中的第一个连接对象,尽可能不用该方法 */
	@Deprecated
	public synchronized void refreshConn() {
		if (connPool == null) {
			logger.warn(Msg.E_CONN_NOT_EXIST);
			return;
		}
		for (ConnectionProxy connProxy : connPool) {
			Connection conn = connProxy.getConnProxy();
			if (isConnected(conn)) {
				if (connProxy.isBusy()) {
					wait(5000); // 如果对象忙则等5秒，5秒后直接刷新
					connProxy.setBusy(false);
					currentCount--;
				}
				break;
			}
			try { // 关闭此连接，用一个新的连接代替它
				closeConn(connProxy.getConn());
				Connection connection = createConn();
				connProxy.setConn(connection);
				connProxy.setBusy(false);
				currentCount--;
			} catch (ConnException e) {
				logger.warn(Msg.E_CONN_REFRESH);
				connError = e;
			}
			break;
		}
	}

	/** 使程序等待给定的毫秒数 */
	private void wait(int mSeconds) {
		try {
			Thread.sleep(mSeconds);
		} catch (InterruptedException e) {
			logger.warn(Msg.E_THREAD_END, e);
		}
	}

	/** 关闭连接池中所有的连接，并清空连接池。 */
	public void close() {
		isInitPool = false;
		if (connPool == null) {
			logger.warn(Msg.E_CONN_NOT_EXIST);
			return;
		}
		try {
			for (ConnectionProxy connProxy : connPool) {
				if (connProxy.isBusy()) {
					wait(1000); // 如果对象忙则等1秒，1秒后直接关闭
				}
				closeConn(connProxy.getConn()); // 关闭此连接
				connProxy = null;
			}
			currentCount = 0;
			connPool.clear(); // 置连接池为空
			logger.info(Msg.M_CONN_CLEAR);
		} catch (Exception e) {
			logger.warn(Msg.E_CONN_CLOSE, e);
		}
	}

	/** 关闭一个数据库连接 */
	public void close(Connection conn) {
		if (connPool.size() <= minPoolSize) {
			freeConnection(conn);
		} else {
			for (int i = connPool.size() - 1; i >= 0; i--) {
				ConnectionProxy connProxy = connPool.get(i);
				if (conn == connProxy.getConn()) {
					closeConn(conn); // 关闭此连接
					connPool.remove(connProxy); // 从连接池向量中删除它
					connProxy = null; // 清空对象
					currentCount--; // 用代理类就不用执行此处
					break;
				}
			}
		}
		String message = Msg.format(Msg.E_CONN_CURRENT_COUNT, currentCount, connPool.size());
		logger.debug(message);
	}

	public int getInitPoolSize() {
		return initPoolSize;
	}

	public void setInitPoolSize(int initPoolSize) {
		if (initPoolSize < 2) {
			this.initPoolSize = 2;
		}
		this.initPoolSize = initPoolSize;
	}

	public int getMinPoolSize() {
		return minPoolSize;
	}

	public void setMinPoolSize(int minPoolSize) {
		if (minPoolSize < 1) {
			this.minPoolSize = 1;
		}
		this.minPoolSize = minPoolSize;
	}

	public int getMaxPoolSize() {
		return maxPoolSize;
	}

	public void setMaxPoolSize(int maxPoolSize) {
		if (maxPoolSize < 10) {
			this.maxPoolSize = 10;
		}
		this.maxPoolSize = maxPoolSize;
	}

	public int getAcquireIncrement() {
		return acquireIncrement;
	}

	public void setAcquireIncrement(int acquireIncrement) {
		if (acquireIncrement < 1) {
			this.acquireIncrement = 1;
		}
		this.acquireIncrement = acquireIncrement;
	}

	public boolean isRefreshFlag() {
		return refreshFlag;
	}

	public void setRefreshFlag(boolean refreshFlag) {
		this.refreshFlag = refreshFlag;
	}

}

/**
 * 用于保存连接池中连接对象的类，此类中有两个成员； 一个是数据库的连接，另一个是指示此连接是否正在使用的标志
 */
class ConnectionProxy implements InvocationHandler {

	private Connection conn;// 数据库连接
	private Connection connProxy;// 数据库连接
	private MyDataSource datasource;
	boolean busy = false; // 此连接是否正在使用的标志，默认没有正在使用

	public ConnectionProxy(MyDataSource datasource) {
		this.datasource = datasource;
	}

	public Connection getConn() {
		return conn;
	}

	public void setConn(Connection conn) {
		this.conn = conn;
		// Proxy.newProxyInstance()方法表示：产生一个新的类，该类继承于第二个参数Connnection，
		// 但该类的方法的现实是由第三个参数this负责实现
		connProxy = (Connection) Proxy.newProxyInstance(conn.getClass().getClassLoader(), new Class[] { Connection.class },
				this);
	}

	public Connection getConnProxy() {
		return connProxy;
	}

	/** 获得对象连接是否忙 */
	public boolean isBusy() {
		return busy;
	}

	/** 设置对象的连接正在忙 */
	public void setBusy(boolean busy) {
		this.busy = busy;
	}

	public Object invoke(Object proxy, Method method, Object[] args) throws Throwable {
		if ("close".equals(method.getName())) {
			datasource.close(conn);
		} else {
			return method.invoke(conn, args);
		}
		return null;
	}

}
