package cn.elwy.common.jdbc;

import java.io.PrintWriter;
import java.sql.Connection;
import java.sql.DriverManager;
import java.sql.SQLException;
import java.sql.SQLFeatureNotSupportedException;
import java.sql.Statement;
import java.util.HashSet;
import java.util.Map;
import java.util.Properties;
import java.util.Set;

import javax.sql.DataSource;

import cn.elwy.common.exception.ConnException;
import cn.elwy.common.i18n.Msg;
import cn.elwy.common.log.Logger;
import cn.elwy.common.log.LoggerFactory;

/**
 * 数据源抽象类
 * @author huangsq
 * @version 1.0, 2018-02-19
 */
public abstract class MyDataSource implements DataSource {

	protected Logger logger = LoggerFactory.getLogger(this.getClass());

	public static final Set<String> driverList = new HashSet<String>();
	protected boolean isInit; // 是否已经初始化
	protected String dsName; // 数据源名称
	protected String driver; // 数据库驱动
	protected String url; // 数据库URL
	protected String user; // 数据库用户名
	protected String password; // 数据库密码
	protected int loginTimeout; // 登录超时时间
	protected String testSQL; // 测试连接是否可用的测试表名，默认没有测试表
	protected String testTable; // 测试连接是否可用的测试表名，默认没有测试表
	protected Properties connProperties; // 数据库连接属性

	public MyDataSource() {
	}

	public MyDataSource(DBInfo dbInfo) {
		this.dsName = dbInfo.getDsName();
		this.driver = dbInfo.getDriver();
		this.url = dbInfo.getUrl();
		if (isEmpty(driver) || isEmpty(url)) {
			dbInfo.initParameter();
			this.driver = dbInfo.getDriver();
			this.url = dbInfo.getUrl();
		}
		this.user = dbInfo.getUser();
		this.password = dbInfo.getPassword();
		this.loginTimeout = dbInfo.getLoginTimeout();
		this.testSQL = dbInfo.getTestQuery();
		this.testTable = dbInfo.getTestTable();
	}

	public MyDataSource(Map<String, Object> dbInfo) {
		this.dsName = (String) dbInfo.get("dsName");
		this.driver = (String) dbInfo.get("driver");
		this.url = (String) dbInfo.get("url");
		this.user = (String) dbInfo.get("user");
		this.password = (String) dbInfo.get("password");
		this.loginTimeout = getIntValue((Integer) dbInfo.get("loginTimeout"));
		this.testSQL = (String) dbInfo.get("testSQL");
		this.testTable = (String) dbInfo.get("testTable");
	}

	public MyDataSource(String driver, String url, String user, String password) {
		this.driver = driver;
		this.url = url;
		this.user = user;
		this.password = password;
	}

	/** 创建一个新的数据库连接 */
	protected Connection createConn() {
		initDataSource();
		if (loginTimeout > 0) {
			DriverManager.setLoginTimeout(loginTimeout);
		}
		try {
			Connection connection = DriverManager.getConnection(url, connProperties);
			return connection;
		} catch (Exception e) {
			throw new ConnException(Msg.E_CONN_CREATE, e);
		}
	}

	/**
	 * 初始化数据源
	 */
	protected synchronized void initDataSource() {
		if (isInit) {
			return;
		}
		if (connProperties == null) {
			this.connProperties = new Properties();
		}
		if (!isEmpty(user)) {
			connProperties.put("user", user);
			connProperties.put("password", password);
		}
		registerDriver(driver);
		isInit = true;
	}

	/** 注册驱动 */
	protected synchronized void registerDriver(String dbDriver) {
		try {
			if (!driverList.contains(dbDriver)) {
				Class.forName(dbDriver);
				driverList.add(dbDriver);
			}
		} catch (ClassNotFoundException e) {
			throw new ConnException(Msg.E_CONN_DRIVER, e);
		}
	}

	/** 判断数据库连接是否可用 */
	public boolean isConnected(Connection conn) {
		try {
			if (conn == null || conn.isClosed()) {
				return false;
			}
		} catch (SQLException e) {
			logger.warn(Msg.E_CONN_DISABLE, e);
			return false;
		}
		Statement stmt = null;
		try {
			if (isNotEmpty(testSQL)) {
				stmt = conn.createStatement();
				stmt.execute(testSQL);
			} else if (isNotEmpty(testTable)) {
				stmt = conn.createStatement();
				stmt.execute("select count(*) from " + testTable);
			} else {
				return conn.isValid(1000);
			}
			return true;
		} catch (Throwable e) {
			logger.warn(Msg.E_CONN_DISABLE, e);
			return false;
		} finally {
			close(stmt);
		}
	}

	/** 测试数据库连接是否可用 */
	public abstract boolean ping();

	/** 关闭conn */
	public abstract void close();

	/** 关闭conn */
	public void close(Connection conn) {
		closeConn(conn);
	}

	/** 关闭conn */
	public void closeConn(Connection conn) {
		if (conn != null) {
			try {
				if (!conn.isClosed()) {
					conn.close();
				}
			} catch (SQLException e) {
				logger.warn(e.getMessage());
			} finally {
				conn = null;
			}
		}
	}

	/**
	 * @return the dsName
	 */
	public String getDsName() {
		return dsName;
	}

	/**
	 * @param dsName the dsName to set
	 */
	public void setDsName(String dsName) {
		this.dsName = dsName;
	}

	/**
	 * @return the driver
	 */
	public String getDriver() {
		return driver;
	}

	/**
	 * @param driver the driver to set
	 */
	public void setDriver(String driver) {
		this.driver = driver;
	}

	/**
	 * @return the url
	 */
	public String getUrl() {
		return url;
	}

	/**
	 * @param url the url to set
	 */
	public void setUrl(String url) {
		this.url = url;
	}

	/**
	 * @return the user
	 */
	public String getUser() {
		return user;
	}

	/**
	 * @param user the user to set
	 */
	public void setUser(String user) {
		this.user = user;
	}

	/**
	 * @return the password
	 */
	public String getPassword() {
		return password;
	}

	/**
	 * @param password the password to set
	 */
	public void setPassword(String password) {
		this.password = password;
	}

	/**
	 * @return the loginTimeout
	 */
	public int getLoginTimeout() {
		return loginTimeout;
	}

	/**
	 * @param loginTimeout the loginTimeout to set
	 */
	public void setLoginTimeout(int loginTimeout) {
		this.loginTimeout = loginTimeout;
	}

	/**
	 * @return the testSQL
	 */
	public String getTestSQL() {
		return testSQL;
	}

	/**
	 * @param testSQL the testSQL to set
	 */
	public void setTestSQL(String testSQL) {
		this.testSQL = testSQL;
	}

	/**
	 * @return the testTable
	 */
	public String getTestTable() {
		return testTable;
	}

	/**
	 * @param testTable the testTable to set
	 */
	public void setTestTable(String testTable) {
		this.testTable = testTable;
	}

	/**
	 * @return the connProperties
	 */
	public Properties getConnProperties() {
		return connProperties;
	}

	/**
	 * @param connProperties the connProperties to set
	 */
	public void setConnProperties(Properties connProperties) {
		this.connProperties = connProperties;
	}

	public PrintWriter getLogWriter() {
		throw new UnsupportedOperationException("getLogWriter");
	}

	public void setLogWriter(PrintWriter pw) throws SQLException {
		throw new UnsupportedOperationException("setLogWriter");
	}

	@SuppressWarnings("unchecked")
	@Override
	public <T> T unwrap(Class<T> iface) throws SQLException {
		if (iface.isInstance(this))
			return (T) this;
		else
			throw new SQLException((new StringBuilder()).append("DataSource of type [").append(getClass().getName())
					.append("] cannot be unwrapped as [").append(iface.getName()).append("]").toString());
	}

	@Override
	public boolean isWrapperFor(Class<?> iface) throws SQLException {
		return iface.isInstance(this);
	}

	/** 获得一个可用的数据库连接 */
	@Override
	public abstract Connection getConnection() throws SQLException;

	@Override
	public Connection getConnection(String username, String password) throws SQLException {
		if (connProperties == null) {
			this.connProperties = new Properties();
		}
		if (!isEmpty(user)) {
			connProperties.put("user", user);
			connProperties.put("password", password);
		}
		return createConn();
	}

	protected boolean isEmpty(String text) {
		return text == null || "".equals(text);
	}

	protected boolean isNotEmpty(String text) {
		return !isEmpty(text);
	}

	protected int getIntValue(Integer value) {
		return value != null ? value : 0;
	}

	public void resetInit() {
		isInit = false;
	}

	/** 关闭stmt */
	protected void close(Statement stmt) {
		if (stmt != null) {
			try {
				stmt.close();
			} catch (SQLException e) {
				logger.warn(e.getMessage());
			} finally {
				stmt = null;
			}
		}
	}

	public java.util.logging.Logger getParentLogger() throws SQLFeatureNotSupportedException {
		return null;
	}

}
