package cn.elwy.common.jdbc;

import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStream;
import java.io.StringReader;
import java.lang.reflect.Field;
import java.sql.Blob;
import java.sql.CallableStatement;
import java.sql.Clob;
import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.ResultSetMetaData;
import java.sql.SQLException;
import java.sql.Statement;
import java.util.ArrayList;
import java.util.Date;
import java.util.HashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.concurrent.ConcurrentHashMap;

import javax.sql.DataSource;

import cn.elwy.common.exception.DaoException;
import cn.elwy.common.i18n.Msg;
import cn.elwy.common.jdbc.callback.CallableStatementHandler;
import cn.elwy.common.jdbc.callback.ConnectionHandler;
import cn.elwy.common.jdbc.callback.PreparedStatementHandler;
import cn.elwy.common.jdbc.callback.ResultData;
import cn.elwy.common.jdbc.callback.ResultSetHandler;
import cn.elwy.common.util.ArrayUtil;
import cn.elwy.common.util.BeanUtil;
import cn.elwy.common.util.CloseUtil;
import cn.elwy.common.util.ConvertTypeUtil;
import cn.elwy.common.util.EnumUtil;
import cn.elwy.common.util.IoUtil;

/**
 * JDBC数据库操作类
 * @author huangsq
 * @version 1.0, 2018-02-19
 */
@SuppressWarnings("unchecked")
public class JdbcSupport {

	/** 线程ID和数据源连接池的hashCode组成的当前连接Map */
	private static final Map<String, CurrentConnection> currentConnMap = new ConcurrentHashMap<String, CurrentConnection>();
	private static final HashMap<Integer, Map<String, List<Integer>>> nameIndexCache = new HashMap<Integer, Map<String, List<Integer>>>();

	/** 数据源连接池 */
	private DataSource dataSource;
	/** 默认字段名前缀 */
	public static final String FIELD_PREFIX = "F";
	/** 字段名前缀，用于删除掉前字段缀字符 */
	protected String fieldPrefix;

	public JdbcSupport() {
	}

	public JdbcSupport(DataSource dataSource) {
		this.dataSource = dataSource;
	}

	/** 字段名前缀，用于删除掉前字段缀字符 */
	public String getFieldPrefix() {
		return fieldPrefix;
	}

	/**
	 * 字段名前缀，用于删除掉前字段缀字符
	 * @param fieldPrefix
	 */
	public void setFieldPrefix(String fieldPrefix) {
		this.fieldPrefix = fieldPrefix;
	}

	/** 是否必须回滚 */
	public boolean isRollbackOnly() {
		return getCurrentConnection().isRollbackOnly();
	}

	/** 设置当前事务环境的回滚状态 */
	public void setRollbackOnly(boolean rollbackOnly) {
		getCurrentConnection().setRollbackOnly(rollbackOnly);
	}

	/** 当前是否存在事务 */
	public boolean existsTransaction() {
		return getCurrentConnection().isExistsTransaction();
	}

	/** 设置当前事务环境 */
	public void setExistsTransaction(boolean existsTransaction) {
		getCurrentConnection().setExistsTransaction(existsTransaction);
	}

	/** 获取当前线程中的数据库连接 */
	public CurrentConnection getCurrentConnection() {
		String currentConnKey = getCurrentConnKey();
		CurrentConnection currentConn = currentConnMap.get(currentConnKey);
		if (currentConn == null) {
			try {
				Connection conn = dataSource.getConnection();
				if (conn != null) {
					currentConn = new CurrentConnection(currentConnKey, conn);
					currentConnMap.put(currentConnKey, currentConn);
				} else {
					throw new DaoException(Msg.E_CONN_GET);
				}
			} catch (SQLException e) {
				throw new DaoException(e);
			}
		}
		return currentConn;
	}

	protected String getCurrentConnKey() {
		return Thread.currentThread().getId() + "_" + dataSource.hashCode();
	}

	/**
	 * 设置数据源连接池
	 * @param dataSource
	 */
	public void setDataSource(DataSource dataSource) {
		this.dataSource = dataSource;
	}

	/**
	 * 获取数据源连接池
	 */
	public DataSource getDataSource() {
		return dataSource;
	}

	/** 开始一个事务 */
	public void beginTransaction() {
		getCurrentConnection().beginTransaction();
	}

	/** 提交事务 */
	public void commit() {
		CurrentConnection currentConn = getCurrentConnection();
		try {
			currentConn.commit();
		} finally {
			removeCurrentConnection(currentConn);
		}
	}

	/** 回滚事务 */
	public void rollback() {
		CurrentConnection currentConn = getCurrentConnection();
		try {
			currentConn.rollback();
		} finally {
			removeCurrentConnection(currentConn);
		}
	}

	/**
	 * 创建Blob对象，连接关闭后需要重新创建Blob对象
	 * @throws SQLException
	 */
	public Blob createBlob() throws SQLException {
		return getCurrentConnection().getConn().createBlob();
	}

	/**
	 * 创建Blob对象，连接关闭后需要重新创建Blob对象
	 * @param content
	 * @throws SQLException
	 */
	public Blob createBlob(byte[] content) throws SQLException {
		Blob blob = createBlob();
		blob.setBytes(1, content);
		return blob;
	}

	// /**
	// * 创建Clob对象，连接关闭后需要重新创建Clob对象
	// * @throws SQLException
	// */
	// public Clob createClob() throws SQLException {
	// return getCurrentConnection().getConn().createClob();
	// }

	/**
	 * 创建Clob对象，连接关闭后需要重新创建Clob对象
	 * @param content
	 * @throws SQLException
	 */
	public StringReader createClob(String content) throws SQLException {
		StringReader reader = new StringReader(content);
		return reader;
	}

	/**
	 * 获得第一列的第一个值
	 * @param sql SQL语句
	 * @param args SQL参数
	 */
	public <T> T getValue(String sql, Object... args) {
		CurrentConnection currentConn = getCurrentConnection();
		Connection conn = currentConn.getConn();
		ResultSet rs = null; // 结果集
		PreparedStatement pstmt = null; // 操作数据库的对象
		try {
			pstmt = conn.prepareStatement(sql);
			setArgs(pstmt, args);
			rs = pstmt.executeQuery();
			if (rs.next()) {
				return (T) rs.getObject(1);
			}
			return null;
		} catch (SQLException e) {
			throw new DaoException(Msg.E_CONN_EXECUTE, e);
		} finally {
			CloseUtil.close(rs, pstmt);
			close(currentConn);
		}
	}

	/**
	 * 获得第一列的第一个值
	 * @param sql SQL语句
	 * @param args SQL参数
	 */
	public <T> T namedGetValue(String sql, Map<String, Object> args) {
		CurrentConnection currentConn = getCurrentConnection();
		Connection conn = currentConn.getConn();
		ResultSet rs = null; // 结果集
		PreparedStatement pstmt = null; // 操作数据库的对象
		try {
			Map<String, List<Integer>> nameIndexMap = null;
			int hashCode = sql.hashCode();
			if (nameIndexCache.containsKey(hashCode)) {
				nameIndexMap = nameIndexCache.get(hashCode);
			} else {
				nameIndexMap = parseSqlParam(sql);
				nameIndexCache.put(hashCode, nameIndexMap);
			}
			pstmt = conn.prepareStatement(sql);
			setArgs(pstmt, args, nameIndexMap);

			rs = pstmt.executeQuery();
			if (rs.next()) {
				return (T) rs.getObject(1);
			}
			return null;
		} catch (SQLException e) {
			throw new DaoException(Msg.E_CONN_EXECUTE, e);
		} finally {
			CloseUtil.close(rs, pstmt);
			close(currentConn);
		}
	}

	/**
	 * 获得第一列的值
	 * @param sql SQL语句
	 * @param args SQL参数
	 */
	public <T> List<T> getValueList(String sql, Object... args) {
		CurrentConnection currentConn = getCurrentConnection();
		Connection conn = currentConn.getConn();
		ResultSet rs = null; // 结果集
		PreparedStatement pstmt = null; // 操作数据库的对象
		List<T> list = new ArrayList<T>();
		try {
			pstmt = conn.prepareStatement(sql);
			setArgs(pstmt, args);
			rs = pstmt.executeQuery();
			while (rs.next()) {
				list.add((T) rs.getObject(1));
			}
			return list;
		} catch (SQLException e) {
			throw new DaoException(e);
		} finally {
			CloseUtil.close(rs, pstmt);
			close(currentConn);
		}
	}

	/**
	 * 获得第一列的值
	 * @param sql SQL语句
	 * @param args SQL参数
	 */
	public <T> List<T> namedGetValueList(String sql, Map<String, ?> args) {
		CurrentConnection currentConn = getCurrentConnection();
		Connection conn = currentConn.getConn();
		ResultSet rs = null; // 结果集
		PreparedStatement pstmt = null; // 操作数据库的对象
		List<T> list = new ArrayList<T>();
		try {
			Map<String, List<Integer>> nameIndexMap = null;
			int hashCode = sql.hashCode();
			if (nameIndexCache.containsKey(hashCode)) {
				nameIndexMap = nameIndexCache.get(hashCode);
			} else {
				nameIndexMap = parseSqlParam(sql);
				nameIndexCache.put(hashCode, nameIndexMap);
			}
			pstmt = conn.prepareStatement(sql);
			setArgs(pstmt, args, nameIndexMap);

			rs = pstmt.executeQuery();
			while (rs.next()) {
				list.add((T) rs.getObject(1));
			}
			return list;
		} catch (SQLException e) {
			throw new DaoException(e);
		} finally {
			CloseUtil.close(rs, pstmt);
			close(currentConn);
		}
	}

	/**
	 * 将查询结果放到一个List集合中，每个对象以Map存储，key对应SQL语句中的列名，key的值全部大写
	 * @param sql SQL语句
	 */
	public List<Map<String, ?>> query(String sql) {
		CurrentConnection currentConn = getCurrentConnection();
		Connection conn = currentConn.getConn();
		ResultSet rs = null; // 结果集
		Statement stmt = null; // 操作数据库的对象
		try {
			stmt = conn.createStatement();
			rs = stmt.executeQuery(sql);
			String[] columnNames = getColumnNames(rs.getMetaData());
			List<Map<String, ?>> list = new ArrayList<Map<String, ?>>();
			while (rs.next()) {
				list.add(rowMapper(rs, columnNames, new HashMap<String, Object>()));
			}
			return list;
		} catch (Exception e) {
			throw new DaoException(e);
		} finally {
			CloseUtil.close(rs, stmt);
			close(currentConn);
		}
	}

	/**
	 * 将查询结果放到一个List集合中，每个对象以Map存储，key对应SQL语句中的列名，key的值全部大写
	 * @param sql SQL语句
	 * @param args SQL参数
	 */
	public List<Map<String, ?>> query(String sql, Object... args) {
		CurrentConnection currentConn = getCurrentConnection();
		Connection conn = currentConn.getConn();
		ResultSet rs = null; // 结果集
		PreparedStatement pstmt = null; // 操作数据库的对象
		try {
			pstmt = conn.prepareStatement(sql);
			setArgs(pstmt, args);
			rs = pstmt.executeQuery();
			String[] columnNames = getColumnNames(rs.getMetaData());
			List<Map<String, ?>> list = new ArrayList<Map<String, ?>>();
			while (rs.next()) {
				list.add(rowMapper(rs, columnNames, new HashMap<String, Object>()));
			}
			return list;
		} catch (SQLException e) {
			throw new DaoException(e);
		} finally {
			CloseUtil.close(rs, pstmt);
			close(currentConn);
		}
	}

	/**
	 * 将查询结果放到一个List集合中，每个对象根据clazz类型创建实例
	 * @param sql SQL语句
	 * @param clazz 对象Class
	 */
	public <T> List<T> query(String sql, Class<T> clazz) {
		CurrentConnection currentConn = getCurrentConnection();
		Connection conn = currentConn.getConn();
		ResultSet rs = null; // 结果集
		Statement stmt = null; // 操作数据库的对象
		try {
			stmt = conn.createStatement();
			rs = stmt.executeQuery(sql);
			return queryForList(rs, clazz);
		} catch (Exception e) {
			throw new DaoException(e);
		} finally {
			CloseUtil.close(rs, stmt);
			close(currentConn);
		}
	}

	/**
	 * 将查询结果放到一个List集合中，每个对象根据clazz类型创建实例
	 * @param sql SQL语句
	 * @param clazz 对象Class
	 * @param args SQL参数
	 */
	public <T> List<T> query(String sql, Class<T> clazz, Object... args) {
		CurrentConnection currentConn = getCurrentConnection();
		Connection conn = currentConn.getConn();
		ResultSet rs = null; // 结果集
		PreparedStatement pstmt = null; // 操作数据库的对象
		try {
			pstmt = conn.prepareStatement(sql);
			setArgs(pstmt, args);
			rs = pstmt.executeQuery();
			return queryForList(rs, clazz);
		} catch (SQLException e) {
			throw new DaoException(e);
		} finally {
			CloseUtil.close(rs, pstmt);
			close(currentConn);
		}
	}

	/**
	 * 执行SQL查询语句，并返回它所生成结果的对象
	 * @param sql SQL语句
	 * @param handler ResultSet处理对象
	 * @param args SQL参数
	 */
	public <T> List<T> query(String sql, ResultSetHandler handler, Object... args) {
		CurrentConnection currentConn = getCurrentConnection();
		Connection conn = currentConn.getConn();
		ResultSet rs = null; // 结果集
		PreparedStatement pstmt = null; // 操作数据库的对象
		try {
			pstmt = conn.prepareStatement(sql);
			setArgs(pstmt, args);
			rs = pstmt.executeQuery();
			return (List<T>) handler.execute(rs);
		} catch (SQLException e) {
			throw new DaoException(e);
		} finally {
			CloseUtil.close(rs, pstmt);
			close(currentConn);
		}
	}

	/**
	 * 执行SQL查询语句，并返回它所生成结果的对象
	 * @param sql SQL语句
	 * @param rsi ResultSet游标参数
	 * @param handler ResultSet处理对象
	 * @param args SQL参数
	 */
	public ResultData query(String sql, ResultSetInfo rsi, Object... args) {
		CurrentConnection currentConn = getCurrentConnection();
		Connection conn = currentConn.getConn();
		ResultSet rs = null; // 结果集
		PreparedStatement pstmt = null; // 操作数据库的对象
		try {
			pstmt = conn.prepareStatement(sql, rsi.getResultSetType(), rsi.getResultSetConcurrency());
			setArgs(pstmt, args);
			rs = pstmt.executeQuery();
			ResultData d = new ResultData(currentConn, pstmt, rs);
			return d;
		} catch (SQLException e) {
			throw new DaoException(e);
		}
	}

	/**
	 * 将查询结果放到一个Map对象中，key对应SQL语句中的列名，key的值全部大写
	 * @param sql SQL语句
	 * @param map 存储对象的Map，如果为null会自动创建一个HashMap
	 */
	public Map<String, ?> queryForMap(String sql, Map<String, ?> map) {
		CurrentConnection currentConn = getCurrentConnection();
		Connection conn = currentConn.getConn();
		ResultSet rs = null; // 结果集
		Statement stmt = null; // 操作数据库的对象
		try {
			stmt = conn.createStatement();
			rs = stmt.executeQuery(sql);
			if (map == null) {
				map = new HashMap<String, Object>();
			}
			if (rs.next()) {
				ResultSetMetaData rsmd = rs.getMetaData();
				// 通过结果集元数据获得列名
				String[] columnNames = getColumnNames(rsmd);
				return rowMapper(rs, columnNames, (Map<String, Object>) map);
			} else {
				return map;
			}
		} catch (SQLException e) {
			throw new DaoException(e);
		} finally {
			CloseUtil.close(rs, stmt);
			close(currentConn);
		}
	}

	/**
	 * 将查询结果放到一个Map对象中，key对应SQL语句中的列名，key的值全部大写
	 * @param sql SQL语句
	 * @param map 存储对象的Map，如果为null会自动创建一个HashMap
	 * @param args SQL参数
	 */
	public Map<String, ?> queryForMap(String sql, Map<String, Object> map, Object... args) {
		CurrentConnection currentConn = getCurrentConnection();
		Connection conn = currentConn.getConn();
		ResultSet rs = null; // 结果集
		PreparedStatement pstmt = null; // 操作数据库的对象
		try {
			pstmt = conn.prepareStatement(sql);
			setArgs(pstmt, args);
			rs = pstmt.executeQuery();
			if (map == null) {
				map = new HashMap<String, Object>();
			}
			if (rs.next()) {
				ResultSetMetaData rsmd = rs.getMetaData();
				// 通过结果集元数据获得列名
				String[] columnNames = getColumnNames(rsmd);
				return rowMapper(rs, columnNames, (Map<String, Object>) map);
			} else {
				return map;
			}
		} catch (SQLException e) {
			throw new DaoException(e);
		} finally {
			CloseUtil.close(rs, pstmt);
			close(currentConn);
		}
	}

	/**
	 * 返回询结果中第一条记录，象根据clazz类型创建实例
	 * @param sql SQL语句
	 * @param clazz 对象Class
	 */
	public <T> T queryForObject(String sql, Class<T> clazz) {
		CurrentConnection currentConn = getCurrentConnection();
		Connection conn = currentConn.getConn();
		ResultSet rs = null; // 结果集
		Statement stmt = null; // 操作数据库的对象
		try {
			stmt = conn.createStatement();
			rs = stmt.executeQuery(sql);
			return queryForObject(rs, clazz);
		} catch (SQLException e) {
			throw new DaoException(e);
		} finally {
			CloseUtil.close(rs, stmt);
			close(currentConn);
		}
	}

	/**
	 * 获得对象值
	 * @param sql SQL语句
	 * @param clazz 对象Class
	 * @param args SQL参数
	 */
	public <T> T queryForObject(String sql, Class<T> clazz, Object... args) {
		CurrentConnection currentConn = getCurrentConnection();
		Connection conn = currentConn.getConn();
		ResultSet rs = null; // 结果集
		PreparedStatement pstmt = null; // 操作数据库的对象
		try {
			pstmt = conn.prepareStatement(sql);
			setArgs(pstmt, args);
			rs = pstmt.executeQuery();
			return queryForObject(rs, clazz);
		} catch (SQLException e) {
			throw new DaoException(e);
		} finally {
			CloseUtil.close(rs, pstmt);
			close(currentConn);
		}
	}

	/**
	 * 获得对象值
	 * @param sql SQL语句
	 * @param handler ResultSet处理对象
	 * @param args SQL参数
	 */
	public <T> T queryForObject(String sql, ResultSetHandler handler, Object... args) {
		CurrentConnection currentConn = getCurrentConnection();
		Connection conn = currentConn.getConn();
		ResultSet rs = null; // 结果集
		PreparedStatement pstmt = null; // 操作数据库的对象
		try {
			pstmt = conn.prepareStatement(sql);
			setArgs(pstmt, args);
			rs = pstmt.executeQuery();
			return (T) handler.execute(rs);
		} catch (SQLException e) {
			throw new DaoException(e);
		} finally {
			CloseUtil.close(rs, pstmt);
			close(currentConn);
		}
	}

	/**
	 * 将查询结果放到一个List集合中，每个对象以Map存储，key对应SQL语句中的列名，key的值全部大写
	 * @param sql
	 * @param args
	 * @return
	 */
	public List<Map<String, ?>> namedQuery(String sql, Map<String, ?> args) {
		CurrentConnection currentConn = getCurrentConnection();
		Connection conn = currentConn.getConn();
		ResultSet rs = null; // 结果集
		PreparedStatement pstmt = null; // 操作数据库的对象
		try {
			Map<String, List<Integer>> nameIndexMap = null;
			int hashCode = sql.hashCode();
			if (nameIndexCache.containsKey(hashCode)) {
				nameIndexMap = nameIndexCache.get(hashCode);
			} else {
				nameIndexMap = parseSqlParam(sql);
				nameIndexCache.put(hashCode, nameIndexMap);
			}
			pstmt = conn.prepareStatement(sql);
			setArgs(pstmt, args, nameIndexMap);
			rs = pstmt.executeQuery();
			String[] columnNames = getColumnNames(rs.getMetaData());
			List<Map<String, ?>> list = new ArrayList<Map<String, ?>>();
			while (rs.next()) {
				list.add(rowMapper(rs, columnNames, new HashMap<String, Object>()));
			}
			return list;
		} catch (SQLException e) {
			throw new DaoException(e);
		} finally {
			CloseUtil.close(rs, pstmt);
			close(currentConn);
		}
	}

	/**
	 * 将查询结果放到一个List集合中，每个对象根据clazz类型创建实例
	 * @param sql SQL语句
	 * @param clazz 对象Class
	 * @param args SQL参数
	 */
	public <T> List<T> namedQuery(String sql, Class<T> clazz, Map<String, ?> args) {
		CurrentConnection currentConn = getCurrentConnection();
		Connection conn = currentConn.getConn();
		ResultSet rs = null; // 结果集
		PreparedStatement pstmt = null; // 操作数据库的对象
		try {
			Map<String, List<Integer>> nameIndexMap = null;
			int hashCode = sql.hashCode();
			if (nameIndexCache.containsKey(hashCode)) {
				nameIndexMap = nameIndexCache.get(hashCode);
			} else {
				nameIndexMap = parseSqlParam(sql);
				nameIndexCache.put(hashCode, nameIndexMap);
			}
			pstmt = conn.prepareStatement(sql);
			setArgs(pstmt, args, nameIndexMap);
			rs = pstmt.executeQuery();
			return queryForList(rs, clazz);
		} catch (SQLException e) {
			throw new DaoException(e);
		} finally {
			CloseUtil.close(rs, pstmt);
			close(currentConn);
		}
	}

	/**
	 * 执行SQL查询语句，并返回它所生成结果的对象
	 * @param sql SQL语句
	 * @param handler ResultSet处理对象
	 * @param args SQL参数
	 */
	public <T> List<T> namedQuery(String sql, ResultSetHandler handler, Map<String, ?> args) {
		CurrentConnection currentConn = getCurrentConnection();
		Connection conn = currentConn.getConn();
		ResultSet rs = null; // 结果集
		PreparedStatement pstmt = null; // 操作数据库的对象
		try {
			Map<String, List<Integer>> nameIndexMap = null;
			int hashCode = sql.hashCode();
			if (nameIndexCache.containsKey(hashCode)) {
				nameIndexMap = nameIndexCache.get(hashCode);
			} else {
				nameIndexMap = parseSqlParam(sql);
				nameIndexCache.put(hashCode, nameIndexMap);
			}
			pstmt = conn.prepareStatement(sql);
			setArgs(pstmt, args, nameIndexMap);
			rs = pstmt.executeQuery();
			return (List<T>) handler.execute(rs);
		} catch (SQLException e) {
			throw new DaoException(e);
		} finally {
			CloseUtil.close(rs, pstmt);
			close(currentConn);
		}
	}

	/**
	 * 执行SQL查询语句，使用:NAME形式占位符，NAME的值和args中的key对应区分大小写，并返回它所生成结果的对象
	 * @param sql SQL语句
	 * @param rsi ResultSet游标参数
	 * @param handler ResultSet处理对象
	 * @param args SQL参数
	 */
	public ResultData namedQuery(String sql, ResultSetInfo rsi, Map<String, ?> args) {
		CurrentConnection currentConn = getCurrentConnection();
		Connection conn = currentConn.getConn();
		ResultSet rs = null; // 结果集
		PreparedStatement pstmt = null; // 操作数据库的对象
		try {
			Map<String, List<Integer>> nameIndexMap = null;
			int hashCode = sql.hashCode();
			if (nameIndexCache.containsKey(hashCode)) {
				nameIndexMap = nameIndexCache.get(hashCode);
			} else {
				nameIndexMap = parseSqlParam(sql);
				nameIndexCache.put(hashCode, nameIndexMap);
			}
			pstmt = conn.prepareStatement(sql, rsi.getResultSetType(), rsi.getResultSetConcurrency());
			setArgs(pstmt, args, nameIndexMap);
			rs = pstmt.executeQuery();
			ResultData d = new ResultData(currentConn, pstmt, rs);
			return d;
		} catch (SQLException e) {
			throw new DaoException(e);
		}
	}

	/**
	 * 将查询结果放到一个Map对象中，key对应SQL语句中的列名，key的值全部大写
	 * @param sql SQL语句
	 * @param msgMaps 存储对象的Map，如果为null会自动创建一个HashMap
	 * @param args SQL参数
	 */
	public Map<String, ?> namedQueryForMap(String sql, Map<String, ?> map, Map<String, ?> args) {
		CurrentConnection currentConn = getCurrentConnection();
		Connection conn = currentConn.getConn();
		ResultSet rs = null; // 结果集
		PreparedStatement pstmt = null; // 操作数据库的对象
		try {
			Map<String, List<Integer>> nameIndexMap = null;
			int hashCode = sql.hashCode();
			if (nameIndexCache.containsKey(hashCode)) {
				nameIndexMap = nameIndexCache.get(hashCode);
			} else {
				nameIndexMap = parseSqlParam(sql);
				nameIndexCache.put(hashCode, nameIndexMap);
			}
			pstmt = conn.prepareStatement(sql);
			setArgs(pstmt, args, nameIndexMap);
			rs = pstmt.executeQuery();
			if (rs.next()) {
				if (map == null) {
					map = new HashMap<String, Object>();
				}
				ResultSetMetaData rsmd = rs.getMetaData();
				// 通过结果集元数据获得列名
				String[] columnNames = getColumnNames(rsmd);
				return rowMapper(rs, columnNames, (Map<String, Object>) map);
			} else {
				return map;
			}
		} catch (SQLException e) {
			throw new DaoException(e);
		} finally {
			CloseUtil.close(rs, pstmt);
			close(currentConn);
		}
	}

	/**
	 * 获得对象值
	 * @param sql SQL语句
	 * @param clazz 对象Class
	 * @param args SQL参数
	 */
	public <T> T namedQueryForObject(String sql, Class<T> clazz, Map<String, ?> args) {
		CurrentConnection currentConn = getCurrentConnection();
		Connection conn = currentConn.getConn();
		ResultSet rs = null; // 结果集
		PreparedStatement pstmt = null; // 操作数据库的对象
		try {
			Map<String, List<Integer>> nameIndexMap = null;
			int hashCode = sql.hashCode();
			if (nameIndexCache.containsKey(hashCode)) {
				nameIndexMap = nameIndexCache.get(hashCode);
			} else {
				nameIndexMap = parseSqlParam(sql);
				nameIndexCache.put(hashCode, nameIndexMap);
			}
			pstmt = conn.prepareStatement(sql);
			setArgs(pstmt, args, nameIndexMap);
			rs = pstmt.executeQuery();
			return queryForObject(rs, clazz);
		} catch (SQLException e) {
			throw new DaoException(e);
		} finally {
			CloseUtil.close(rs, pstmt);
			close(currentConn);
		}
	}

	/**
	 * 获得对象值
	 * @param sql SQL语句
	 * @param handler ResultSet处理对象
	 * @param args SQL参数
	 */
	public <T> T namedQueryForObject(String sql, ResultSetHandler handler, Map<String, ?> args) {
		CurrentConnection currentConn = getCurrentConnection();
		Connection conn = currentConn.getConn();
		ResultSet rs = null; // 结果集
		PreparedStatement pstmt = null; // 操作数据库的对象
		try {
			Map<String, List<Integer>> nameIndexMap = null;
			int hashCode = sql.hashCode();
			if (nameIndexCache.containsKey(hashCode)) {
				nameIndexMap = nameIndexCache.get(hashCode);
			} else {
				nameIndexMap = parseSqlParam(sql);
				nameIndexCache.put(hashCode, nameIndexMap);
			}
			pstmt = conn.prepareStatement(sql);
			setArgs(pstmt, args, nameIndexMap);
			rs = pstmt.executeQuery();
			return (T) handler.execute(rs);
		} catch (SQLException e) {
			throw new DaoException(e);
		} finally {
			CloseUtil.close(rs, pstmt);
			close(currentConn);
		}
	}

	/**
	 * 执行静态SQL更新语句 并返回影响数据的条数
	 * @param sql SQL语句
	 */
	public int update(String sql) {
		CurrentConnection currentConn = getCurrentConnection();
		Connection conn = currentConn.getConn();
		Statement stmt = null; // 操作数据库的对象
		try {
			stmt = conn.createStatement();
			return stmt.executeUpdate(sql);
		} catch (SQLException e) {
			throw new DaoException(e);
		} finally {
			CloseUtil.close(stmt);
			close(currentConn);
		}
	}

	/**
	 * 执行SQL更新语句 并返回影响数据的条数
	 * @param sql SQL语句
	 * @param args SQL参数
	 */
	public int update(String sql, Object... args) {
		int row = 0;
		CurrentConnection currentConn = getCurrentConnection();
		Connection conn = currentConn.getConn();
		PreparedStatement pstmt = null; // 操作数据库的对象
		try {
			pstmt = conn.prepareStatement(sql);
			setArgs(pstmt, args);
			row = pstmt.executeUpdate();
			return row;
		} catch (SQLException e) {
			throw new DaoException(e);
		} finally {
			CloseUtil.close(pstmt);
			close(currentConn);
		}
	}

	/**
	 * 执行SQL更新语句 并返回影响数据的条数
	 * @param sql SQL语句
	 * @param args SQL参数
	 */
	public int namedUpdate(String sql, Map<String, ?> args) {
		int row = 0;
		CurrentConnection currentConn = getCurrentConnection();
		Connection conn = currentConn.getConn();
		PreparedStatement pstmt = null; // 操作数据库的对象
		try {
			Map<String, List<Integer>> nameIndexMap = null;
			int hashCode = sql.hashCode();
			if (nameIndexCache.containsKey(hashCode)) {
				nameIndexMap = nameIndexCache.get(hashCode);
			} else {
				nameIndexMap = parseSqlParam(sql);
				nameIndexCache.put(hashCode, nameIndexMap);
			}
			pstmt = conn.prepareStatement(sql);
			setArgs(pstmt, args, nameIndexMap);

			row = pstmt.executeUpdate();
			return row;
		} catch (SQLException e) {
			throw new DaoException(e);
		} finally {
			CloseUtil.close(pstmt);
			close(currentConn);
		}
	}

	/**
	 * 执行静态SQL语句
	 * @param sql SQL语句
	 */
	public boolean execute(String sql) {
		CurrentConnection currentConn = getCurrentConnection();
		Connection conn = currentConn.getConn();
		Statement stmt = null; // 操作数据库的对象
		try {
			stmt = conn.createStatement();
			return stmt.execute(sql);
		} catch (SQLException e) {
			throw new DaoException(e);
		} finally {
			CloseUtil.close(stmt);
			close(currentConn);
		}
	}

	/**
	 * 通过给定数据库连接调用回调函数
	 * @param sql SQL语句
	 * @param connHandler ConnectionHandler处理对象
	 * @param args SQL参数
	 */
	public Object execute(ConnectionHandler connHandler) {
		CurrentConnection currentConn = getCurrentConnection();
		Connection conn = currentConn.getConn();
		try {
			return connHandler.execute(conn);
		} finally {
			close(currentConn);
		}
	}

	/**
	 * 执行动态SQL语句
	 * @param sql SQL语句
	 * @param args SQL参数
	 */
	public boolean execute(String sql, Object... args) {
		CurrentConnection currentConn = getCurrentConnection();
		Connection conn = currentConn.getConn();
		PreparedStatement pstmt = null; // 操作数据库的对象
		try {
			pstmt = conn.prepareStatement(sql);
			setArgs(pstmt, args);
			return pstmt.execute();
		} catch (SQLException e) {
			throw new DaoException(e);
		} finally {
			CloseUtil.close(pstmt);
			close(currentConn);
		}
	}

	/**
	 * 执行动态SQL语句
	 * @param sql SQL语句
	 * @param args SQL参数
	 */
	public boolean namedExecute(String sql, Map<String, ?> args) {
		CurrentConnection currentConn = getCurrentConnection();
		Connection conn = currentConn.getConn();
		PreparedStatement pstmt = null; // 操作数据库的对象
		try {
			Map<String, List<Integer>> nameIndexMap = null;
			int hashCode = sql.hashCode();
			if (nameIndexCache.containsKey(hashCode)) {
				nameIndexMap = nameIndexCache.get(hashCode);
			} else {
				nameIndexMap = parseSqlParam(sql);
				nameIndexCache.put(hashCode, nameIndexMap);
			}
			pstmt = conn.prepareStatement(sql);
			setArgs(pstmt, args, nameIndexMap);

			return pstmt.execute();
		} catch (SQLException e) {
			throw new DaoException(e);
		} finally {
			CloseUtil.close(pstmt);
			close(currentConn);
		}
	}

	/**
	 * 执行动态SQL语句
	 * @param sql SQL语句
	 * @param handler PreparedStatement处理对象
	 * @param args SQL参数
	 */
	public Object execute(String sql, PreparedStatementHandler handler, Object... args) {
		CurrentConnection currentConn = getCurrentConnection();
		Connection conn = currentConn.getConn();
		PreparedStatement pstmt = null; // 操作数据库的对象
		try {
			pstmt = conn.prepareStatement(sql);
			setArgs(pstmt, args);
			return handler.execute(pstmt);
		} catch (SQLException e) {
			throw new DaoException(e);
		} finally {
			CloseUtil.close(pstmt);
			close(currentConn);
		}
	}

	/**
	 * 执行动态SQL语句
	 * @param sql SQL语句
	 * @param handler PreparedStatement处理对象
	 * @param args SQL参数
	 */
	public Object namedExecute(String sql, PreparedStatementHandler handler, Map<String, ?> args) {
		CurrentConnection currentConn = getCurrentConnection();
		Connection conn = currentConn.getConn();
		PreparedStatement pstmt = null; // 操作数据库的对象
		try {
			Map<String, List<Integer>> nameIndexMap = null;
			int hashCode = sql.hashCode();
			if (nameIndexCache.containsKey(hashCode)) {
				nameIndexMap = nameIndexCache.get(hashCode);
			} else {
				nameIndexMap = parseSqlParam(sql);
				nameIndexCache.put(hashCode, nameIndexMap);
			}
			pstmt = conn.prepareStatement(sql);
			setArgs(pstmt, args, nameIndexMap);

			return handler.execute(pstmt);
		} catch (SQLException e) {
			throw new DaoException(e);
		} finally {
			CloseUtil.close(pstmt);
			close(currentConn);
		}
	}

	/**
	 * 批处理多条SQL语句，SQL语句务参数
	 * @param sqls SQL语句
	 */
	public int[] executeBatch(String[] sqls) {
		if (sqls == null || sqls.length < 1) {
			throw new DaoException("没有SQL语句！");
		}
		CurrentConnection currentConn = getCurrentConnection();
		Connection conn = currentConn.getConn();
		Statement stmt = null; // 操作数据库的对象
		try {
			stmt = conn.createStatement(); // 获取 Statement 对象的实例
			int count = sqls.length;
			for (int i = 0; i < count; i++) { // 一条一条的添加
				stmt.addBatch(sqls[i]);
			}
			int[] executeBatch = stmt.executeBatch();
			return executeBatch; // 统一执行，返回响应行数
		} catch (SQLException e) {
			throw new DaoException("执行批处理时出现了异常！", e);
		} finally {
			CloseUtil.close(stmt);
			close(currentConn);
		}
	}

	/**
	 * 批处理多条SQL语句，SQL语句带参数
	 * @param sqls SQL语句
	 * @param batchSize 批量处理大小
	 */
	public int[] executeBatch(String[] sqls, int batchSize) {
		if (sqls == null || sqls.length < 1) {
			throw new DaoException("没有SQL语句！");
		}
		CurrentConnection currentConn = getCurrentConnection();
		Connection conn = currentConn.getConn();
		Statement stmt = null; // 操作数据库的对象
		try {
			stmt = conn.createStatement(); // 获取 Statement 对象的实例
			int count = sqls.length;
			int[] result = new int[count];
			for (int i = 1; i <= count; i++) { // 一条一条的添加
				stmt.addBatch(sqls[i - 1]);
				if (i % batchSize == 0) {
					int[] executeBatch = stmt.executeBatch();
					System.arraycopy(executeBatch, 0, result, i - batchSize, executeBatch.length);
				}
			}
			if (count % batchSize != 0) {
				int[] executeBatch = stmt.executeBatch();
				System.arraycopy(executeBatch, 0, result, result.length - executeBatch.length, executeBatch.length);
			}
			return result;
		} catch (SQLException e) {
			throw new DaoException("执行批处理时出现了异常！", e);
		} finally {
			CloseUtil.close(stmt);
			close(currentConn);
		}
	}

	/**
	 * 批处理多条SQL语句，SQL语句带参数
	 * @param sql SQL语句
	 * @param batchSize 批量处理大小
	 * @param args SQL参数
	 */
	public int[] executeBatch(String sql, int batchSize, Object[][] args) {
		if (sql == null || args == null || args.length < 1) {
			throw new DaoException("没有SQL语句！");
		}
		CurrentConnection currentConn = getCurrentConnection();
		Connection conn = currentConn.getConn();
		PreparedStatement pstmt = null;
		try {
			pstmt = conn.prepareStatement(sql);
			int count = args.length;
			int[] result = new int[count];
			for (int i = 1; i <= count; i++) {
				setArgs(pstmt, args[i - 1]);
				pstmt.addBatch();
				if (i % batchSize == 0) {
					int[] executeBatch = pstmt.executeBatch();
					System.arraycopy(executeBatch, 0, result, i - batchSize, executeBatch.length);
				}
			}
			if (count % batchSize != 0) {
				int[] executeBatch = pstmt.executeBatch();
				System.arraycopy(executeBatch, 0, result, result.length - executeBatch.length, executeBatch.length);
			}
			return result;
		} catch (SQLException e) {
			throw new DaoException("执行批处理时出现了异常！", e);
		} finally {
			CloseUtil.close(pstmt);
			close(currentConn);
		}
	}

	/**
	 * 批处理多条SQL语句，SQL语句带参数
	 * @param sql SQL语句
	 * @param batchSize 批量处理大小
	 * @param args SQL参数
	 */
	public int[] namedExecuteBatch(String sql, int batchSize, List<Map<String, Object>> args) {
		if (sql == null || args == null || args.size() < 1) {
			throw new DaoException("没有SQL语句！");
		}
		CurrentConnection currentConn = getCurrentConnection();
		Connection conn = currentConn.getConn();
		PreparedStatement pstmt = null;
		try {
			Map<String, List<Integer>> nameIndexMap = null;
			int hashCode = sql.hashCode();
			if (nameIndexCache.containsKey(hashCode)) {
				nameIndexMap = nameIndexCache.get(hashCode);
			} else {
				nameIndexMap = parseSqlParam(sql);
				nameIndexCache.put(hashCode, nameIndexMap);
			}
			pstmt = conn.prepareStatement(sql);

			int count = args.size();
			int[] result = new int[count];
			for (int i = 1; i <= count; i++) {
				setArgs(pstmt, args, nameIndexMap);
				pstmt.addBatch();
				if (i % batchSize == 0) {
					int[] executeBatch = pstmt.executeBatch();
					System.arraycopy(executeBatch, 0, result, i - batchSize, executeBatch.length);
				}
			}
			if (count % batchSize != 0) {
				int[] executeBatch = pstmt.executeBatch();
				System.arraycopy(executeBatch, 0, result, result.length - executeBatch.length, executeBatch.length);
			}
			return result;
		} catch (SQLException e) {
			throw new DaoException("执行批处理时出现了异常！", e);
		} finally {
			CloseUtil.close(pstmt);
			close(currentConn);
		}
	}

	/**
	 * 调用存储过程
	 * @param sql SQL语句
	 * @param args SQL参数
	 */
	public boolean executeProc(String sql, Object... args) {
		CurrentConnection currentConn = getCurrentConnection();
		Connection conn = currentConn.getConn();
		CallableStatement cstmt = null; // 操作数据库的对象
		try {
			cstmt = conn.prepareCall(sql);
			setArgs(cstmt, args);
			return cstmt.execute();
		} catch (SQLException e) {
			throw new DaoException(e);
		} finally {
			CloseUtil.close(cstmt);
			close(currentConn);
		}
	}

	/**
	 * 调用存储过程
	 * @param sql SQL语句
	 * @param args SQL参数
	 */
	public boolean namedExecuteProc(String sql, Map<String, ?> args) {
		CurrentConnection currentConn = getCurrentConnection();
		Connection conn = currentConn.getConn();
		CallableStatement cstmt = null; // 操作数据库的对象
		try {
			Map<String, List<Integer>> nameIndexMap = null;
			int hashCode = sql.hashCode();
			if (nameIndexCache.containsKey(hashCode)) {
				nameIndexMap = nameIndexCache.get(hashCode);
			} else {
				nameIndexMap = parseSqlParam(sql);
				nameIndexCache.put(hashCode, nameIndexMap);
			}

			cstmt = conn.prepareCall(sql);
			setArgs(cstmt, args, nameIndexMap);
			return cstmt.execute();
		} catch (SQLException e) {
			throw new DaoException(e);
		} finally {
			CloseUtil.close(cstmt);
			close(currentConn);
		}
	}

	/**
	 * 调用存储过程
	 * @param sql SQL语句
	 * @param handler CallableStatement处理对象
	 * @param args SQL参数
	 */
	public Object executeProc(String sql, CallableStatementHandler handler, Object... args) {
		CurrentConnection currentConn = getCurrentConnection();
		Connection conn = currentConn.getConn();
		CallableStatement cstmt = null; // 操作数据库的对象
		try {
			cstmt = conn.prepareCall(sql);
			setArgs(cstmt, args);
			return handler.execute(cstmt);
		} catch (SQLException e) {
			throw new DaoException(e);
		} finally {
			CloseUtil.close(cstmt);
			close(currentConn);
		}
	}

	/**
	 * 调用存储过程
	 * @param sql SQL语句
	 * @param handler CallableStatement处理对象
	 * @param args SQL参数
	 */
	public Object namedExecuteProc(String sql, CallableStatementHandler handler, Map<String, ?> args) {
		CurrentConnection currentConn = getCurrentConnection();
		Connection conn = currentConn.getConn();
		CallableStatement cstmt = null; // 操作数据库的对象
		try {
			Map<String, List<Integer>> nameIndexMap = null;
			int hashCode = sql.hashCode();
			if (nameIndexCache.containsKey(hashCode)) {
				nameIndexMap = nameIndexCache.get(hashCode);
			} else {
				nameIndexMap = parseSqlParam(sql);
				nameIndexCache.put(hashCode, nameIndexMap);
			}

			cstmt = conn.prepareCall(sql);
			setArgs(cstmt, args, nameIndexMap);
			return handler.execute(cstmt);
		} catch (SQLException e) {
			throw new DaoException(e);
		} finally {
			CloseUtil.close(cstmt);
			close(currentConn);
		}
	}

	/**
	 * 释放连接，根据是否开启事物来决定是否关闭当前连接
	 * @param currentConn 当前连接
	 */
	protected void close(CurrentConnection currentConn) {
		if (currentConn.close()) {
			removeCurrentConnection(currentConn);
		}
	}

	private void removeCurrentConnection(CurrentConnection currentConn) {
		currentConn.setConn(null);
		currentConnMap.remove(currentConn.getKey());
	}

	/**
	 * 查询结果集将结果封装成Class<T>类型的List返回
	 * @param rs 结果集
	 * @param clazz 对象Class
	 * @throws SQLException
	 */
	protected <T> List<T> queryForList(ResultSet rs, Class<T> clazz) throws SQLException {
		ClassLoader ocl = Thread.currentThread().getContextClassLoader();
		ClassLoader ncl = this.getClass().getClassLoader();
		try {
			Thread.currentThread().setContextClassLoader(ncl);

			List<T> list = new ArrayList<T>();
			ResultSetMetaData rsmd = rs.getMetaData();
			// 通过结果集元数据获得列名
			String[][] fieldNames = getFieldNames(rsmd);
			// 获得对象的所有属性
			Field[] fields = getBeanFields(clazz, null);

			while (rs.next()) {
				list.add((T) rowMapper(rs, fieldNames, fields, clazz));
			}
			return list;
		} finally {
			Thread.currentThread().setContextClassLoader(ocl);
		}
	}

	/**
	 * 查询结果集将结果封装成Class<T>对象返回
	 * @param rs 结果集
	 * @param clazz 对象Class
	 */
	protected <T> T queryForObject(ResultSet rs, Class<T> clazz) throws SQLException {
		if (rs.next()) {
			ClassLoader ocl = Thread.currentThread().getContextClassLoader();
			ClassLoader ncl = this.getClass().getClassLoader();
			try {
				Thread.currentThread().setContextClassLoader(ncl);

				ResultSetMetaData rsmd = rs.getMetaData();
				// 通过结果集元数据获得列名
				String[][] fieldNames = getFieldNames(rsmd);
				// 获得对象的所有属性
				Field[] fields = getBeanFields(clazz, null);

				return rowMapper(rs, fieldNames, fields, clazz);
			} finally {
				Thread.currentThread().setContextClassLoader(ocl);
			}
		} else {
			return null;
		}
	}

	/**
	 * 获得结果集的列名
	 * @param rsmd 结果集元数据
	 * @throws SQLException
	 */
	public static String[] getColumnNames(ResultSetMetaData rsmd) throws SQLException {
		// 通过结果集元数据获得列数
		int columnCount = rsmd.getColumnCount();
		String[] columnNames = new String[columnCount];
		// 获得查询属性名称
		for (int i = 0; i < columnCount; i++) {
			columnNames[i] = rsmd.getColumnLabel(i + 1).toUpperCase();
		}
		return columnNames;
	}

	/**
	 * 获得结果集的列名和规范的属性名
	 * @param rsmd 结果集元数据
	 * @throws SQLException
	 */
	protected String[][] getFieldNames(ResultSetMetaData rsmd) throws SQLException {
		// 通过结果集元数据获得列数
		int columnCount = rsmd.getColumnCount();
		String[] columnNames = new String[columnCount];
		String[] fieldNames = new String[columnCount];
		// 获得查询属性名称
		for (int i = 0; i < columnCount; i++) {
			columnNames[i] = rsmd.getColumnLabel(i + 1);
			String field = columnNames[i];
			if (fieldPrefix != null) {
				field = field.replaceFirst(fieldPrefix, "");
			}
			fieldNames[i] = field.replaceAll("_", "");
		}
		return new String[][] { fieldNames, columnNames };
	}

	/**
	 * 根据结果集，返回Map格式的对象
	 * @param rs 结果集
	 * @param fieldNames 查询结果集的列名和规范的属性名
	 * @param fields 对象的字段
	 * @param clazz 对象Class
	 */
	public static Map<String, ?> rowMapper(ResultSet rs, String[] fieldNames, Map<String, Object> map) {
		String attributeName = null;
		try {
			for (int i = 0; i < fieldNames.length; i++) {
				attributeName = fieldNames[i];
				Object value = parseValue(null, rs.getObject(attributeName));
				map.put(fieldNames[i], value);
			}
			return map;
		} catch (Exception e) {
			throw new DaoException(Msg.format(Msg.F_SET_PROPERTY, attributeName), e);
		}
	}

	private static Object parseValue(Class<?> type, Object value) throws SQLException, IOException {
		if (value instanceof Clob) {
			Clob clob = (Clob) value;
			BufferedReader br = new BufferedReader(clob.getCharacterStream());
			char[] buffer = new char[8192];
			int length;
			StringBuffer content = new StringBuffer();
			while ((length = br.read(buffer)) != -1) {
				content.append(buffer, 0, length);
			}
			CloseUtil.close(br);
			value = content.toString();
		} else if (value instanceof Blob) {
			Blob blob = (Blob) value;
			InputStream in = blob.getBinaryStream();
			value = IoUtil.toByteArray(in);
		}
		if (type != null) {
			if (Enum.class.isAssignableFrom(type)) {
				value = EnumUtil.getEnum(type, value);
			} else if (boolean.class == type) {
				value = ConvertTypeUtil.toBoolean(value);
			} else if (Boolean.class == type) {
				value = ConvertTypeUtil.toBoolean(value, null);
			}
		}
		return value;
	}

	/**
	 * 根据结果集，实例化指定的对象
	 * @param rs 结果集
	 * @param fieldNames 查询结果集的列名和规范的属性名
	 * @param fields 对象的字段
	 * @param clazz 对象Class
	 */
	public static <T> T rowMapper(ResultSet rs, String[][] fieldNames, Field[] fields, Class<T> clazz) {
		String attributeName = null;
		try {
			// 通过默认构造方法创建一个新的对象
			Object object = clazz.newInstance();
			for (int i = 0; i < fieldNames[0].length; i++) {
				// 获得查询属性名称
				attributeName = fieldNames[0][i];
				for (Field field : fields) {
					// 获取对象的属性名称
					String fieldName = field.getName();
					if (attributeName.equalsIgnoreCase(fieldName.replaceAll("_", ""))) {
						Object value = null;
						Class<?> type = field.getType();
						if (type == Date.class) {
							value = rs.getTimestamp(fieldNames[1][i]);
						} else {
							value = rs.getObject(fieldNames[1][i]);
						}
						if (null != value) {
							value = parseValue(type, value);
							BeanUtil.copyProperty(object, fieldName, value);
						}
						break;
					}
				}
			}
			return (T) object;
		} catch (Exception e) {
			throw new DaoException(Msg.format(Msg.F_SET_PROPERTY, attributeName), e);
		}
	}

	/**
	 * 获取对象的所有属性，包含父类的属性
	 * @param clazz 对象Class
	 * @param fields 所有字段数组
	 * @return
	 */
	public static Field[] getBeanFields(Class<?> clazz, Field[] fields) {
		fields = (Field[]) ArrayUtil.addAll(fields, clazz.getDeclaredFields());
		if (clazz.getSuperclass() != null) {
			fields = getBeanFields(clazz.getSuperclass(), fields);
		}
		return fields;
	}

	/**
	 * 初始化SQL参数
	 * @param pstmt 要进行初始化的 PreparedStatement对象
	 * @param args SQL参数
	 */
	protected static void setArgs(PreparedStatement pstmt, Object... args) {
		if (args == null) {
			return;
		}
		try {
			for (int i = 0; i < args.length; i++) {
				pstmt.setObject(i + 1, args[i]);
			}
		} catch (SQLException e) {
			throw new DaoException(Msg.E_INIT_SQL_ARGS, e);
		}
	}

	public static final Map<String, List<Integer>> parseSqlParam(String sql) {
		Map<String, List<Integer>> paramMap = new HashMap<String, List<Integer>>();
		int length = sql.length();
		boolean inSingleQuote = false;
		boolean inDoubleQuote = false;
		int index = 1;
		for (int i = 0; i < length; i++) {
			char c = sql.charAt(i);
			if (inSingleQuote) {
				if (c == '\'') {
					inSingleQuote = false;
				}
			} else if (inDoubleQuote) {
				if (c == '"') {
					inDoubleQuote = false;
				}
			} else {
				if (c == '\'') {
					inSingleQuote = true;
				} else if (c == '"') {
					inDoubleQuote = true;
				} else if (c == ':' && i + 1 < length && Character.isJavaIdentifierStart(sql.charAt(i + 1))) {
					int j = i + 2;
					while (j < length && Character.isJavaIdentifierPart(sql.charAt(j))) {
						j++;
					}
					String name = sql.substring(i + 1, j);
					i += name.length();

					List<Integer> indexList = (List<Integer>) paramMap.get(name);
					if (indexList == null) {
						indexList = new LinkedList<Integer>();
						paramMap.put(name, indexList);
					}
					indexList.add(index);

					index++;
				}
			}
		}
		return paramMap;
	}

	/**
	 * 初始化SQL参数
	 * @param pstmt 要进行初始化的 PreparedStatement对象
	 * @param args SQL参数
	 * @param indexMap 参数名称所在位置
	 */
	protected static void setArgs(PreparedStatement pstmt, Map<String, Object> args,
			Map<String, List<Integer>> indexMap) {
		if (args == null || indexMap == null) {
			return;
		}
		try {
			for (Entry<String, Object> entry : args.entrySet()) {
				String key = entry.getKey();
				List<Integer> list = indexMap.get(key);
				if (list == null) {
					throw new DaoException("Parameter not found: " + key);
				}
				for (int index : list) {
					pstmt.setObject(index, entry.getValue());
				}
			}
		} catch (SQLException e) {
			throw new DaoException(Msg.E_INIT_SQL_ARGS, e);
		}
	}

}