package cn.elwy.common.jdbc;

import java.io.Serializable;
import java.util.Date;
import java.util.Properties;

/**
 * 数据库配置信息
 * @author huangsq
 * @version 1.0, 2018-02-19
 */
public class DBInfo implements Cloneable, Serializable {

	private static final long serialVersionUID = 1L;
	public static int SINGLE = 0;
	public static int CLUSTER = 1;
	public static int JNDI = 2;

	private String id; // ID
	private String dsName; // 数据源名称
	private String dbType; // 数据库类型
	private String jndiName; // JNDI名称
	private String driver; // 数据库驱动
	private String url; // 数据库URL
	private String host; // 数据库主机
	private int port; // 数据库端口
	private String dbName; // 数据库名称
	private String user; // 数据库用户名
	private String password; // 数据库密码
	private int loginTimeout; // 登录超时时间
	private String parameter; // 数据库连接参数
	private String testQuery; // 测试连接是否可用的测试表名，默认没有测试表
	private String testTable; // 测试连接是否可用的测试表名，默认没有测试表
	private boolean isEncrypted; // 数据库密码是否加密
	private boolean isEnabled; // 是否可用
	private int connectType; // 数据库连接类型 0：单机连接，1：集群连接，2：JNDI连接
	private boolean autoInit; // 自动初始化数据库连接池
	private int maxIdleTime; // 最大空闲时间
	private int acquisitionTimeout; // 回收时间
	private boolean shareTransactionConnections;
	private boolean deferConnectionRelease;
	private int initPoolSize; // 初始化连接池个数
	private int minPoolSize; // 最小连接池个数
	private int maxPoolSize; // 最大连接池个数
	private int acquireIncrement; // 连接池自动增加的大小
	private int sslFlag; // SSL通讯
	private String keystore; // 密钥库
	private String keystorePwd; // 密钥库密码
	private String truststore; // 客户机的信任库
	private String truststorePwd; // 信任库密码
	private Date loginTime;
	private Properties connProperties = new Properties(); // 数据库连接属性

	public DBInfo() {
	}

	public DBInfo(String id, String dbType, String url, String user, String password) {
		this.id = id;
		this.dbType = dbType;
		this.url = url;
		this.user = user;
		this.password = password;
	}

	public DBInfo(String id, String dbType, String host, int port, String dbName, String user, String password) {
		this.id = id;
		this.dbType = dbType;
		this.host = host;
		this.port = port;
		this.dbName = dbName;
		this.user = user;
		this.password = password;
	}

	public void initParameter() {
		if (this.parameter == null) {
			this.parameter = "";
		}
		initDBInfo(DBType.getEnumType(dbType));
	}

	protected void initDBInfo(DBType type) {
		switch (type) {
		case MYSQL: // 连接MySql数据库
			driver = "com.mysql.jdbc.Driver";
			if (isEmpty(url)) {
				url = "jdbc:mysql://" + host + ":" + port + "/" + dbName + parameter;
			}
			break;
		case ORACLE: // 连接Oracle数据库
			driver = "oracle.jdbc.OracleDriver";
			if (isEmpty(url)) {
				url = "jdbc:oracle:thin:@" + host + ":" + port + ":" + dbName + parameter;
			}
			break;
		case SQLITE: // 连接SQLITE数据库
			driver = "org.sqlite.JDBC";
			if (isEmpty(url)) {
				url = "jdbc:sqlite:" + dbName + parameter;
			}
			break;
		case DB2: // 连接DB2数据库
			driver = "com.ibm.db2.jcc.DB2Driver";
			if (isEmpty(parameter)) {
				parameter = ":progressiveStreaming=2;fullyMaterializeLobData=false;traceLevel=3;";
			}
			if (isEmpty(url)) {
				url = "jdbc:db2://" + host + ":" + port + "/" + dbName + parameter;
			}
			break;
		case MSSQL: // 连接Sql Server数据库
			driver = "net.sourceforge.jtds.jdbc.Driver";
			if (isEmpty(url)) {
				url = "jdbc:jtds:sqlserver://" + host + ":" + port + ";DatabaseName=" + dbName + parameter;
			}
			break;
		case MSSQL2000: // 连接Sql Server7.0/2000数据库
			driver = "com.microsoft.jdbc.sqlserver.SQLServerDriver";
			if (isEmpty(url)) {
				url = "jdbc:microsoft:sqlserver://" + host + ":" + port + ";DatabaseName=" + dbName + parameter;
			}
			break;
		case MSSQL2005: // 连接Sql Server2005数据库
			driver = "com.microsoft.sqlserver.jdbc.SQLServerDriver";
			if (isEmpty(url)) {
				url = "jdbc:sqlserver://" + host + ":" + port + ";DatabaseName=" + dbName + parameter;
			}
			break;
		case HSQLDB: // 连接hsqldb数据库
			driver = "org.hsqldb.JdbcDriver";
			if (isEmpty(url)) {
				url = "jdbc:" + dbName + ":mem:score" + parameter;
			}
			break;
		case ACCESS: // 直连用ODBC连接access数据库
			driver = "sun.jdbc.odbc.JdbcOdbcDriver";
			if (isEmpty(url)) {
				url = "jdbc:odbc:Driver={MicroSoft Access Driver (*.mdb)};DBQ=" + dbName + parameter;
			}
			break;
		case INFORMIX: // 连接Informix数据库
			driver = "com.informix.jdbc.IfxDriver";
			if (isEmpty(url)) {
				url = "jdbc:informix-sqli://" + host + ":" + port + "/" + dbName + ":INFORMIXSERVER=myserver" + parameter;
			}
			break;
		case POSTGRESQL: // 连接PostgreSQL数据库
			driver = "org.postgresql.Driver";
			if (isEmpty(url)) {
				url = "jdbc:postgresql://" + host + ":" + port + "/" + dbName + parameter;
			}
			url = "jdbc:postgresql://localhost/db_ice";
			break;
		case SYBASE: // 连接Sybase数据库
			driver = "com.sybase.jdbc.SybDriver";
			if (isEmpty(url)) {
				url = "jdbc:sybase:Tds:" + host + ":" + port + "/" + dbName + parameter;
			}
			break;
		default: // 连接MySql数据库
			driver = "com.mysql.jdbc.Driver";
			if (isEmpty(url)) {
				url = "jdbc:mysql://" + host + ":" + port + "/" + dbName + parameter;
			}
			break;
		}
	}

	public Properties getProperties() {
		return connProperties;
	}

	public String getProperty(String key) {
		return connProperties.getProperty(key);
	}

	public void setProperty(String key, String value) {
		connProperties.setProperty(key, value);
	}

	public boolean isEmpty(String text) {
		return text == null || "".equals(text);
	}

	public boolean isNotEmpty(String text) {
		return !isEmpty(text);
	}

	public String getId() {
		return id;
	}

	public void setId(String id) {
		this.id = id;
	}

	public String getDsName() {
		if (dsName == null) {
			return id;
		} else {
			return dsName;
		}
	}

	public void setDsName(String dbSourceName) {
		this.dsName = dbSourceName;
	}

	public String getDbType() {
		return dbType;
	}

	public void setDbType(String dbType) {
		this.dbType = dbType;
	}

	public String getDriver() {
		return driver;
	}

	public String getJndiName() {
		return jndiName;
	}

	public void setJndiName(String jndiName) {
		this.jndiName = jndiName;
	}

	public void setDriver(String driver) {
		this.driver = driver;
	}

	public String getUrl() {
		return url;
	}

	public void setUrl(String url) {
		this.url = url;
	}

	public String getHost() {
		return host;
	}

	public void setHost(String host) {
		this.host = host;
	}

	public int getPort() {
		return port;
	}

	public void setPort(int port) {
		this.port = port;
	}

	public String getDbName() {
		return dbName;
	}

	public void setDbName(String dbName) {
		this.dbName = dbName;
	}

	public String getUser() {
		return user;
	}

	public void setUser(String user) {
		this.user = user;
	}

	public String getPassword() {
		return password;
	}

	public void setPassword(String password) {
		this.password = password;
	}

	public int getLoginTimeout() {
		return loginTimeout;
	}

	public void setLoginTimeout(int loginTimeout) {
		this.loginTimeout = loginTimeout;
	}

	public String getParameter() {
		return parameter;
	}

	public void setParameter(String parameter) {
		this.parameter = parameter;
	}

	public String getTestQuery() {
		return testQuery;
	}

	public void setTestQuery(String testQuery) {
		this.testQuery = testQuery;
	}

	public String getTestTable() {
		return testTable;
	}

	public void setTestTable(String testTable) {
		this.testTable = testTable;
	}

	public boolean isEncrypted() {
		return isEncrypted;
	}

	public void setEncrypted(boolean isEncrypted) {
		this.isEncrypted = isEncrypted;
	}

	public boolean isEnabled() {
		return isEnabled;
	}

	public void setEnabled(boolean isEnabled) {
		this.isEnabled = isEnabled;
	}

	public int getConnectType() {
		return connectType;
	}

	public void setConnectType(int connectType) {
		this.connectType = connectType;
	}

	public boolean isAutoInit() {
		return autoInit;
	}

	public void setAutoInit(boolean autoInit) {
		this.autoInit = autoInit;
	}

	public int getMaxIdleTime() {
		return maxIdleTime;
	}

	public void setMaxIdleTime(int maxIdleTime) {
		this.maxIdleTime = maxIdleTime;
	}

	public int getAcquisitionTimeout() {
		return acquisitionTimeout;
	}

	public void setAcquisitionTimeout(int acquisitionTimeout) {
		this.acquisitionTimeout = acquisitionTimeout;
	}

	public boolean isShareTransactionConnections() {
		return shareTransactionConnections;
	}

	public void setShareTransactionConnections(boolean shareTransactionConnections) {
		this.shareTransactionConnections = shareTransactionConnections;
	}

	public boolean isDeferConnectionRelease() {
		return deferConnectionRelease;
	}

	public void setDeferConnectionRelease(boolean deferConnectionRelease) {
		this.deferConnectionRelease = deferConnectionRelease;
	}

	public int getInitPoolSize() {
		return initPoolSize;
	}

	public void setInitPoolSize(int initPoolSize) {
		this.initPoolSize = initPoolSize;
	}

	public int getMinPoolSize() {
		return minPoolSize;
	}

	public void setMinPoolSize(int minPoolSize) {
		this.minPoolSize = minPoolSize;
	}

	public int getMaxPoolSize() {
		return maxPoolSize;
	}

	public void setMaxPoolSize(int maxPoolSize) {
		this.maxPoolSize = maxPoolSize;
	}

	public int getAcquireIncrement() {
		return acquireIncrement;
	}

	public void setAcquireIncrement(int acquireIncrement) {
		this.acquireIncrement = acquireIncrement;
	}

	public int getSslFlag() {
		return sslFlag;
	}

	public boolean isSsl() {
		return sslFlag == 1 ? true : false;
	}

	public void setSslFlag(int sslFlag) {
		this.sslFlag = sslFlag;
	}

	public String getKeystore() {
		return keystore;
	}

	public void setKeystore(String keystore) {
		this.keystore = keystore;
	}

	public String getKeystorePwd() {
		return keystorePwd;
	}

	public void setKeystorePwd(String keystorePwd) {
		this.keystorePwd = keystorePwd;
	}

	public String getTruststore() {
		return truststore;
	}

	public void setTruststore(String truststore) {
		this.truststore = truststore;
	}

	public String getTruststorePwd() {
		return truststorePwd;
	}

	public void setTruststorePwd(String truststorePwd) {
		this.truststorePwd = truststorePwd;
	}

	public Date getLoginTime() {
		return loginTime;
	}

	public void setLoginTime(Date loginTime) {
		this.loginTime = loginTime;
	}

	@Override
	public DBInfo clone() {
		try {
			return (DBInfo) super.clone();
		} catch (CloneNotSupportedException e) {
			e.printStackTrace();
			return null;
		}
	}

	@Override
	public boolean equals(Object obj) {
		if (obj instanceof DBInfo) {
			DBInfo dbInfo = (DBInfo) obj;
			if (this.url != null && this.url.equalsIgnoreCase(dbInfo.getUrl())) {
				return true;
			} else if (this.dbType.equalsIgnoreCase(dbInfo.getDbType()) && this.host.equalsIgnoreCase(dbInfo.getHost())
					&& this.port == dbInfo.getPort() && this.dbName.equalsIgnoreCase(dbInfo.getDbName())) {
				return true;
			}
		}
		return false;
	}

	@Override
	public int hashCode() {
		if (url != null && url.length() > 0) {
			return url.hashCode();
		} else {
			return (dbType + host + port + dbName).hashCode();
		}
	}

}