package cn.elwy.common.jdbc;

import java.sql.Connection;
import java.sql.SQLException;

import cn.elwy.common.exception.DaoException;
import cn.elwy.common.i18n.Msg;
import cn.elwy.common.log.Logger;
import cn.elwy.common.log.LoggerFactory;
import cn.elwy.common.util.CloseUtil;

/**
 * @author huangsq
 * @version 1.0, 2018-02-19
 */
public class CurrentConnection {

	private static Logger logger = LoggerFactory.getLogger(CurrentConnection.class);

	private String key;
	/** 当前连接 */
	private Connection conn;
	/** 当前是否处于事务环境 */
	private boolean existsTransaction;
	/** 是否必须回滚 */
	private boolean rollbackOnly;
	/** 是否自动提交 */
	private boolean autoCommit = true;

	public CurrentConnection(String key, Connection conn) {
		this.key = key;
		this.conn = conn;
	}

	public void beginTransaction() {
		if (!existsTransaction) {
			logger.debug("===============开启一个新事务===============");
			setAutoCommit(false);
			setExistsTransaction(true);
		} else {
			logger.debug("===============已存在事务，加入当前事务===============");
		}
		logger.debug(conn.toString());
	}

	public void commit() {
		try {
			try {
				if (!conn.getAutoCommit()) {
					conn.commit();
				}
			} catch (SQLException e) {
				logger.error(Msg.E_CONN_COMMIT, e);
				throw new DaoException(Msg.E_CONN_COMMIT, e);
			} finally {
				setExistsTransaction(false);
				setAutoCommit(true);
			}
			logger.debug("===============事务已经提交===============");
		} catch (DaoException e) {
			throw e;
		} catch (Exception e) {
			throw new DaoException(Msg.E_CONN_COMMIT, e);
		} finally {
			CloseUtil.close(conn);
		}
	}

	/** 回滚事务 */
	public void rollback() {
		try {
			try {
				if (!conn.getAutoCommit()) {
					conn.rollback();
				}
			} catch (SQLException e) {
				logger.error(Msg.E_CONN_ROLLBACK, e);
				throw new DaoException(Msg.E_CONN_ROLLBACK, e);
			} finally {
				setExistsTransaction(false);
				setAutoCommit(autoCommit);
			}
			logger.debug("===============事务已经回滚===============");
		} catch (DaoException e) {
			throw e;
		} catch (Exception e) {
			throw new DaoException(Msg.E_CONN_ROLLBACK, e);
		} finally {
			CloseUtil.close(conn);
		}
	}

	/** 设置事务提交状态 */
	public void setAutoCommit(boolean flag) {
		try {
			if (conn.getAutoCommit() == flag) {
				return;
			}
			autoCommit = conn.getAutoCommit();
			conn.setAutoCommit(flag);
		} catch (SQLException e) {
			throw new DaoException(Msg.E_CONN_AUTO_COMMIT, e);
		}
	}

	public String getKey() {
		return key;
	}

	public void setKey(String key) {
		this.key = key;
	}

	public Connection getConn() {
		return conn;
	}

	public void setConn(Connection conn) {
		this.conn = conn;
	}

	public boolean isExistsTransaction() {
		return existsTransaction;
	}

	public void setExistsTransaction(boolean existsTransaction) {
		this.existsTransaction = existsTransaction;
	}

	public boolean isRollbackOnly() {
		return rollbackOnly;
	}

	public void setRollbackOnly(boolean rollbackOnly) {
		this.rollbackOnly = rollbackOnly;
	}

	public boolean isAutoCommit() {
		return autoCommit;
	}

	public boolean close() {
		if (!existsTransaction) {
			CloseUtil.close(conn);
			return true;
		}
		return false;
	}
}