package cn.elwy.common.id;

import java.util.Random;
import java.util.UUID;
import java.util.concurrent.atomic.AtomicInteger;

import cn.elwy.common.util.AssertUtil;
import cn.elwy.common.util.NumberUtil;

/**
 * ID生成工具类,提供uuid、根据时间+服务器标识+随机数+自增生成ID两种方式。
 * @author huangsq
 * @version 1.0, 2018-02-19
 */
public final class IdUtil {

	private static int MIN_VALUE = 100000000;
	private static int MAX_VALUE = 999999999;

	// 顺序号计数器
	private static AtomicInteger globalCount = new AtomicInteger(MIN_VALUE);
	private static int localId;

	static {
		setServerId("localhost");
	}

	/**
	 * 当前时间(7~11位62进制，20810805前7位，88881201前8位) + 服务器ID(3位62进制IP随机值) +
	 * 随机数(3位62进制，23万个随机值) + 自增序列(共7位：由9位10进制自增序列组成，前5位采用62进制进行编码长度为3位，
	 * 后4位采用10进制，最多可拆分8192个数据库) ID总长度不超过25位。不同服务器每毫秒支持9亿次并发，理论上唯一。
	 * @author huangsq
	 * @param currentTimeMillis 当前时间戳
	 * @return 自定义ID
	 */
	public static String nextId() {
		return nextId(System.currentTimeMillis());
	}

	/**
	 * 当前时间(7~11位62进制，20810805前7位，88881201前8位) + 服务器ID(3位62进制IP随机值) +
	 * 随机数(3位62进制，23万个随机值) + 自增序列(共7位：由9位10进制自增序列组成，前5位采用62进制进行编码长度为3位，
	 * 后4位采用10进制，最多可拆分8192个数据库) ID总长度不超过25位。不同服务器每毫秒支持9亿次并发，理论上唯一。
	 * @author huangsq
	 * @param currentTimeMillis 当前时间戳
	 * @return 自定义ID
	 */
	public static String nextId(long currentTimeMillis) {
		StringBuilder sb = new StringBuilder();
		sb.append(NumberUtil.toString(currentTimeMillis));
		sb.append(NumberUtil.toString(localId));
		Random random = new Random();
		sb.append(NumberUtil.toString(getRandomInt(random, 3)));
		Integer nextId = getNextId();
		sb.append(NumberUtil.toString(nextId / 10000));
		sb.append(nextId.toString().substring(5));
		return sb.toString();
	}

	/**
	 * 以62进制(字母加数字)生成19位UUID，最短的UUID
	 * @return
	 */
	public static String uuid() {
		UUID uuid = UUID.randomUUID();
		StringBuilder sb = new StringBuilder();
		sb.append(digits(uuid.getMostSignificantBits() >> 32, 8));
		sb.append(digits(uuid.getMostSignificantBits() >> 16, 4));
		sb.append(digits(uuid.getMostSignificantBits(), 4));
		sb.append(digits(uuid.getLeastSignificantBits() >> 48, 4));
		sb.append(digits(uuid.getLeastSignificantBits(), 12));
		return sb.toString();
	}

	public static void setServerId(String serverId) {
		if (AssertUtil.isNotEmpty(serverId)) {
			int hashCode = serverId.hashCode();
			Random random = new Random(hashCode);
			IdUtil.localId = getRandomInt(random, 3);
		}
	}

	private static int getNextId() {
		int id = globalCount.getAndIncrement();
		if (id > MAX_VALUE) {
			id = MIN_VALUE;
			globalCount.getAndSet(id);
		}
		return id;
	}

	private static int getRandomInt(Random random, int length) {
		if (length <= 3) {
			return random.nextInt(234484) + 3844;
		} else if (length == 4) {
			return random.nextInt(14538008) + 238328;
		} else {
			return random.nextInt(901356496) + 14776336;
		}
	}

	private static String digits(long val, int digits) {
		long hi = 1L << (digits * 4);
		return NumberUtil.toString(hi | (val & (hi - 1)), NumberUtil.MAX_RADIX).substring(1);
	}

}
