/*******************************************************************************
 * Copyright (c) 2005, 2010 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * Contributors:
 * IBM - Initial API and implementation
 *******************************************************************************/
package cn.elwy.common.i18n;

import java.io.IOException;
import java.io.InputStream;
import java.lang.reflect.Field;
import java.lang.reflect.Modifier;
import java.security.AccessController;
import java.security.PrivilegedAction;
import java.text.MessageFormat;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Properties;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import cn.elwy.common.log.Logger;
import cn.elwy.common.log.LoggerFactory;

/**
 * @author huangsq
 * @version 1.0, 2018-02-19
 */
public abstract class NLS {

	private static final String ARGS_PLACEHOLDER = "\\{\\}";
	private static final String EXTENSION = ".properties"; //$NON-NLS-1$
	private static String[] nlSuffixes;
	// private static final String PROP_WARNINGS = "osgi.nls.warnings";
	// //$NON-NLS-1$
	// private static final String IGNORE = "ignore"; //$NON-NLS-1$
	private static final boolean ignoreWarnings = true;// IGNORE.equals(FrameworkProperties.getProperty(PROP_WARNINGS));

	/*
	 * NOTE do not change the name of this field; it is set by the Framework using
	 * reflection
	 */
	private static Logger logger = LoggerFactory.getLogger(NLS.class);

	static final int SEVERITY_ERROR = 0x04;
	static final int SEVERITY_WARNING = 0x02;
	/*
	 * This object is assigned to the value of a field map to indicate that a
	 * translated message has already been assigned to that field.
	 */
	static final Object ASSIGNED = new Object();

	/**
	 * Creates a new NLS instance.
	 */
	protected NLS() {
		super();
	}

	/**
	 * Initialize the given class with the values from the message properties
	 * specified by the base name. The base name specifies a fully qualified base
	 * name to a message properties file, including the package where the message
	 * properties file is located. The class loader of the specified class will be
	 * used to load the message properties resources.
	 * <p>
	 * For example, if the locale is set to en_US and
	 * <code>org.eclipse.example.nls.messages</code> is used as the base name then
	 * the following resources will be searched using the class loader of the
	 * specified class:
	 * 
	 * <pre>
	 *   org/eclipse/example/nls/messages_en_US.properties
	 *   org/eclipse/example/nls/messages_en.properties
	 *   org/eclipse/example/nls/messages.properties
	 * </pre>
	 * </p>
	 * @param baseName the base name of a fully qualified message properties file.
	 * @param clazz the class where the constants will exist
	 */
	public static Map<String, String> initializeMessages(final String baseName, final Class<?> clazz) {
		if (System.getSecurityManager() == null) {
			return load(baseName, clazz);
		}
		final HashMap<String, String> hashMap = new HashMap<String, String>();
		AccessController.doPrivileged(new PrivilegedAction<Object>() {
			public Object run() {
				hashMap.putAll(load(baseName, clazz));
				return null;
			}
		});
		return hashMap;
	}

	@Deprecated
	public static String bind(String msg, Object... args) {
		return format(msg, args);
	}

	public static String format(String msg, Object... args) {
		if (args != null) {
			Pattern p = Pattern.compile(ARGS_PLACEHOLDER);
			Matcher m = p.matcher(msg);
			int i = 0;
			while (m.find()) {
				msg = msg.replaceFirst(ARGS_PLACEHOLDER, "{" + i + "}");
				i++;
			}
		}
		return MessageFormat.format(msg, args);
	}

	/*
	 * Build an array of property files to search. The returned array contains the
	 * property fields in order from most specific to most generic. So, in the FR_fr
	 * locale, it will return file_fr_FR.properties, then file_fr.properties, and
	 * finally file.properties.
	 */
	private static String[] buildVariants(String root) {
		if (nlSuffixes == null) {
			// build list of suffixes for loading resource bundles
			String nl = Locale.getDefault().toString();
			List<String> result = new ArrayList<String>(4);
			int lastSeparator;
			while (true) {
				result.add('_' + nl + EXTENSION);
				lastSeparator = nl.lastIndexOf('_');
				if (lastSeparator == -1)
					break;
				nl = nl.substring(0, lastSeparator);
			}
			// add the empty suffix last (most general)
			result.add(EXTENSION);
			nlSuffixes = result.toArray(new String[result.size()]);
		}
		root = root.replace('.', '/');
		String[] variants = new String[nlSuffixes.length];
		for (int i = 0; i < variants.length; i++)
			variants[i] = root + nlSuffixes[i];
		return variants;
	}

	private static void computeMissingMessages(String bundleName, Class<?> clazz, Map<Object, Object> fieldMap,
			Field[] fieldArray, boolean isAccessible) {
		// iterate over the fields in the class to make sure that there aren't any empty
		// ones
		final int MOD_EXPECTED = Modifier.PUBLIC | Modifier.STATIC;
		final int MOD_MASK = MOD_EXPECTED | Modifier.FINAL;
		final int numFields = fieldArray.length;
		for (int i = 0; i < numFields; i++) {
			Field field = fieldArray[i];
			if ((field.getModifiers() & MOD_MASK) != MOD_EXPECTED)
				continue;
			// if the field has a a value assigned, there is nothing to do
			if (fieldMap.get(field.getName()) == ASSIGNED)
				continue;
			try {
				// Set a value for this empty field. We should never get an exception here
				// because
				// we know we have a public static non-final field. If we do get an exception,
				// silently
				// log it and continue. This means that the field will (most likely) be
				// un-initialized and
				// will fail later in the code and if so then we will see both the NPE and this
				// error.
				String value = "NLS missing message: " + field.getName() + " in: " + bundleName; //$NON-NLS-1$ //$NON-NLS-2$
				if (logger.isDebugEnabled()) {
					logger.debug(value);
				}
				log(SEVERITY_WARNING, value, null);
				if (!isAccessible)
					field.setAccessible(true);
				field.set(null, value);
			} catch (Exception e) {
				log(SEVERITY_ERROR, "Error setting the missing message value for: " + field.getName(), e); //$NON-NLS-1$
			}
		}
	}

	/*
	 * Load the given resource bundle using the specified class loader.
	 */
	static Map<String, String> load(final String bundleName, Class<?> clazz) {
		long start = System.currentTimeMillis();
		final Field[] fieldArray = clazz.getDeclaredFields();
		ClassLoader loader = clazz.getClassLoader();

		boolean isAccessible = (clazz.getModifiers() & Modifier.PUBLIC) != 0;

		// build a msgMaps of field names to Field objects
		final int len = fieldArray.length;
		Map<Object, Object> fields = new HashMap<Object, Object>(len * 2);
		for (int i = 0; i < len; i++)
			fields.put(fieldArray[i].getName(), fieldArray[i]);

		// search the variants from most specific to most general, since
		// the MessagesProperties.put method will mark assigned fields
		// to prevent them from being assigned twice
		final String[] variants = buildVariants(bundleName);
		Map<String, String> msgMap = new HashMap<String, String>(len * 2);
		for (int i = 0; i < variants.length; i++) {
			// loader==null if we're launched off the Java boot classpath
			final InputStream input = loader == null ? ClassLoader.getSystemResourceAsStream(variants[i])
					: loader.getResourceAsStream(variants[i]);
			if (input == null)
				continue;
			try {
				final MessagesProperties properties = new MessagesProperties(fields, bundleName, isAccessible, msgMap);
				properties.load(input);
			} catch (IOException e) {
				log(SEVERITY_ERROR, "Error loading " + variants[i], e); //$NON-NLS-1$
			} finally {
				if (input != null)
					try {
						input.close();
					} catch (IOException e) {
						// ignore
					}
			}
		}
		computeMissingMessages(bundleName, clazz, fields, fieldArray, isAccessible);
		if (logger.isDebugEnabled()) {
			logger
					.debug("Time to load message bundle: " + bundleName + " was " + (System.currentTimeMillis() - start) + "ms.");
		}
		return msgMap;
	}

	/*
	 * The method adds a log entry based on the error message and exception. The
	 * output is written to the System.err. This method is only expected to be
	 * called if there is a problem in the NLS mechanism. As a msgMaps, translation
	 * facility is not available here and messages coming out of this log are
	 * generally not translated.
	 * @param severity - severity of the message (SEVERITY_ERROR or
	 * SEVERITY_WARNING)
	 * @param message - message to log
	 * @param e - exception to log
	 */
	static void log(int severity, String message, Exception e) {
		if (severity == SEVERITY_WARNING && ignoreWarnings) {
			return;
		} else if (severity == SEVERITY_ERROR) {
			if (e == null) {
				logger.error(message);
			} else {
				logger.error(message, e);
			}
		} else if (e == null) {
			logger.warn(message);
		} else {
			logger.warn(message, e);
		}
	}

	/*
	 * Class which sub-classes java.util.Properties and uses the #put method to set
	 * field values rather than storing the values in the table.
	 */
	private static class MessagesProperties extends Properties {

		private static final int MOD_EXPECTED = Modifier.PUBLIC | Modifier.STATIC;
		private static final int MOD_MASK = MOD_EXPECTED | Modifier.FINAL;
		private static final long serialVersionUID = 1L;

		private final String bundleName;
		private final Map<Object, Object> fields;
		private final Map<String, String> msgMap;
		private final boolean isAccessible;

		public MessagesProperties(Map<Object, Object> fieldMap, String bundleName, boolean isAccessible,
				Map<String, String> msgMap) {
			super();
			this.fields = fieldMap;
			this.msgMap = msgMap;
			this.bundleName = bundleName;
			this.isAccessible = isAccessible;
		}

		/*
		 * (non-Javadoc)
		 * @see java.util.Hashtable#put(java.lang.Object, java.lang.Object)
		 */
		public synchronized Object put(Object key, Object value) {
			msgMap.put((String) key, (String) value);
			Object fieldObject = fields.put(key, ASSIGNED);
			// if already assigned, there is nothing to do
			if (fieldObject == ASSIGNED)
				return null;
			if (fieldObject == null) {
				final String msg = "NLS unused message: " + key + " in: " + bundleName;//$NON-NLS-1$ //$NON-NLS-2$
				if (logger.isDebugEnabled()) {
					logger.debug(msg);
				}
				log(SEVERITY_WARNING, msg, null);
				return null;
			}
			final Field field = (Field) fieldObject;
			// can only set value of public static non-final fields
			if ((field.getModifiers() & MOD_MASK) != MOD_EXPECTED)
				return null;
			try {
				// Check to see if we are allowed to modify the field. If we aren't (for
				// instance
				// if the class is not public) then change the accessible attribute of the field
				// before trying to set the value.
				if (!isAccessible)
					field.setAccessible(true);
				// Set the value into the field. We should never get an exception here because
				// we know we have a public static non-final field. If we do get an exception,
				// silently
				// log it and continue. This means that the field will (most likely) be
				// un-initialized and
				// will fail later in the code and if so then we will see both the NPE and this
				// error.

				// Extra care is taken to be sure we create a String with its own backing char[]
				// (bug 287183)
				// This is to ensure we do not keep the key chars in memory.
				field.set(null, new String(((String) value).toCharArray()));
			} catch (Exception e) {
				log(SEVERITY_ERROR, "Exception setting field value.", e); //$NON-NLS-1$
			}
			return null;
		}
	}
}