package cn.elwy.common;

import java.io.File;
import java.net.MalformedURLException;
import java.net.URL;
import java.net.URLDecoder;

import cn.elwy.common.exception.RunException;
import cn.elwy.common.i18n.Msg;
import cn.elwy.common.log.Logger;
import cn.elwy.common.log.LoggerFactory;

/**
 * 获取路径工具类，不支持获取OSGI工程路径
 * @author huangsq
 * @version 1.0, 2018-02-19
 */
public class PathSupport implements Constant {

	private static Logger logger = LoggerFactory.getLogger(PathSupport.class);

	public static String ABSOLUTE_PATH_FLAG = "///";
	public static String BACKSLASH = "[\\\\]{1,}";
	public static String SLASH = "/";

	private static class LazyHolder {
		private static final PathSupport INSTANCE = new PathSupport();
	}

	/** User Dir Floder */
	protected File userDir;
	/** User Home Floder */
	protected File userHome;
	/** 安装路径 */
	protected URL installUrl;
	/** 安装路径 */
	protected File installHome;
	/** 安装路径 */
	protected String installPath;

	/** 应用路径 */
	protected File appHome;
	protected String appHomePath;
	protected URL appHomeUrl;

	/** 全局配置路径 */
	protected File globalHome;
	protected String globalHomePath;
	protected URL globalHomeUrl;

	/** 日志路径 */
	protected File logHome;
	/** 测试路径 */
	protected File testDir;

	public PathSupport() {
		init();
	}

	public static PathSupport getInstance() {
		return LazyHolder.INSTANCE;
	}

	/** 获取系统运行目录 */
	public File getUserDir() {
		if (userDir == null) {
			userDir = new File(System.getProperty("user.dir"));
		}
		return userDir;
	}

	/** 获取用户根目录 */
	public File getUserHome() {
		if (userHome == null) {
			userHome = new File(System.getProperty("user.home"));
		}
		return userHome;
	}

	/** 初始化路径 */
	public void init() {
		setInstallHome(getUserDir());
		setAppHome(new File(getInstallHome(), APP_HOME_DIR));
		setGlobalHome(new File(getAppHome(), GLOBAL_DIR));
		setLogHome(new File(getInstallHome(), LOG_DIR));
	}

	/** 获取安装目录 */
	public File getInstallHome() {
		return installHome;
	}

	/** 获取安装路径 */
	public URL getInstallURL() {
		return installUrl;
	}

	/** 获取安装目录 */
	public String getInstallPath() {
		return installPath;
	}

	/** 设置安装目录 */
	protected synchronized void setInstallHome(File file) {
		if (file == null) {
			logger.warn(Msg.format(Msg.E_IS_NULL, "设置installHome"));
			return;
		}
		try {
			installHome = file;
			installPath = getFormatPath(file);
			installUrl = file.toURI().toURL();
			// 设置应用安装路径到系统属性
			System.setProperty(INSTALL_HOME, installPath);
		} catch (MalformedURLException e) {
			throw new RunException(e);
		}
	}

	public File getAppHome() {
		return appHome;
	}

	public String getAppHomePath() {
		return appHomePath;
	}

	public URL getAppHomeUrl() {
		return appHomeUrl;
	}

	public synchronized void setAppHome(File file) {
		if (file == null) {
			logger.warn(Msg.format(Msg.E_IS_NULL, "设置appHome"));
			return;
		}
		try {
			appHome = file;
			appHomePath = getFormatPath(file);
			appHomeUrl = file.toURI().toURL();
			System.setProperty(APP_HOME, appHomePath);
			logger.info("APP_HOME: " + appHome);
		} catch (MalformedURLException e) {
			throw new RunException(e);
		}
	}

	public File getGlobalHome() {
		return globalHome;
	}

	public String getGlobalHomePath() {
		return globalHomePath;
	}

	public URL getGlobalHomeUrl() {
		return globalHomeUrl;
	}

	public synchronized void setGlobalHome(File file) {
		if (file == null) {
			logger.warn(Msg.format(Msg.E_IS_NULL, "设置globalHome"));
			return;
		}
		try {
			globalHome = file;
			globalHomePath = getFormatPath(file);
			globalHomeUrl = file.toURI().toURL();
			System.setProperty(GLOBAL_HOME, globalHomePath);
			logger.info("GLOBAL_HOME: " + globalHome);
		} catch (MalformedURLException e) {
			throw new RunException(e);
		}
	}

	public File getLogHome() {
		if (logHome == null) {
			this.setLogHome(new File(getAppHome(), LOG_DIR));
		}
		return logHome;
	}

	public synchronized void setLogHome(File file) {
		if (file == null || file.isFile()) {
			logger.warn(Msg.format(Msg.E_PATH_INVALID, "设置logHome：" + file));
			return;
		}
		logHome = file;
		System.setProperty(LOG_HOME, file.getAbsolutePath());
		logger.info("LOG_HOME: " + logHome);
	}

	/** 获取测试目录 */
	public File getTestDir() {
		if (testDir == null) {
			testDir = new File(installHome, "test");
		}
		return testDir;
	}

	/** 获取测试目录 */
	public synchronized void setTestDir(File testDir) {
		this.testDir = testDir;
	}

	/**
	 * 获取相对于URL的资源
	 * @param url 查找路径
	 * @param filePath 文件路径
	 * @return
	 */
	public URL getResource(URL url, String filePath) {
		if (url == null) {
			return null;
		}
		filePath = getFilePath(filePath);
		try {
			URL resource = new URL(url, filePath);
			File file = toFile(resource);
			if (file != null && file.exists()) {
				return resource;
			}
		} catch (Exception e) {
			throw new RunException(e);
		}
		return null;
	}

	/**
	 * 获取相对于dir的资源
	 * @param dir 查找路径
	 * @param filePath 文件路径
	 * @return
	 */
	public File getResource(File dir, String filePath) {
		File file = new File(dir, filePath);
		if (file != null && file.exists()) {
			return file;
		}
		return null;
	}

	/**
	 * 获取相对于URL的资源
	 * @param url 查找路径
	 * @param filePath 文件路径
	 * @return
	 */
	public File getResourceFile(URL url, String filePath) {
		return toFile(getResource(url, filePath));
	}

	/**
	 * 获取资源路径，按顺序查找appHome,globalHomePath,installHome找到资源直接返回
	 * @param filePath 文件路径
	 * @return
	 */
	public URL getResources(String filePath) {
		return getResources(null, filePath);
	}

	/**
	 * 获取资源路径，先按顺序查找appHome,globalHomePath,installHome找到资源直接返回，否则查找相对于classPath的资源
	 * @param clazz 类对象的class
	 * @param filePath 文件路径
	 * @return
	 */
	public URL getResources(Class<?> clazz, String filePath) {
		URL url = getResource(appHomeUrl, filePath);
		if (url == null) {
			if (!appHomePath.equalsIgnoreCase(globalHomePath)) {
				url = getResource(globalHomeUrl, filePath);
			}
			if (url == null) {
				url = getResource(installUrl, filePath);
			}
		}
		if (url == null && clazz != null) {
			url = getResource(clazz, filePath);
		}
		return url;
	}

	/**
	 * 获取资源路径，先按顺序查找appHome,installHome找到资源直接返回
	 * @param filePath 文件路径
	 * @return
	 */
	public String getResourcesPath(String filePath) {
		return toFilePath(getResources(null, filePath));
	}

	/**
	 * 获取资源路径，先按顺序查找appHome,installHome找到资源直接返回，否则查找相对于classPath的资源
	 * @param clazz 类对象的class
	 * @param filePath 文件路径
	 * @return
	 */
	public String getResourcesPath(Class<?> clazz, String filePath) {
		return toFilePath(getResources(clazz, filePath));
	}

	/**
	 * 获取资源路径，先按顺序查找appHome,installHome找到资源直接返回
	 * @param filePath 文件路径
	 * @return
	 */
	public File getResourcesFile(String filePath) {
		return toFile(getResources(null, filePath));
	}

	/**
	 * 获取资源路径，先按顺序查找appHome,installHome找到资源直接返回，否则查找相对于classPath的资源
	 * @param clazz 类对象的class
	 * @param filePath 文件路径
	 * @return
	 */
	public File getResourcesFile(Class<?> clazz, String filePath) {
		return toFile(getResources(clazz, filePath));
	}

	/**
	 * 获取资源路径，相对于classPath 注意：不可以将URL转换成file读取文件
	 * @param clazz 类对象的class
	 * @param filePath 文件路径
	 * @return
	 */
	public URL getResource(Class<?> clazz, String filePath) {
		if (clazz == null) {
			return null;
		}
		URL url = clazz.getResource(filePath);
		if (url == null) {
			if (clazz.getClassLoader() != null) {
				url = clazz.getClassLoader().getResource(filePath);
			}
		}
		return url;
	}

	/**
	 * 获取资源路径，相对于classPath 注意：不可用于获取jar包中的资源
	 * @param clazz 类对象的class
	 * @param filePath 文件路径
	 * @return
	 */
	public String getResourcePath(Class<?> clazz, String filePath) {
		return toFilePath(getResource(clazz, filePath));
	}

	public File toFile(URL url) {
		if (url == null) {
			return null;
		}
		try {
			String filePath = URLDecoder.decode(url.getFile(), "UTF-8");
			return new File(filePath);
		} catch (Exception e) {
			String path = url.getFile();
			path = path.replace("jar:", "").replaceAll("file:/", "");
			logger.warn(e.getMessage(), e);
			return new File(path);
		}
	}

	public String toFilePath(URL url) {
		File file = toFile(url);
		return toFilePath(file, false);
	}

	public String toFilePath(File file) {
		return toFilePath(file, false);
	}

	public String toFilePath(File file, boolean isDirectory) {
		String filePath = null;
		if (file != null) {
			if (file.isDirectory() || isDirectory) {
				filePath = file.getAbsoluteFile() + File.separator;
			} else {
				filePath = file.getAbsolutePath();
			}
			return filePath.replace("\\", "/");
		}
		return filePath;
	}

	protected String getFilePath(String filePath) {
		if (filePath != null && filePath.length() > 0) {
			if (filePath.startsWith("/")) {
				filePath = filePath.substring(1);
			}
		}
		return filePath;
	}

	/**
	 * 格式化路径，以"/"分割返回路径
	 * @param filePath 路径
	 * @return
	 */
	public static String getFormatPath(String filePath) {
		if (filePath != null && filePath.length() > 0) {
			filePath = filePath.replaceAll(BACKSLASH, SLASH);
		} else if (filePath == null) {
			filePath = "";
		}
		return filePath;
	}

	/**
	 * 格式化路径，以"/"分割返回路径
	 * @param filePath 路径
	 * @return
	 */
	public static String getFormatPath(File file) {
		if (file != null) {
			String filePath = file.getAbsolutePath().replaceAll(BACKSLASH, SLASH);
			if (file.isDirectory()) {
				filePath += "/";
			}
			return filePath;
		}
		return null;
	}

	/**
	 * 格式化路径，以"/"分割返回路径
	 * @param filePath 路径
	 * @return
	 */
	public static String replaceFilePath(String filePath, String replacePath) {
		filePath = getFormatPath(filePath);
		replacePath = getFormatPath(replacePath);

		if (filePath != null && replacePath != null) {
			filePath = filePath.replace(replacePath, "");
		}
		return filePath;
	}

	/**
	 * 将两个路径以"/"分割拼接成完整的路径
	 * @param root 根路径
	 * @param filePath 相对于根的路径
	 * @return 拼接后的路径
	 */
	public static String getFilePath(String root, String filePath) {
		return mergeFilePath(getFormatPath(root), getFormatPath(filePath), SLASH);
	}

	/**
	 * 获取文件的绝对路径，如果filePath以"///"开头，直接返回绝对路径。否则将两个路径以"/"分割拼接成完整的路径
	 * @param root 根路径
	 * @param filePath 相对于根的路径
	 * @return 拼接后的路径
	 */
	public static String getAbsolutePath(String root, String filePath) {
		if (filePath.startsWith(ABSOLUTE_PATH_FLAG)) {
			return filePath.replace(ABSOLUTE_PATH_FLAG, "");
		}
		return mergeFilePath(getFormatPath(root), getFormatPath(filePath), SLASH);
	}

	/**
	 * 将多个路径以"/"分割拼接成完整的路径
	 * @param root 根路径
	 * @param filePaths 相对于根的路径
	 * @return 拼接后的路径
	 */
	public static String getFilePath(String root, String... filePaths) {
		String result = getFormatPath(root);
		if (filePaths != null) {
			for (int i = 0; i < filePaths.length; i++) {
				result = mergeFilePath(result, getFormatPath(filePaths[i]), SLASH);
			}
		}
		return result;
	}

	/**
	 * 将两个路径以分割符拼接成完整的路径
	 * @param root 根路径
	 * @param filePath 相对于根的路径
	 * @param separator 分割符
	 * @return 拼接后的路径
	 */
	public static String mergeFilePath(String root, String filePath, String separator) {
		if ("".equals(filePath)) {
			return root;
		}
		if (root.endsWith(separator)) {
			if (filePath.startsWith(separator)) {
				return root + filePath.substring(separator.length());
			} else {
				return root + filePath;
			}
		} else {
			if (filePath.startsWith(separator)) {
				return root + filePath;
			} else {
				return root + separator + filePath;
			}
		}
	}

	/**
	 * 获取相对路径，返回的路径不以"/"开头
	 * @param filePath
	 * @return
	 */
	public static String getRelativePath(String filePath) {
		if (filePath != null && filePath.length() > 0) {
			filePath = getFormatPath(filePath);
			if (filePath.startsWith(SLASH)) {
				filePath = filePath.substring(SLASH.length());
			}
		}
		return filePath;
	}

	/**
	 * 获取相对路径，返回相对语basePath的路径
	 * @param basePath
	 * @param filePath
	 * @return
	 */
	public static String getRelativePath(String basePath, String filePath) {
		if (filePath != null && filePath.length() > 0) {
			basePath = getFormatPath(basePath);
			filePath = getFormatPath(filePath);
			return filePath.replace(basePath, "");
		}
		return filePath;
	}

}
