package cn.easyutil.project.service.easySqlExecuter;

import cn.easyutil.project.base.bean.Page;
import cn.easyutil.project.service.easySqlExecuter.config.TransactionUtil;
import cn.easyutil.project.base.sqlExecuter.SQLExecuter;
import cn.easyutil.project.base.sqlExecuter.annotation.TableField;
import cn.easyutil.project.base.sqlExecuter.annotation.TableId;
import cn.easyutil.project.base.sqlExecuter.annotation.TableIgnore;
import cn.easyutil.project.base.sqlExecuter.annotation.TableName;
import cn.easyutil.util.javaUtil.LoggerUtil;
import cn.easyutil.util.javaUtil.StringUtil;

import javax.sql.DataSource;
import java.lang.reflect.Field;
import java.sql.*;
import java.util.*;
import java.util.Map.Entry;


/**
 * easySql的具体执行类 内置基础增删改查和分页,事务</br>
 * 在使用之前请务必了解一下内容:</br>
 * 一:本工具提供了以下4个注解</br>
 * &nbsp;&nbsp;&nbsp;1:TableName(name="") 在你需要操作的bean类上添加，用来标注数据库表名,此为必填项</br>
 * &nbsp;&nbsp;&nbsp;2:TableId 在你需要操作的bean类属性上添加，用来标注数据库表主键名称，此为必填项</br>
 * &nbsp;&nbsp;&nbsp;3:TableField(name="")
 * 在你需要操作的bean类属性上添加，用来标注数据库表字段,如一致，则非必填</br>
 * &nbsp;&nbsp;&nbsp;4:TableIgnore
 * 在你需要操作的bean类属性上添加，用来标注数据库表中没有的字段，新增时不会添加进表</br>
 * 二:如果你需要操作的类属性有非基本数据，则无法使用本工具
 * 
 * @author shyFly
 *
 */
public class EasySqlExecution {
	private DataSource dataSource;
	/** 当前连接 */
	private Connection connect;
	private static Object lock = new Object();
	private boolean autuoCommit = false;
	/** 是否打印日志 */
	private boolean doLogger = true;
	private String driver = "MYSQL";
	// 连接字符编码
	private String charset = "utf8mb4";
	/** 默认连接池别名 */
	private static final String CONFIG_DEFAULT = "CONFIG_DEFAULT";
	private static final String USER_DEFAULT = "USER_DEFAULT";

	/** 默认数据源及连接池，从项目中读取 */
	public EasySqlExecution() {
		if (EasySqlConfiguration.dataSource == null) {
			EasySqlConfiguration.dataSource = setDataSource(CONFIG_DEFAULT, null, null, null, null);
		}
		this.dataSource = EasySqlConfiguration.dataSource;
	}

	/** 传入配置文件名称,自动扫描 */
	public EasySqlExecution(String propertiesPath, boolean isProperties) {
		EasySqlConfiguration.dataSource = new EasySqlConfiguration().dataSource(propertiesPath);
		this.dataSource = EasySqlConfiguration.dataSource;
	}

	public EasySqlExecution(Connection connect) {
		this.connect = connect;
	}

	/**
	 * 根据标记获取连接池,如果不存在则创建
	 * 
	 * @param flag
	 *            标记
	 */
	public EasySqlExecution(String flag) {
		this.dataSource = setDataSource(flag, null, null, null, null);
	}

	/**
	 * 传入一个连接池，并作为本次对象使用的连接池
	 * 
	 * @param dataSource
	 *            外部传入连接池
	 */
	public EasySqlExecution(DataSource dataSource) {
		this.dataSource = dataSource;
	}

	/**
	 * 传入一个连接池作为本次对象使用的连接池。 配置此连接池标记,并将此连接池交予多数据源方案管理
	 * 
	 * @param flag
	 *            标记
	 * @param dataSource
	 *            外部传入连接池
	 */
	public EasySqlExecution(String flag, DataSource dataSource) {
		this.dataSource = setDataSource(flag, dataSource, null, null, null);
	}

	/**
	 * 传入一个数据源,easySql将为其配置连接池并交予多数据源方案管理
	 * 
	 * @param url
	 *            数据源连接的url
	 * @param username
	 *            数据源连接的username
	 * @param password
	 *            数据源连接的password
	 */
	public EasySqlExecution(String url, String username, String password) {
		this.dataSource = setDataSource(USER_DEFAULT, null, url, username, password);
	}

	/**
	 * 传入一个数据源,easySql将为其配置连接池并交予多数据源方案管理
	 * 
	 * @param flag
	 *            标记
	 * @param url
	 *            数据源连接的url
	 * @param username
	 *            数据源连接的username
	 * @param password
	 *            数据源连接的password
	 */
	public EasySqlExecution(String flag, String url, String username, String password) {
		this.dataSource = setDataSource(flag, null, url, username, password);
	}

	/**
	 * 使用oracle驱动
	 */
	public void useOracleDriver() {
		this.driver = "ORACLE";
	}

	/**
	 * 获取当前连接池
	 * 
	 * @return
	 */
	public DataSource getDataSource() {
		return this.dataSource;
	}

	/** 事务处理工具 */
	private TransactionUtil transactionUtil = new TransactionUtil();

	/**
	 * 多数据源存储
	 * 
	 * @param flag
	 * @param dataSource
	 * @param url
	 * @param username
	 * @param password
	 * @return
	 */
	private DataSource setDataSource(String flag, DataSource dataSource, String url, String username, String password) {
		if (flag == null) {
			throw new RuntimeException("连接池别名不能为空,或请使用默认构造器");
		}
		if (EasySqlConfiguration.getDataSourceMapValue(flag) == null) {
			synchronized (lock) {
				if (EasySqlConfiguration.getDataSourceMapValue(flag) == null) {
					if (dataSource != null) {
						EasySqlConfiguration.setDataSourceMapValue(flag, dataSource);
					} else {
						EasySqlConfiguration.url = url;
						EasySqlConfiguration.username = username;
						EasySqlConfiguration.password = password;
						EasySqlConfiguration.setDataSourceMapValue(flag, null);
					}
				}
			}
		}
		return EasySqlConfiguration.getDataSourceMapValue(flag);
	}

	/** 获取当前持有的connection连接 */
	public Connection getConcurrentConnect() {
		Connection connection = null;
		try {
			if (TransactionUtil.localConnection.get() != null) {
				connection = TransactionUtil.localConnection.get();
				if (!connection.isClosed()) {
					return connection;
				}
			}
			if (this.connect != null) {
				connection = this.connect;
				if (connection.isClosed()) {
					throw new RuntimeException("连接已关闭,无可用连接");
				}
			} else if (this.dataSource != null) {
				connection = this.dataSource.getConnection();
			}
			TransactionUtil.localConnection.set(connection);
			connection.setAutoCommit(autuoCommit);
			TransactionUtil.isTransaction.set(false);
		} catch (SQLException e) {
			throw new RuntimeException(e);
		}
		return connection;
	}

	/** 获取当前连接,并默认手动提交事务 */
	private Connection getConnect() {
		return getConcurrentConnect();
	}

	/** 新增时获取主键 */
	public Object insert(String sql, Object... params) {
		Object id = executeInsert(sql, Arrays.asList(params));
		hasTransaction();
		toCommit();
		return id;
	}

	/** 新增时获取主键 */
	private Object executeInsert(String sql, List<Object> params) {
		if (!sql.toUpperCase().startsWith("INSERT")) {
			throw new RuntimeException("当前非新增语句");
		}
		Connection connect = getConnect();
		PreparedStatement statement;
		try {
			connect.prepareStatement("set names " + charset).executeQuery();
			statement = connect.prepareStatement(sql, 1);
			for (int i = 0; i < params.size(); i++) {
				statement.setObject(i + 1, params.get(i));
			}
			executeUpdate(statement);
			printLogger("==============================");
			printLogger("执行新增语句:" + sql.toString());
			printLogger("执行参数为:" + params);
			printLogger("==============================");
			ResultSet result = statement.getGeneratedKeys();
			if (result.next()) {
				return result.getObject(1);
			}
		} catch (Exception e) {
			throw new RuntimeException(e);
		}
		return null;
	}

	/** 装载statment对象 */
	private PreparedStatement executeSql(String sql, List<Object> params) {
		Connection connect = getConnect();
		PreparedStatement statement;
		try {
			connect.prepareStatement("set names " + charset).executeQuery();
			statement = connect.prepareStatement(sql);
			printLogger("==============================");
			printLogger("执行语句:" + sql);
			printLogger("执行参数为:" + params);
			printLogger("==============================");
			if (params == null || params.size() == 0) {
				return statement;
			}
			for (int i = 0; i < params.size(); i++) {
				statement.setObject(i + 1, params.get(i));
			}
			return statement;
		} catch (SQLException e) {
			throw new RuntimeException(e);
		}
	}

	/** 多条查询 */
	private List<Map<String, Object>> executeQuery(PreparedStatement statement) {
		ResultSetMetaData res = null;
		try {
			ResultSet result = statement.executeQuery();
			res = result.getMetaData();
			int count = res.getColumnCount();
			List<Map<String, Object>> list = new ArrayList<Map<String, Object>>();
			while (result.next()) {
				Map<String, Object> map = new HashMap<String, Object>();
				for (int i = 1; i <= count; i++) {
					map.put(res.getColumnLabel(i), result.getObject(i));
				}
				list.add(map);
			}
			return list;
		} catch (Exception e) {
			throw new RuntimeException(e);
		}
	}

	/** 修改 */
	private int executeUpdate(PreparedStatement statement) {
		try {
			int count = statement.executeUpdate();
			return count;
		} catch (Exception e) {
			throw new RuntimeException(e);
		}
	}

	/** 查询条目数 */
	private Double executeCount(PreparedStatement statement) {
		try {
			ResultSet result = statement.executeQuery();
			Object count = 0;
			while(result.next()){
				count = result.getObject(1);
			}
			if (doLogger) {
				printLogger("当前查询到总条数为:" + count);
			}
			return Double.valueOf(count.toString());
		} catch (Exception e) {
			throw new RuntimeException(e);
		}
	}

	/**
	 * 设置提交模式
	 * 
	 * @param autoCommit
	 *            true:自动提交
	 */
	public void setAutoCommit(boolean autoCommit) {
		this.autuoCommit = autoCommit;
		try {
			getConcurrentConnect().setAutoCommit(autoCommit);
		} catch (SQLException e) {
			throw new RuntimeException(e);
		}
		TransactionUtil.autoCommit.set(autoCommit);
	}

	/**
	 * 当前方法是否含有事务
	 * 
	 */
	private void hasTransaction() {
		if (transactionUtil.isTranMothed()) {
			TransactionUtil.isTransaction.set(true);
			printLogger("====当前有事务开启===");
		}
	}

	/** 提交事务 */
	private void toCommit() {
		try {
			if (TransactionUtil.localConnection.get() != null) {
				Connection conn = TransactionUtil.localConnection.get();
				if (conn.isClosed()) {
					conn.close();
					TransactionUtil.localConnection.remove();
					return;
				}
				if (conn.getAutoCommit()) {
					conn.close();
					TransactionUtil.localConnection.remove();
					return;
				} else {
					if (TransactionUtil.isTransaction.get() != null && !TransactionUtil.isTransaction.get()) {
						conn.commit();
						conn.close();
						TransactionUtil.localConnection.remove();
						TransactionUtil.isTransaction.remove();
					}
				}
			}
		} catch (Exception e) {
			throw new RuntimeException(e);
		}
	}

	/** 提交事务 */
	public void commit() {
		TransactionUtil.isTransaction.set(false);
		toCommit();
		printLogger("====当前事务被提交===");
	}

	/** 回滚事务 */
	public void rollBack() {
		try {
			if (TransactionUtil.localConnection.get() != null) {
				Connection conn = TransactionUtil.localConnection.get();
				if (conn.isClosed()) {
					conn.close();
					TransactionUtil.localConnection.remove();
					return;
				}
				if (conn.getAutoCommit()) {
					conn.close();
					TransactionUtil.localConnection.remove();
					return;
				} else {
					if (TransactionUtil.isTransaction.get() != null && !TransactionUtil.isTransaction.get()) {
						conn.rollback();
						conn.close();
						TransactionUtil.localConnection.remove();
						TransactionUtil.isTransaction.remove();
					}
				}
			}
			printLogger("====当前事务已回滚===");
		} catch (SQLException e) {
			throw new RuntimeException(e);
		}
	}

	/**
	 * 自定义语句分页查询
	 * 
	 * @param sql
	 *            自定义sql
	 * @param currentPageParam
	 *            当前页
	 * @param showCountParam
	 *            每页显示条目数
	 *            sql参数
	 * @return
	 */
	@SuppressWarnings("rawtypes")
	public <T> EasySqlResult<Map> queryListPage(String sql, Integer currentPageParam, Integer showCountParam) {
		Object[] objs = new Object[] {};
		return queryListPage(sql, currentPageParam, showCountParam, objs);
	}

	/**
	 * 自定义语句分页查询
	 * 
	 * @param sql
	 *            自定义sql
	 * @param currentPageParam
	 *            当前页
	 * @param showCountParam
	 *            每页显示条目数
	 * @param params
	 *            sql参数
	 * @return
	 */
	@SuppressWarnings("rawtypes")
	public <T> EasySqlResult<Map> queryListPage(String sql, Integer currentPageParam, Integer showCountParam,
			Object... params) {
		return queryListPage(sql, Map.class, currentPageParam, showCountParam, params);
	}

	/**
	 * 自定义语句分页查询
	 * 
	 * @param sql
	 *            自定义sql
	 * @param clazz
	 *            封装的返回值类型
	 * @param currentPageParam
	 *            当前页
	 * @param showCountParam
	 *            每页显示条目数
	 * @return
	 */
	public <T> EasySqlResult<T> queryListPage(String sql, Class<T> clazz, Integer currentPageParam,
			Integer showCountParam) {
		Object[] objs = new Object[] {};
		return queryListPage(sql, clazz, currentPageParam, showCountParam, objs);
	}

	/**
	 * 自定义语句分页查询
	 * 
	 * @param sql
	 *            自定义sql
	 * @param clazz
	 *            封装的返回值类型
	 * @param currentPageParam
	 *            当前页
	 * @param showCountParam
	 *            每页显示条目数
	 * @param params
	 *            sql参数
	 * @return
	 */
	public <T> EasySqlResult<T> queryListPage(String sql, Class<T> clazz, Integer currentPageParam,
			Integer showCountParam, Object... params) {
		if (sql != null && !sql.trim().toUpperCase().startsWith("SELECT")) {
			throw new RuntimeException("当前非查询语句");
		}
		String countSql = "select count(1) from (" + sql + ") tmp";
		int totalResult = executeCount(executeSql(countSql, Arrays.asList(params))).intValue();
		EasySqlResult<T> easySqlResult = new EasySqlResult<T>();
		if (totalResult == 0) {
			return easySqlResult;
		}
		Page page = new Page();
		page.setShowCount(showCountParam);
		page.setCurrentPage(currentPageParam);
		// 设置总记录数
		page.setTotalResult(totalResult);
		// 设置总页数
		page.setTotalPage((totalResult - 1) / page.getShowCount() + 1);
		Integer currentPage = page.getCurrentPage();
		Integer showCount = page.getShowCount();
		// // 开始的数据
		// Integer start = (currentPage - 1) * showCount;
		// // 结束的数据
		// Integer end = showCount;
		// // 如果当前页大于总页数,则返回最后一页
		// if (start >= totalResult) {
		// start = (page.getTotalPage() - 1) * showCount;
		// end = -1;
		// }
		// start = start < 0 ? 0 : start;
		// String limitSql = sql + " limit " + start + "," + end;
		String limitSql = getListPageSql(sql, currentPage, showCount);
		List<T> list = new ArrayList<T>();
		List<Map<String, Object>> result = query(limitSql, params);
		if (result.size() > 0) {
			list = mapToBean(result, clazz);
			easySqlResult.setResultList(list);
			easySqlResult.setPage(page);
		}
		return easySqlResult;
	}

	/**
	 * 自定义语句查询
	 * 
	 * @param sql
	 *            自定义sql
	 *            自定义sql的参数
	 * @return
	 */
	public Map<String, Object> queryOne(String sql) {
		Object[] objs = new Object[] {};
		return queryOne(sql, objs);
	}

	/**
	 * 自定义语句查询
	 * 
	 * @param sql
	 *            自定义sql
	 * @param params
	 *            自定义sql的参数
	 * @return
	 */
	public Map<String, Object> queryOne(String sql, Object... params) {
		if (sql.toUpperCase().contains("LIMIT")) {
			sql = sql.substring(0, sql.indexOf("limit")) + " limit 1";
		} else {
			sql = sql + " limit 1";
		}
		List<Map<String, Object>> list = query(sql, params);
		if (list != null && list.size() == 0) {
			return null;
		}
		return list.get(0);
	}

	/**
	 * 自定义语句查询
	 * 
	 * @param sql
	 * @return
	 */
	public <T> T queryOne(String sql, Class<T> clazz) {
		Object[] objs = new Object[] {};
		return queryOne(sql, clazz, objs);
	}

	/**
	 * 自定义语句查询
	 * 
	 * @param sql
	 *            自定义sql
	 * @param clazz
	 *            返回值类型
	 * @param params
	 *            自定义sql的参数
	 * @return
	 */
	public <T> T queryOne(String sql, Class<T> clazz, Object... params) {
		sql = sql + " limit 1";
		List<T> list = query(sql, clazz, params);
		if (list == null || list.size() == 0) {
			return null;
		}
		return list.get(0);
	}

	/**
	 * 自定义语句查询
	 * 
	 * @param sql
	 *            自定义sql
	 * @param params
	 *            自定义sql的参数
	 * @return
	 */
	public List<Map<String, Object>> query(String sql, Object... params) {
		if (sql != null && !sql.trim().toUpperCase().startsWith("SELECT")) {
			throw new RuntimeException("当前非查询语句");
		}
		List<Object> list = Arrays.asList(params);
		List<Map<String, Object>> result = executeQuery(executeSql(sql, list));
		hasTransaction();
		toCommit();
		if (result == null) {
			result = new ArrayList<Map<String, Object>>();
		}
		return result;
	}

	/**
	 * 自定义语句查询
	 * 
	 * @param sql
	 * @return
	 */
	public List<Map<String, Object>> query(String sql) {
		Object[] objs = new Object[] {};
		return query(sql, objs);
	}

	/**
	 * 自定义语句查询
	 * 
	 * @param sql
	 * @return
	 */
	public <T> List<T> query(String sql, Class<T> clazz) {
		Object[] objs = new Object[] {};
		return query(sql, clazz, objs);
	}

	public <T> List<T> customQuery(String sql, Class<T> clazz){
		Object[] objs = new Object[] {};
		return customQuery(sql, clazz, objs);
	}
	public List<Map<String, Object>> customQuery(String sql) {
		Object[] objs = new Object[] {};
		return customQuery(sql, objs);
	}
	public List<Map<String, Object>> customQuery(String sql, Object... params) {
		List<Object> list = Arrays.asList(params);
		List<Map<String, Object>> result = executeQuery(executeSql(sql, list));
		hasTransaction();
		toCommit();
		if (result == null) {
			result = new ArrayList<Map<String, Object>>();
		}
		return result;
	}
	
	/**
	 * 自定义语句查询
	 * 
	 * @param sql
	 *            自定义sql
	 * @param clazz
	 *            返回值类型
	 * @param params
	 *            自定义sql的参数
	 * @return
	 */
	public <T> List<T> customQuery(String sql, Class<T> clazz, Object... params) {
		List<Object> list = Arrays.asList(params);
		List<Map<String, Object>> result = executeQuery(executeSql(sql, list));
		hasTransaction();
		toCommit();
		if (result == null) {
			result = new ArrayList<Map<String, Object>>();
		}
		return mapToBean(result, clazz);
	}
	
	/**
	 * 自定义语句查询
	 * 
	 * @param sql
	 *            自定义sql
	 * @param clazz
	 *            返回值类型
	 * @param params
	 *            自定义sql的参数
	 * @return
	 */
	public <T> List<T> query(String sql, Class<T> clazz, Object... params) {
		if (sql != null && !sql.trim().toUpperCase().startsWith("SELECT")) {
			throw new RuntimeException("当前非查询语句");
		}
		List<Object> list = Arrays.asList(params);
		List<Map<String, Object>> result = executeQuery(executeSql(sql, list));
		hasTransaction();
		toCommit();
		if (result == null) {
			result = new ArrayList<Map<String, Object>>();
		}
		return mapToBean(result, clazz);
	}

	/**
	 * 自定义语句修改
	 * 
	 * @param sql
	 *            自定义sql
	 * @param params
	 *            自定义sql的参数
	 * @return
	 */
	public int update(String sql, Object... params) {
		if (sql != null && sql.trim().toUpperCase().startsWith("SELECT")) {
			throw new RuntimeException("当前非修改语句");
		}
		List<Object> list = Arrays.asList(params);
		int count = executeUpdate(executeSql(sql, list));
		hasTransaction();
		toCommit();
		return count;
	}

	/**
	 * 自定义语句修改
	 * 
	 * @param sql
	 * @return
	 */
	public int update(String sql) {
		Object[] objs = new Object[] {};
		return update(sql, objs);
	}

	/**
	 * 依据不为空的条件查询多条数据
	 * 
	 * @return
	 */
	public <T> List<T> select(T obj) {
		return select(new SQLExecuter(obj));
	}

	/**
	 * 依据不为空的字段查询,支持排序
	 * 
	 * @param obj
	 * @param orderByTableField
	 * @param desc
	 * @return
	 */
	public <T> List<T> select(T obj, String orderByTableField, boolean desc) {
		return select(new SQLExecuter(obj).orderBy(orderByTableField, desc));
	}

	/**
	 * 依据不为空的条件查询多条数据
	 * 
	 * @return
	 */
	public <T> T selectOne(T obj) {
		return selectOne(obj, null, false);
	}

	/**
	 * 依据不为空的条件查询多条数据
	 * 
	 * @return
	 */
	public <T> T selectOne(SQLExecuter executer) {
		List<T> list = select(executer.setLimitStart(1));
		if (list == null || list.size() == 0) {
			return null;
		}
		return list.get(0);
	}

	/**
	 * 依据不为空属性查询，支持聚合与排序
	 * 
	 * @param obj
	 * @param orderByTableField
	 * @param desc
	 * @return
	 */
	public <T> T selectOne(T obj, String orderByTableField, boolean desc) {
		SQLExecuter executer = new SQLExecuter(obj);
		if (orderByTableField != null) {
			executer.orderBy(orderByTableField, desc);
		}
		executer.setLimitStart(0);
		executer.setLimitSize(1);
		List<T> list = select(executer);
		if (list == null || list.size() == 0) {
			return null;
		}
		return list.get(0);
	}

	/**
	 * 查询
	 * 
	 * @param executer
	 *            sql执行器
	 * @return
	 */
	public <T> List<T> select(SQLExecuter executer) {
		if (executer.getBean() == null) {
			throw new RuntimeException("参数对象不能为空");
		}
		SqlStatement st = creatSelectSql(executer);
		List<Map<String, Object>> list = executeQuery(executeSql(st.getSql().toString(), st.getParams()));
		toCommit();
		if (list == null) {
			return new ArrayList<T>();
		}
		Object obj = executer.getBean();
		@SuppressWarnings("unchecked")
		List<T> resulTs = (List<T>) mapToBean(list, obj);
		return resulTs;
	}

	/**
	 * 查询总条数
	 * 
	 * @param obj
	 * @return
	 */
	public <T> Double selectCount(T obj) {
		return selectCount(new SQLExecuter(obj));
	}

	/**
	 * 查询总条数
	 * 
	 * @return
	 */
	public <T> Double selectCount(SQLExecuter executer) {
		if (executer.getBean() == null) {
			throw new RuntimeException("参数对象不能为空");
		}
		T obj = executer.getBean();
		String tableName = getTableName(obj);
		Map<String, Object> map = getAttributes(obj);
		String rule = executer.getReturnParam();
		if(rule==null || rule.length()==0){
			rule = " COUNT(1) ";
		}
		StringBuffer sql = new StringBuffer("select "+rule+" from " + tableName + " where 1=1 ");
		Set<Entry<String, Object>> entry = map.entrySet();
		List<Object> params = new ArrayList<Object>();
		for (Entry<String, Object> en : entry) {
			sql.append(" and " + en.getKey() + "=? ");
			params.add(en.getValue());
		}
		if (executer.getSql() != null && executer.getSql().length() > 0) {
			sql.append(executer.getSql());
			params.addAll(executer.getParams());
		}
		Double count = executeCount(executeSql(sql.toString(), params));
		toCommit();
		printLogger("==============================");
		printLogger("执行查询语句:" + sql.toString().replace("*", "count(1)"));
		printLogger("执行参数为:" + params);
		printLogger("==============================");
		return count;
	}

	/** 拼装基础查询语句 */
	private SqlStatement creatSelectSql(SQLExecuter executer) {
		SqlStatement st = new SqlStatement();
		Object obj = executer.getBean();
		String tableName = getTableName(obj);
		if (tableName == null) {
			throw new RuntimeException("数据库表名为空，或无@TableName注解");
		}
		Map<String, Object> attributes = getAttributes(obj);
		String returnParam = " * ";
		if (executer.getReturnParam() != null && executer.getReturnParam().length() > 0) {
			returnParam = executer.getReturnParam();
		}
		StringBuffer sql = new StringBuffer("select " + returnParam + " from " + tableName + " where 1=1 ");
		List<Object> params = new ArrayList<Object>();
		for (String key : attributes.keySet()) {
			sql.append("and " + key + "=? ");
			params.add(attributes.get(key));
		}
		if (executer.getSql() != null) {
			sql.append(executer.getSql());
			params.addAll(executer.getParams());
		}
		if (executer.getLimitStart()!=null && executer.getLimitStart()!=0) {
			sql.append(" limit " + executer.getLimitStart());
			if (executer.getLimitSize()!=null && executer.getLimitSize()!=0) {
				sql.append("," + executer.getLimitSize());
			}
		}
		st.setParams(params);
		st.setSql(sql);
		return st;
	}

	/**
	 * 分页查询（依据不为空的条件查询） 查询完毕将page信息回填到参数bean的page属性中
	 * 
	 * @return
	 */
	@SuppressWarnings("unchecked")
	public <T> EasySqlResult<T> listPage(SQLExecuter executer, Integer currentPageParam, Integer showCountParam) {
		EasySqlResult<T> easySqlResult = new EasySqlResult<T>();
		if (showCountParam <= 0) {
			throw new RuntimeException("每页显示条数不能小于0");
		}
		SqlStatement st = creatSelectSql(executer);
		// 获取总记录数
		int totalResult = executeCount(
				executeSql("select count(1) from (" + st.getSql().toString() + ") tmp", st.getParams())).intValue();
		if (totalResult == 0) {
			return easySqlResult;
		}
		Page page = new Page();
		page.setShowCount(showCountParam);
		page.setCurrentPage(currentPageParam);
		// 设置总记录数
		page.setTotalResult(totalResult);
		// 设置总页数
		page.setTotalPage((totalResult - 1) / page.getShowCount() + 1);
		Integer currentPage = page.getCurrentPage();
		Integer showCount = page.getShowCount();
		String limitSql = getListPageSql(st.getSql().toString(), currentPage, showCount);
		List<T> list = (List<T>) query(limitSql, executer.getBean().getClass(), st.getParams().toArray());
		if (list.size() > 0) {
			easySqlResult.setResultList(list);
			easySqlResult.setPage(page);
		}
		return easySqlResult;
	}

	/**
	 * 分页查询，支持排序
	 * 
	 * @param obj
	 * @param currentPageParam
	 * @param showCountParam
	 * @param orderByTableField
	 * @param desc
	 * @return
	 */
	public <T> EasySqlResult<T> listPage(T obj, Integer currentPageParam, Integer showCountParam,
			String orderByTableField, boolean desc) {
		return listPage(new SQLExecuter(obj).orderBy(orderByTableField, desc), currentPageParam, showCountParam);
	}

	/**
	 * 分页查询，支持聚合
	 * 
	 * @param obj
	 * @param currentPageParam
	 * @param showCountParam
	 * @return
	 */
	public <T> EasySqlResult<T> listPage(T obj, Integer currentPageParam, Integer showCountParam) {
		return listPage(new SQLExecuter(obj), currentPageParam, showCountParam);
	}

	/**
	 * 拼接分页
	 * 
	 * @param sql
	 * @param currentPage
	 * @param showCount
	 * @return
	 */
	private String getListPageSql(String sql, int currentPage, int showCount) {
		StringBuffer pageSql = new StringBuffer();
		if ("ORACLE".equals(driver)) {
			pageSql.append("select * from (select tmp_tb.*,ROWNUM row_id from (");
			pageSql.append(sql);
			pageSql.append(") as tmp_tb where ROWNUM<=");
			pageSql.append((currentPage - 1) * showCount + showCount);
			pageSql.append(") where row_id>");
			pageSql.append((currentPage - 1) * showCount);
		} else if ("MYSQL".equalsIgnoreCase(driver)) {
			pageSql.append(sql);
			pageSql.append(" limit ").append((currentPage - 1) * showCount).append(",").append(showCount);
		}
		return pageSql.toString();
	}

	/**
	 * 一次插入多条
	 * 
	 * @param objs
	 * @return
	 */
	public <T> void insert(Collection<T> objs) {
		if (objs == null || objs.size() == 0) {
			throw new RuntimeException("要插入的数据不能为空");
		}
		Iterator<T> it = objs.iterator();
		T obj = null;
		if (it.hasNext()) {
			obj = it.next();
		}
		String tableName = getTableName(obj);
		if (tableName == null) {
			throw new RuntimeException("数据库表名为空，或无@TableName注解");
		}
		Map<String, Object> attributes = getAttributes(obj);
		// 不能添加空数据
		if (attributes == null || attributes.isEmpty()) {
			throw new RuntimeException("数据库表不允许添加空数据");
		}
		StringBuffer sql = new StringBuffer();
		List<Object> params = new ArrayList<Object>();
		Set<String> keys = attributes.keySet();
		sql.append("insert into " + tableName + " (");
		for (String key : keys) {
			sql.append("`" + key + "` ,");
		}
		sql.delete(sql.length() - 1, sql.length());
		sql.append(" )values");
		Iterator<T> iterator = objs.iterator();
		// 循环添加
		while (iterator.hasNext()) {
			sql.append("( ");
			for (@SuppressWarnings("unused")
			String key : keys) {
				sql.append(" ?,");
			}
			Map<String, Object> attr = getAttributes(iterator.next());
			for (String key : keys) {
				params.add(attr.get(key));
			}
			sql.delete(sql.length() - 1, sql.length());
			sql.append(" ),");
		}
		sql.delete(sql.length() - 1, sql.length());
		executeInsert(sql.toString(), params);
		hasTransaction();
		toCommit();
	}

	/**
	 * 添加数据(自动补填id)
	 * 
	 * @param obj
	 * @return
	 */
	public <T> T insert(T obj) {
		// 自增主键不允许加入
		if (getTableIdValue(obj) != null) {
			throw new RuntimeException("主键:" + getTableIdName(obj) + "不允许作为参数");
		}
		String tableName = getTableName(obj);
		if (tableName == null) {
			throw new RuntimeException("数据库表名为空，或无@TableName注解");
		}
		Map<String, Object> attributes = getAttributes(obj);
		// 不能添加空数据
		if (attributes == null || attributes.isEmpty()) {
			throw new RuntimeException("数据库表不允许添加空数据");
		}
		StringBuffer sql = new StringBuffer();
		List<Object> params = new ArrayList<Object>();
		sql.append("insert into " + tableName + " (");
		for (String key : attributes.keySet()) {
			sql.append("`" + key + "` ,");
			params.add(attributes.get(key));
		}
		sql.delete(sql.length() - 1, sql.length());
		sql.append(" )values( ");
		for (@SuppressWarnings("unused")
		String key : attributes.keySet()) {
			sql.append(" ?,");
		}
		sql.delete(sql.length() - 1, sql.length());
		sql.append(" )");
		Object id = executeInsert(sql.toString(), params);
		hasTransaction();
		toCommit();
		String tableIdName = getTableIdJavaName(obj);
		try {
			if (tableIdName == null) {
				printLogger("数据插入成功,但未找到@TableId 注解,无法将主键返回");
				return obj;
			}
			setAttribute(obj, tableIdName, id);
		} catch (Exception e) {
			throw new RuntimeException(e);
		}
		return obj;
	}

	/**
	 * 给对象的属性赋值
	 * 
	 * @param obj
	 *            对象
	 * @param attrName
	 *            对象的属性名
	 * @param value
	 *            对象的属性值
	 */
	@SuppressWarnings("rawtypes")
	private void setAttribute(Object obj, String attrName, Object value) {
		try {
			Class clazz = obj.getClass();
			while (!clazz.equals(Object.class)) {
				try {
					Field f = clazz.getDeclaredField(attrName);

					if (f == null) {
						continue;
					}
					f.setAccessible(true);
					if (value == null || f.getType().isAssignableFrom(value.getClass())) {
						f.set(obj, value);
					} else {
						f.set(obj, parseToObject(value, f.getType()));
					}
					f.setAccessible(false);

					return;
				} catch (NoSuchFieldException e) {
					clazz = clazz.getSuperclass();
				}
			}
		} catch (Exception e) {
			throw new RuntimeException(e);
		}
	}

	/***
	 * 转换类型
	 * 
	 * @param value
	 *            字符串的值
	 * @param type
	 *            要转换的类型
	 * @return 转换后的值
	 */
	@SuppressWarnings("unchecked")
	private <T> T parseToObject(Object value, Class<T> type) {
		Object result = null;
		if (value == null || type == String.class) {
			result = value == null ? null : value.toString();
		} else if (type == Character.class || type == char.class) {
			char[] chars = value.toString().toCharArray();
			result = chars.length > 0 ? chars.length > 1 ? chars : chars[0] : Character.MIN_VALUE;
		} else if (type == Boolean.class || type == boolean.class) {
			result = Boolean.parseBoolean(value.toString());
		}
		// 处理boolean值转换
		else if (type == Double.class || type == double.class) {
			result = value.toString().equalsIgnoreCase("true") ? true
					: value.toString().equalsIgnoreCase("false") ? false : value;
		} else if (type == Long.class || type == long.class) {
			result = Long.parseLong(value.toString());
		} else if (type == Integer.class || type == int.class) {
			result = Integer.parseInt(value.toString());
		} else if (type == Double.class || type == double.class) {
			result = Double.parseDouble(value.toString());
		} else if (type == Float.class || type == float.class) {
			result = Float.parseFloat(value.toString());
		} else if (type == Byte.class || type == byte.class) {
			result = Byte.parseByte(value.toString());
		} else if (type == Short.class || type == short.class) {
			result = Short.parseShort(value.toString());
		}
		return (T) result;
	}

	/**
	 * 修改数据（依据id修改）
	 * 
	 * @return
	 */
	public <T> Integer update(SQLExecuter executer) {
		Object obj = executer.getBean();
		if (obj == null) {
			throw new RuntimeException("参数对象不能为空");
		}
		String tableName = getTableName(obj);
		if (tableName == null) {
			throw new RuntimeException("数据库表名为空，或无@TableName注解");
		}
		Object tableIdValue = null;
		// 参数中的主键
		Object param_id = getTableIdName(obj);
		if (param_id != null) {
			tableIdValue = getTableIdValue(obj);
		}
		// id不能为空
		if (tableIdValue == null && executer.getSql() == null) {
			throw new RuntimeException("主键值为空，或无@TableId注解");
		}
		Map<String, Object> attributes = getAttributes(obj);
		attributes.remove(param_id);
		if (attributes == null || attributes.isEmpty()) {
			throw new RuntimeException("参数属性为空");
		}
		StringBuffer sql = new StringBuffer();
		sql.append("update " + tableName + " set ");
		for (String key : attributes.keySet()) {
			sql.append("`" + key + "`" + " = ? ,");
		}
		// 添加自增字段
		if (!executer.getIncrMap().isEmpty()) {
			Map<String, Object> incrMap = executer.getIncrMap();
			Set<Entry<String, Object>> set = incrMap.entrySet();
			for (Entry<String, Object> entry : set) {
				sql.append("`" + entry.getKey() + "`" + " = " + "`" + entry.getKey() + "+`" + entry.getValue() + ",");
			}
		}
		//添加null值
		if(!executer.getNullValFields().isEmpty()){
			List<String> nullVal = executer.getNullValFields();
			for (String field : nullVal) {
				sql.append("`" + field + "`" + " = NULL,");
			}
		}
		sql.delete(sql.length() - 1, sql.length());
		List<Object> params = new ArrayList<Object>();
		for (String key : attributes.keySet()) {
			params.add(attributes.get(key));
		}
		if (executer.getSql() != null) {
			sql.append(" where 1=1 ").append(executer.getSql());
			params.addAll(executer.getParams());
			if (param_id != null && tableIdValue != null) {
				sql.append(" and " + param_id + " = ?");
				params.add(tableIdValue);
			}
		} else {
			sql.append(" where " + param_id + " = ?");
			params.add(tableIdValue);
		}
		int num = update(sql.toString(), params.toArray());
		hasTransaction();
		toCommit();
		return num;
	}

	public <T> Integer update(T obj) {
		return update(new SQLExecuter(obj));
	}

	/**
	 * 删除数据(依据不为空的属性删除,支持聚合)
	 * 
	 * @return 影响行数
	 */
	public <T> Integer delete(SQLExecuter executer) {
		Object obj = executer.getBean();
		String tableName = getTableName(obj);
		if (tableName == null) {
			return null;
		}
		Map<String, Object> attributes = getAttributes(obj);
		if (executer.getSql() == null && (attributes == null || attributes.isEmpty())) {
			throw new RuntimeException("属性为空,暂不允许删除全表数据");
		}
		StringBuffer sql = new StringBuffer();
		sql.append("delete from " + tableName + " where 1=1 ");
		for (String key : attributes.keySet()) {
			sql.append("and " + key + " =? ");
		}
		List<Object> params = new ArrayList<Object>();
		for (String key : attributes.keySet()) {
			params.add(attributes.get(key));
		}
		if (executer.getSql() != null) {
			sql.append(executer.getSql());
			params.addAll(executer.getParams());
		}
		int num = update(sql.toString(), params.toArray());
		hasTransaction();
		toCommit();
		return num;
	}

	/**
	 * 依据不为空的属性删除数据
	 * 
	 * @param obj
	 * @return
	 */
	public <T> Integer delete(T obj) {
		return delete(new SQLExecuter(obj));
	}

	/**
	 * 获取表名
	 * 
	 * @param <T>
	 */
	private <T> String getTableName(T t) {
		Class<?> tClass = t.getClass();
		String tableName = tClass.getSimpleName();
		// 获取表名注解
		TableName annotation = tClass.getDeclaredAnnotation(TableName.class);
		if (annotation != null) {
			if (!annotation.value().equals("")) {
				tableName = annotation.value();
			}
			if (!tableName.equals("")) {
				return tableName;
			}
		}
		if (tClass.getSimpleName().contains("Bean") && tableName.equals(tClass.getSimpleName())) {
			tableName = tableName.substring(0, tClass.getSimpleName().indexOf("Bean"));
        }
		tableName = StringUtil.conversionMapUnderscore(tableName);
		return "`"+tableName+"`";
	}

	/**
	 * 获取主键id的名称
	 * 
	 * @param <T>
	 */
	@SuppressWarnings("rawtypes")
	private <T> String getTableIdName(Object t) {
		String tableId = null;
		for (Class clazz = t.getClass(); !clazz.equals(Object.class); clazz = clazz.getSuperclass()) {
			Field[] fs = clazz.getDeclaredFields();
			for (Field f : fs) {
				// 获取tableId注解
				TableId annotation = f.getDeclaredAnnotation(TableId.class);
				TableField tableField = f.getDeclaredAnnotation(TableField.class);
				if (annotation != null) {
					if (tableField != null) {
						if (!tableField.name().equals("")) {
							tableId = tableField.name();
						} else if (!tableField.value().equals("")) {
							tableId = tableField.value();
						}
					} else {
						tableId = f.getName();
					}
				}
				if (tableId != null) {
					return tableId;
				}
			}
		}
		return null;
	}

	/**
	 * 获取主键id的值
	 * 
	 * @param <T>
	 */
	@SuppressWarnings("rawtypes")
	private <T> Object getTableIdValue(Object t) {
		for (Class clazz = t.getClass(); !clazz.equals(Object.class); clazz = clazz.getSuperclass()) {
			Field[] fs = clazz.getDeclaredFields();
			for (Field f : fs) {
				// 获取tableId注解
				TableId annotation = f.getDeclaredAnnotation(TableId.class);
				if (annotation != null) {
					f.setAccessible(true);
					try {
						return f.get(t);
					} catch (Exception e) {
						throw new RuntimeException(e);
					}
				}
			}
		}
		return null;
	}

	/**
	 * 获取原类属性id的名字
	 * 
	 * @param <T>
	 */
	@SuppressWarnings("rawtypes")
	private <T> String getTableIdJavaName(Object t) {
		for (Class clazz = t.getClass(); !clazz.equals(Object.class); clazz = clazz.getSuperclass()) {
			Field[] fs = clazz.getDeclaredFields();
			for (Field f : fs) {
				// 获取tableId注解
				TableId annotation = f.getDeclaredAnnotation(TableId.class);
				if (annotation != null) {
					return f.getName();
				}
			}
		}
		return null;
	}

	/**
	 * 获取对象中的所有属性
	 * 
	 * @param bean
	 *            对象
	 * @return 属性和值(Map[属性名, 属性值])
	 */
	@SuppressWarnings("rawtypes")
	private Map<String, Object> getAttributes(Object bean) {
		try {
			Map<String, Object> map = new HashMap<String, Object>();
			// 主键id字段是否被改变
			boolean changeTableId = false;
			for (Class clazz = bean.getClass(); !clazz.equals(Object.class); clazz = clazz.getSuperclass()) {
				Field[] fs = clazz.getDeclaredFields();
				for (Field f : fs) {
					f.setAccessible(true);
					if (f.get(bean) == null || (changeTableId && f.getName().equals("id"))) {
						f.setAccessible(false);
						continue;
					}
					// 如果该字段被标注忽略，则不往数据库添加此字段
					TableIgnore tableIgnore = f.getDeclaredAnnotation(TableIgnore.class);
					if (tableIgnore != null) {
						continue;
					}
					// 如果该字段被标注，则不从basebean中取id
					TableId tableId = f.getDeclaredAnnotation(TableId.class);
					// 如果当前类里面属性含有自定义注解，则处理属性名
					// if (clazz == bean.getClass()) {
					TableField tableField = f.getDeclaredAnnotation(TableField.class);
					if (tableId != null && tableField != null && !tableField.name().equals("")) {
						changeTableId = true;
					}
					if (tableField != null) {
						String tableFieldName = "";
						if (!tableField.name().equals("")) {
							tableFieldName = tableField.name();
						}
						if (!tableField.value().equals("")) {
							tableFieldName = tableField.value();
						}
						if (tableFieldName.equals("")) {
							continue;
						}
						f.setAccessible(true);
						map.put(tableFieldName, f.get(bean));
						f.setAccessible(false);
						continue;
					}
					// }
					// 子类最大，父类值不覆盖子类
					if (map.containsKey(f.getName())) {
						continue;
					}
					f.setAccessible(true);
					Object value = f.get(bean);
					f.setAccessible(false);
					if (value == null) {
						continue;
					}
					map.put(f.getName(), value);
				}
			}
			map.remove("serialVersionUID");
			return map;
		} catch (Exception e) {
			throw new RuntimeException(e);
		}
	}

	/**
	 * 设置是否打印日志
	 * 
	 * @param doLogger
	 */
	public void printLogger(boolean doLogger) {
		this.doLogger = doLogger;
	}

	private void printLogger(String message) {
		if (this.doLogger) {
			LoggerUtil.info(this.getClass(), message);
		}
	}

	/**
	 * 设置字符编码
	 * 
	 * @param charset
	 */
	public void setCharSet(String charset) {
		if (charset == null || charset.length() == 0) {
			return;
		}
		this.charset = charset;
	}

	/**
	 * 设置连接类型为oracle
	 * 
	 */
	public void setOracleDriver() {
		this.driver = "ORACLE";
	}

	/***
	 * 校验是否是九种基础类型(即：非用户定义的类型)
	 * 
	 * @param value
	 *            字符串的值 要校验的值
	 * @return 是否是基础类型(true:已经是基础类型了)
	 */
	private boolean isBaseClass(@SuppressWarnings("rawtypes") Class value) {
		if (value == null) {
			return true;
		} else if (value.equals(Long.class)) {
			return true;
		} else if (value.equals(Integer.class)) {
			return true;
		} else if (value.equals(Double.class)) {
			return true;
		} else if (value.equals(Float.class)) {
			return true;
		} else if (value.equals(Byte.class)) {
			return true;
		} else if (value.equals(Boolean.class)) {
			return true;
		} else if (value.equals(Short.class)) {
			return true;
		} else if (value.equals(Character.class)) {
			return true;
		} else if (value.equals(String.class)) {
			return true;
		}
		return false;
	}

	/**
	 * 将map转成bean
	 * 
	 * @param list
	 * @param obj
	 * @return
	 */
	@SuppressWarnings({ "rawtypes", "unchecked" })
	private <T> List<T> mapToBean(List<Map<String, Object>> list, T obj) {
		// 最终封装后的返回
		List<T> resulTs = new ArrayList<T>();
		try {
			for (Map<String, Object> temp : list) {
				// 每条结果都封装在新obj对象里
				Object t = obj.getClass().newInstance();
				for (Class clazz = t.getClass(); !clazz.equals(Object.class); clazz = clazz.getSuperclass()) {
					Field[] fields = clazz.getDeclaredFields();
					for (Field f : fields) {
						// 如果该类的属性名与表里不一致，则取表字段名
						TableField tableField = f.getDeclaredAnnotation(TableField.class);
						try {
							if (tableField != null) {
								String tableFieldName = "";
								if (!tableField.name().equals("")) {
									tableFieldName = tableField.name();
								}
								if (!tableField.value().equals("")) {
									tableFieldName = tableField.value();
								}
								if (tableFieldName.equals("")) {
									continue;
								}
								f.setAccessible(true);
								f.set(t, temp.get(tableFieldName));
								f.setAccessible(false);
							}
							if (temp.get(f.getName()) != null) {
								f.setAccessible(true);
								f.set(t, temp.get(f.getName()));
								f.setAccessible(false);
							}
						} catch (Exception e) {
							throw new RuntimeException("类属性" + f.getName() + "与表字段类型不符", e);
						}
					}
				}
				resulTs.add((T) t);
			}
		} catch (Exception e) {
			throw new RuntimeException(e);
		}
		return resulTs;
	}

	/**
	 * 将map转成bean
	 * 
	 * @param list
	 * @param clazz
	 * @return
	 */
	@SuppressWarnings("unchecked")
	public <T> List<T> mapToBean(List<Map<String, Object>> list, Class<T> clazz) {
		try {
			if (list.size() == 0) {
				return new ArrayList<>();
			}
			if (isBaseClass(clazz)) {
				if (list.size() > 0 && list.get(0).size() == 1) {
					List<T> result = new ArrayList<T>();
					T value = null;
					String mapKey = null;
					for (String key : list.get(0).keySet()) {
						mapKey = key;
						value = (T) list.get(0).get(key);
					}
					for (Map<String, Object> map : list) {
						value = (T) map.get(mapKey);
						result.add(value);
					}
					return result;
				}
			}
			if (clazz.equals(Map.class)) {
				return (List<T>) list;
			}
			T t = clazz.newInstance();
			return mapToBean(list, t);
		} catch (Exception e) {
			throw new RuntimeException(e);
		}
	}

	class SqlStatement {
		private StringBuffer sql;
		private List<Object> params;

		public StringBuffer getSql() {
			return sql;
		}

		public void setSql(StringBuffer sql) {
			this.sql = sql;
		}

		public List<Object> getParams() {
			return params;
		}

		public void setParams(List<Object> params) {
			this.params = params;
		}
	}
}
