package cn.easyutil.easyapi.interview.controller;

import cn.easyutil.easyapi.content.AuthModuleBean;
import cn.easyutil.easyapi.content.AuthModuleButtonsEnum;
import cn.easyutil.easyapi.content.ProjectContext;
import cn.easyutil.easyapi.entity.common.AccessAuth;
import cn.easyutil.easyapi.entity.common.ApidocComment;
import cn.easyutil.easyapi.entity.db.auth.*;
import cn.easyutil.easyapi.exception.ApidocException;
import cn.easyutil.easyapi.interview.dto.*;
import cn.easyutil.easyapi.interview.session.CurrentSession;
import cn.easyutil.easyapi.interview.session.SessionUser;
import cn.easyutil.easyapi.interview.vo.UserProjectBean;
import cn.easyutil.easyapi.mybatis.service.*;
import cn.easyutil.easyapi.util.AssertUtil;
import cn.easyutil.easyapi.util.StringUtil;
import com.baomidou.mybatisplus.core.toolkit.Wrappers;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import org.springframework.beans.BeanUtils;
import org.springframework.boot.autoconfigure.condition.ConditionalOnMissingBean;
import org.springframework.util.CollectionUtils;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RestController;

import java.util.*;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;

@RestController
@ConditionalOnMissingBean(name = "easyapiUserController")
@RequestMapping("/easyapi/doc/user")
public class UserController {

    private final UserService userService = new UserService();

    private final UserTemporaryAuthService userTemporaryAuthService = new UserTemporaryAuthService();

    private final ProjectService projectService = new ProjectService();

    @ApidocComment("登陆接口")
    @PostMapping("/login")
    @AccessAuth(code = 1000)
    public DBUserEntity login(@RequestBody LoginDto dto){
        AssertUtil.isNull(StringUtil.isEmpty(dto.getAccount()), "用户名不能为空");
        AssertUtil.isNull(StringUtil.isEmpty(dto.getPassword()), "密码不能为空");
        DBUserEntity user = userService.getByAccount(dto.getAccount());
        AssertUtil.isNull(user, "用户名或密码错误");
        AssertUtil.isTrue(!user.getPassword().equals(StringUtil.toMD5(dto.getPassword())), "用户名或密码错误");
        AssertUtil.isTrue(user.getDisable()==1, "您已被禁用");
        //查询用户的角色
        SessionUser sessionUser = new SessionUser();
        BeanUtils.copyProperties(user, sessionUser);
        CurrentSession.loginSuccess(sessionUser);
        if(user.getSuperAdminStatus() == 1){
            sessionUser.setAuthCodes(Stream.of(AuthModuleButtonsEnum.values()).map(AuthModuleButtonsEnum::getCode).collect(Collectors.toList()));
            user.setCodes(sessionUser.getAuthCodes());
            return user;
        }
        List<String> codes = userTemporaryAuthService.list(Wrappers.lambdaQuery(DBUserTemporaryAuthEntity.class)
                .eq(DBUserTemporaryAuthEntity::getUserId, user.getId())
        ).stream().map(DBUserTemporaryAuthEntity::getAuthCode).collect(Collectors.toList());

        sessionUser.setAuthCodes(codes);

        user.setCodes(codes);
        return user;
    }

    @ApidocComment("修改个人密码")
    @PostMapping("/updatePassword")
    @AccessAuth(code = 1000)
    public void updatePassword(@RequestBody UpdatePasswordDto dto){
        SessionUser user = CurrentSession.getCurrentUser();
        if(user.getSuperAdminStatus()==1 && !ProjectContext.allConfiguration.getUserConfiguration().isEnableSuperAdminModify()){
            //如果是超管，并且配置文件禁止修改个人信息，则不允许修改
            throw new ApidocException("已配置不允许修改超管信息");
        }
        AssertUtil.isTrue(!user.getPassword().equals(StringUtil.toMD5(dto.getOldPassword())), "旧密码输入不正确");
        user.setPassword(dto.getNewPassword());
        DBUserEntity userEntity = new DBUserEntity();
        userEntity.setId(user.getId());
        userEntity.setPassword(StringUtil.toMD5(dto.getNewPassword()));
        userService.updateById(userEntity);
    }

    @ApidocComment("条件查询用户列表")
    @PostMapping("/selectUsers")
    @AccessAuth(code = 1000)
    public Page<DBUserEntity> selectUsers(@RequestBody SelectUsersDto dto){
        SessionUser currentUser = CurrentSession.getCurrentUser();
        if(currentUser.getSuperAdminStatus() == 0){
            Long currentProjectId = CurrentSession.getCurrentProjectId();
            dto.setProjectId(currentProjectId);
        }
        DBUserEntity query = new DBUserEntity();
        BeanUtils.copyProperties(dto, query);
        query.setHidden(0);
        return userService.page(new Page<>(dto.getCurrentPage(),dto.getShowCount()),Wrappers.lambdaQuery(query));
    }

    @ApidocComment("修改用户信息")
    @PostMapping("/updateUser")
    @AccessAuth(code = 1000)
    public void updateUser(@RequestBody DBUserEntity user){
        AssertUtil.isNull(user.getId(), "要修改的用户id不能为空");
        DBUserEntity query = userService.getById(user.getId());
        AssertUtil.isNull(query, "用户不存在");
        if(!StringUtil.isEmpty(user.getAccount()) && !query.getAccount().equals(user.getAccount())){
            DBUserEntity byAccount = userService.getByAccount(user.getAccount());
            AssertUtil.isTrue(byAccount != null, "账号已存在");
        }
        if(!StringUtil.isEmpty(user.getPassword()) && !user.getPassword().equals(query.getPassword())){
            user.setPassword(StringUtil.toMD5(user.getPassword()));
        }
        userService.updateById(user);
    }

    @ApidocComment("删除用户信息")
    @PostMapping("/deleteUser")
    @AccessAuth(code = 1000)
    public void deleteUser(@RequestBody IdDto dto){
        userService.removeById(dto.getId());
    }

    @ApidocComment("添加用户")
    @PostMapping("/addUser")
    @AccessAuth(code = 1000)
    public void addUser(@RequestBody DBUserEntity entity){
        AssertUtil.isNull(StringUtil.isEmpty(entity.getAccount()), "账号不能为空");
        DBUserEntity byAccount = userService.getByAccount(entity.getAccount());
        AssertUtil.isTrue(byAccount != null, "账号已存在");
        entity.setId(null);
        if(StringUtil.isEmpty(entity.getPassword())){
           entity.setPassword(StringUtil.toMD5("123456"));
        }else{
            entity.setPassword(StringUtil.toMD5(entity.getPassword()));
        }
        if(entity.getProjectId() == null){
            entity.setProjectId(ProjectContext.currentProjectId);
        }
        userService.save(entity);
        userTemporaryAuthService.binds(entity,Stream.of(AuthModuleButtonsEnum.values()).filter(item -> item.getDefaultStatus()==1).map(AuthModuleButtonsEnum::getCode).collect(Collectors.toList()));

    }

    @ApidocComment("为用户绑定按钮权限")
    @PostMapping("/bindAuth")
    @AccessAuth(code = 1000)
    public void bindAuth(@RequestBody BindAuthDto bind){
        if(CollectionUtils.isEmpty(bind.getCodes())){
            return ;
        }
        userTemporaryAuthService.delByUserId(bind.getUserId());
        DBUserEntity user = userService.getById(bind.getUserId());
        userTemporaryAuthService.binds(user,bind.getCodes());
    }

    @ApidocComment("查看用户的按钮权限")
    @PostMapping("/getAuthInfo")
    @AccessAuth(code = 1000)
    public List<AuthModuleBean> getAuthInfo(@RequestBody IdDto dto){
        Set<String> codes = userTemporaryAuthService.list(Wrappers.lambdaQuery(DBUserTemporaryAuthEntity.class)
                .eq(DBUserTemporaryAuthEntity::getUserId, dto.getId())
        ).stream().map(DBUserTemporaryAuthEntity::getAuthCode).collect(Collectors.toSet());

        return AuthModuleButtonsEnum.getModules(codes);
    }

    @ApidocComment("查看全部权限按钮")
    @PostMapping("/getAllAuth")
    @AccessAuth(code = 1000)
    public List<AuthModuleBean> getAllAuth(){
        return AuthModuleButtonsEnum.getModules(null);
    }

    @ApidocComment("绑定项目")
    @PostMapping("/bindProjects")
    @AccessAuth(code = 1000)
    public void bindProjects(@RequestBody BindProjectDto dto){
        DBUserEntity user = userService.getById(dto.getUserId());
        user.setBindProjectIds(String.join(",",dto.getProjectIds()));
        userService.updateById(user);
    }

    @ApidocComment("获取用户绑定的项目")
    @PostMapping("/getUserProjects")
    @AccessAuth(code = 1000)
    public List<UserProjectBean> getUserProjects(@RequestBody IdDto dto){
        List<DBProjectEntity> projects = projectService.list();
        if(CollectionUtils.isEmpty(projects)){
            return null;
        }
        DBUserEntity user = userService.getById(dto.getId());
        List<String> bindProjectIds = Arrays.asList(Optional.ofNullable(user.getBindProjectIds()).orElse("").split(","));
        List<UserProjectBean> result = new ArrayList<>();
        for (DBProjectEntity project : projects) {
            UserProjectBean upb = new UserProjectBean();
            upb.setProjectName(project.getProjectName());
            upb.setProjectId(project.getId());
            if(bindProjectIds.contains(project.getId().toString())){
                upb.setChecked(1);
            }else{
                upb.setChecked(0);
            }
            result.add(upb);
        }
        return result;
    }

}
