package cn.easyutil.easyapi.javadoc.util;

import cn.easyutil.easyapi.javadoc.reader.ClassComment;
import cn.easyutil.easyapi.javadoc.reader.MethodComment;
import cn.easyutil.easyapi.logic.creator.MethodParam;
import cn.easyutil.easyapi.util.JsonUtil;
import cn.easyutil.easyapi.util.StringUtil;
import org.springframework.util.CollectionUtils;

import java.lang.reflect.Method;
import java.util.*;

/**
 * 不确定的参数解析，只解析Map和Object
 */
public class UncertainParamParser {

    private MethodComment methodComment;

    private ClassComment classComment;

    private Method method;

    private Class<?> clazz;

    private UncertainParamParser(){}

    /**
     * 构建解析器
     * @param clazz 所属类
     * @param method    所属方法
     * @param classComment  类源码
     * @param methodComment 方法源码
     */
    public static UncertainParamParser builder(Class<?> clazz,Method method,ClassComment classComment,MethodComment methodComment){
        UncertainParamParser parser = new UncertainParamParser();
        parser.clazz = clazz;
        parser.method = method;
        parser.classComment = classComment;
        parser.methodComment = methodComment;
        if(parser.classComment == null){
            parser.classComment = new ClassComment();
        }
        if(parser.methodComment == null){
            parser.methodComment = new MethodComment();
        }
        if(CollectionUtils.isEmpty(parser.classComment.getLines())){
            parser.classComment.setLines(Collections.emptyList());
        }
        if(CollectionUtils.isEmpty(parser.methodComment.getLines())){
            parser.methodComment.setLines(Collections.emptyList());
        }
        return parser;
    }

    /**
     * 获取参数为Object类型的实际类型
     * @param paramName 参数名称
     * @return  参数类型集合
     */
    public MethodParam parseRequestObj(String paramName){
        /*
         * 如果参数是Object,只解析以下情况,其他情况不解析
         * A a = (A) obj;
         * A a = invork(obj);
         * A a = invork(obj,p1,p2);
         */
        Class<?> parseType = null;
        String variableName = null;
        for (String line : methodComment.getLines()) {
            if(!CommentStringUtil.isAssignLine(paramName,line)){
                //不是赋值的行
                continue;
            }
            MapKeyVal<String> assign = CommentStringUtil.getAssign(line.split("=")[0]);
            //获取赋值类型
            Class<?> aClass = CommentTypeUtil.findQuoteClass(assign.getKey(), methodComment.getLines(), classComment.getLines(), clazz);
            if(aClass == null){
                return null;
            }
            parseType = aClass;
            variableName = assign.getVal();
            break;
        }
        if(parseType == null){
            return null;
        }
        MethodParam param = new MethodParam();
        param.setParamName(paramName);
        param.setParamType(parseType);
        if(!Map.class.isAssignableFrom(parseType)){
            //非map类型直接返回
            return param;
        }
        //map类的参数还得继续处理
        List<MethodParam> children = getMapGetValByVariableName(variableName);
        param.setChildren(children);
        return param;
    }

    /**
     * 获取源码中指定map对象put进去的key和val
     * @param mapVariableName   对象变量名称
     */
    public List<MethodParam> getMapPutValByVariableName(String mapVariableName){
        List<MethodParam> result = new ArrayList<>();
        for (String line : methodComment.getLines()) {
            MapPutMethodsPreEnum preEnum = MapPutMethodsPreEnum.hasKeyword(mapVariableName, null, line);
            if(preEnum == null){
                continue;
            }
            String params = line.substring(line.indexOf(preEnum.getKeyword()) + preEnum.getKeyword().length());
            params = params.substring(0,params.lastIndexOf(")")).trim();
            String[] split = params.split(",");
            if(split.length <=1){
                continue;
            }
            String splitKey = params.substring(0,params.indexOf(","));
            String splitVal = params.substring(params.indexOf(",")+1);
            /*
             * 先处理key，支持两种方式
             * 1. map.put("key",val)
             * 2. map.put(Const.key,val)
             */
            String key = CommentTypeUtil.getMapKeyName(splitKey,clazz,methodComment.getLines(),classComment.getLines());
            if(StringUtil.isEmpty(key)){
                continue;
            }
            //下面处理val
            Class<?> aClass = CommentTypeUtil.findQuoteClass(splitVal, methodComment.getLines(), classComment.getLines(), clazz);
            if(aClass == null){
                continue ;
            }
            MethodParam param = new MethodParam();
            param.setParamName(key);
            param.setParamType(aClass);
            result.add(param);
        }
        return result;
    }

    /**
     * 获取指定map变量get出来的key和val
     * @param mapVariableName   map变量名称
     */
    public List<MethodParam> getMapGetValByVariableName(String mapVariableName){
        List<MethodParam> result = new ArrayList<>();
        //代码中重新给变量赋值的变量(有些傻逼)
        String variableName = null;
        for (String line : methodComment.getLines()) {
            if(CommentStringUtil.isAssignLine(mapVariableName,line)){
                //重新赋值的代码
                MapKeyVal<String> assign = CommentStringUtil.getAssign(line);
                variableName = assign.getVal();
            }
            String searchKey = null;
            Class<?> val = null;
            MapGetMethodsPreEnum preEnum = MapGetMethodsPreEnum.hasKeyword(mapVariableName, null, line);
            if(preEnum != null){
                searchKey = mapVariableName+preEnum.getKeyword();
                val = preEnum.getClazz();
            }
            if(variableName != null){
                MapGetMethodsPreEnum varEnum = MapGetMethodsPreEnum.hasKeyword(variableName, null, line);
                if(varEnum != null){
                    searchKey = variableName+varEnum.getKeyword();
                    val = varEnum.getClazz();
                }
            }
            if(searchKey == null){
                continue;
            }
            if(val == null){
                //val是空的话，获取val的类型
                if(!line.contains("=")){
                    continue;
                }
                MapKeyVal<String> assign = CommentStringUtil.getAssign(line.split("=")[0]);
                val = CommentTypeUtil.findQuoteClass(assign.getKey(),methodComment.getLines(),classComment.getLines(),clazz);
            }
            if(val == null){
                continue;
            }
            //再处理key
            String params = line.substring(line.indexOf(searchKey)+searchKey.length());
            params = params.substring(0,params.lastIndexOf(")")).trim();
            String key = CommentTypeUtil.getMapKeyName(params,clazz,methodComment.getLines(),classComment.getLines());
            if(StringUtil.isEmpty(key)){
                continue;
            }
            MethodParam param = new MethodParam();
            param.setParamName(key);
            param.setParamType(val);
            result.add(param);
        }
        return result;
    }



    /**
     * map参数对应的key和val
     * @param paramName map参数的变量名
     */
    public MethodParam parseRequestMap(String paramName){
        /*
         * 参数是Map类型,只解析以下情况
         * 1.A a = (A) map.get("a"); //直接取变量并赋值
         * 2.A a = (A) map.get(Const.a); //参数名从其他类的常量中获取
         * 3.A amap = map;
         *   B b = amap.get("b");
         *   B b = amap.get(Const.b);
         */
        List<MethodParam> list = getMapGetValByVariableName(paramName);
        MethodParam param = new MethodParam();
        param.setParamName(paramName);
        param.setParamType(Map.class);
        param.setChildren(list);
        return param;
    }

    public MethodParam parseRequestMapField(String paramName,String fieldName){
        /**
         * 参数属性是Map类型,只解析以下情况
         * 1.A a = (A) param.getMap().get("a"); //直接取变量并赋值
         * 2.A a = (A) param.getMap().get(Const.a); //参数名从其他类的常量中获取
         * 3.A amap = param.getMap();
         *   B b = amap.get("b");
         *   B b = amap.get(Const.b);
         */
        String start = fieldName.substring(0,1).toUpperCase();
        return parseRequestMap(paramName+".get"+start+fieldName.substring(1)+"()");
    }

    public MethodParam parseRequestObjField(String paramName,String fieldName){
        /**
         * 参数属性是Object类型,只解析以下情况
         * 1.A a = (A) param.getObject(); //直接取变量并赋值
         * 2.A a = invork(param.getObject()); //参数名从其他类的常量中获取
         */
        String start = fieldName.substring(0,1).toUpperCase();
        return parseRequestObj(paramName+".get"+start+fieldName.substring(1)+"()");
    }

    public MethodParam parseResponseObj(){
        /**
         * 返回值是Object类型，解析以下情况
         * 1. A a = new A();
         *   return a;
         *
         * 2. A a = invork();
         *   return a;
         *
         * 3. return invork();
         */
        List<String> lines = JsonUtil.jsonToList(JsonUtil.beanToJson(methodComment.getLines()),String.class);
        Collections.reverse(lines);
        String returnVariableName = CommentStringUtil.getReturnVariable(lines);
        if(returnVariableName == null){
            return null;
        }
        Class<?> aClass = CommentTypeUtil.findVariableClassByReflection(returnVariableName,clazz,method);
        if(aClass == null){
            aClass = CommentTypeUtil.findQuoteClass(returnVariableName, methodComment.getLines(), classComment.getLines(), clazz);
            if(aClass == null){
                return null;
            }
        }
        MethodParam param = new MethodParam();
        param.setParamName(returnVariableName);
        param.setParamType(aClass);
        if(!Map.class.isAssignableFrom(aClass)){
            return param;
        }
        List<MethodParam> children = getMapPutValByVariableName(returnVariableName);
        param.setChildren(children);
        return param;
    }

    public MethodParam parseResponseMap(){
        /**
         * 返回值是map类型，解析以下情况
         * 1. map.put("key",val);
         *    return map;
         *
         * 2. map.put(Const.key,val);
         *    return map;
         *
         * 3. map.put("key",invork());
         *    return map;
         *
         * 4. map.put("key",new A());
         *    return map;
         */
        return parseResponseObj();
    }

    public MethodParam parseResponseMapField(String fieldName){
        /**
         * 返回类型的属性是map类型，解析以下情况
         * 1. result.setMap(a);
         *    return result;
         */

        return parseResponseObjField(fieldName);
    }

    public MethodParam parseResponseObjField(String fieldName){
        /**
         * 返回类型的属性是Object类型，解析以下情况
         * 1. result.setObject(a);
         *    return result;
         */
        List<String> lines = JsonUtil.jsonToList(JsonUtil.beanToJson(methodComment.getLines()),String.class);
        Collections.reverse(lines);
        String returnVariable = CommentStringUtil.getReturnVariable(lines);
        if(StringUtil.isEmpty(returnVariable)){
            return null;
        }
        String params = null;
        for (String line : lines) {
            String keyword = returnVariable.toUpperCase()+".SET"+fieldName.toUpperCase()+"(";
            if(!line.toUpperCase().contains(keyword)){
                continue;
            }
            int index = line.toUpperCase().indexOf(keyword)+keyword.length();
            params = line.substring(index);
            params = params.substring(0,params.indexOf(")"));
            break;
        }
        if(params == null){
            return null;
        }
        Class<?> aClass = CommentTypeUtil.findVariableClassByReflection(params, clazz, method);
        if(aClass == null){
            aClass = CommentTypeUtil.findQuoteClass(params,methodComment.getLines(),classComment.getLines(),clazz);
        }
        if(aClass == null){
            return null;
        }
        MethodParam param = new MethodParam();
        param.setParamName(fieldName);
        param.setParamType(aClass);
        if(!Map.class.isAssignableFrom(aClass)){
            return param;
        }
        List<MethodParam> children = getMapPutValByVariableName(params);
        param.setChildren(children);
        return param;
    }
}
