package cn.easyutil.easyapi;

import cn.easyutil.easyapi.configuration.*;
import cn.easyutil.easyapi.content.AuthModuleButtonsEnum;
import cn.easyutil.easyapi.content.DBTableClassify;
import cn.easyutil.easyapi.content.DBTables;
import cn.easyutil.easyapi.content.ProjectContext;
import cn.easyutil.easyapi.entity.auth.AuthModule;
import cn.easyutil.easyapi.entity.db.auth.*;
import cn.easyutil.easyapi.entity.db.doc.*;
import cn.easyutil.easyapi.entity.db.unit.DBSimpleUnitEntity;
import cn.easyutil.easyapi.exception.ApidocException;
import cn.easyutil.easyapi.handler.operator.provide.*;
import cn.easyutil.easyapi.hook.ApiRunAround;
import cn.easyutil.easyapi.hook.DefaultApiRunAround;
import cn.easyutil.easyapi.interview.dto.OpenSyncDto;
import cn.easyutil.easyapi.logic.creator.MethodParam;
import cn.easyutil.easyapi.logic.run.DocCreate;
import cn.easyutil.easyapi.mybatis.MybatisUtil;
import cn.easyutil.easyapi.mybatis.SqlExecMapper;
import cn.easyutil.easyapi.mybatis.service.*;
import cn.easyutil.easyapi.util.AssertUtil;
import cn.easyutil.easyapi.util.JsonUtil;
import cn.easyutil.easyapi.util.StringUtil;
import com.alibaba.druid.pool.DruidDataSource;
import com.baomidou.mybatisplus.core.toolkit.Wrappers;
import org.springframework.context.ApplicationContext;

import javax.sql.DataSource;
import java.io.File;
import java.io.IOException;
import java.lang.reflect.Field;
import java.net.InetAddress;
import java.net.UnknownHostException;
import java.util.*;
import java.util.stream.Collectors;
import java.util.stream.Stream;

/**
 * 接口文档启动类
 */
public class EasyapiRun {

    private ProjectService projectService;
    private UserService userService;
    private DictService dictService;
    private UserTemporaryAuthService userTemporaryAuthService;
    private HostService hostService;
    private ModuleService moduleService;
    private RoleService roleService;
    private RoleAuthService roleAuthService;
    private RoleProjectService roleProjectService;
    private ControllerService controllerService;
    private InterfaceService interfaceService;
    private InterfaceParamService interfaceParamService;
    private SimpleUnitService simpleUnitService;

    private final AllConfiguration all;
    private final ApplicationContext appContext;
    private ApiRunAround apiRunAround = new DefaultApiRunAround();

    public EasyapiRun(AllConfiguration all, ApplicationContext appContext) {
        this.all = all;
        this.appContext = appContext;
        AssertUtil.isNull(all,"配置信息不能为空");
        AssertUtil.isNull(appContext,"spring环境信息不能为空");
    }

    public void setApiRunAround(ApiRunAround apiRunAround) {
        this.apiRunAround = apiRunAround;
    }

    public void run(){
        EasyApiBaseConfiguration configuration = all.getConfiguration();
        configuration.getUnique();
        if(!configuration.isEnable()){
            return ;
        }
        try {
            initBase(all);
            initDataSource(all);
            initService();
            apiRunAround.dataSourceInitialized();
            createDB();
            apiRunAround.dbCleared();
            DBProjectEntity project = initProject();
            apiRunAround.projectInitialized(project);
            clearDB();
            apiRunAround.dbCleared();
            //文档生成前置处理
            apiRunAround.beforeCreateApi();
            DocCreate instance = DocCreate.getInstance(all,appContext);
            if(configuration.isRescan()){
                instance.createApi();
            }
            //文档生成后置处理
            apiRunAround.apiCreated();
        }catch (Exception e) {
            throw new RuntimeException("接口文档生成失败",e);
        }
    }

    private void initService(){
        projectService = new ProjectService();
        userService = new UserService();
        dictService = new DictService();
        userTemporaryAuthService = new UserTemporaryAuthService();
        hostService = new HostService();
        moduleService = new ModuleService();
        roleService = new RoleService();
        roleAuthService = new RoleAuthService();
        roleProjectService = new RoleProjectService();
        controllerService = new ControllerService();
        interfaceService = new InterfaceService();
        interfaceParamService = new InterfaceParamService();
        simpleUnitService = new SimpleUnitService();
    }

    private void initBase(AllConfiguration all){
        ProjectContext.allConfiguration = all;
        Set<String> projectSourcePaths = all.getConfiguration().getProjectSourcePaths();
        String filePost = File.separator + "src" + File.separator + "main" + File.separator + "java" + File.separator;
        if(projectSourcePaths.isEmpty()){
            try {
                String path = new File("").getCanonicalPath();
                projectSourcePaths.add(path+ filePost);
            } catch (IOException e) {
                throw new ApidocException(e);
            }
        }
        Iterator<String> iterator = projectSourcePaths.iterator();
        Set<String> copyPaths = new HashSet<>();
        while (iterator.hasNext()){
            String next = iterator.next();
            if(!next.endsWith(File.separator)){
                next += File.separator;
            }
            if(!next.endsWith(filePost)){
                next += filePost;
            }
            copyPaths.add(next);
        }
        projectSourcePaths = copyPaths;
        ProjectContext.projectBasePath = projectSourcePaths;
        ProjectContext.controllerOperator = new ControllerReaderProvider();
        ProjectContext.interfaceOperator = new InterfaceReaderProvider();
        ProjectContext.beanOperator = new BeanReaderProvider();
        ProjectContext.mockTemplateOperator = new MockReaderProvider();
        ProjectContext.requestOperator = new RequestReaderProvider();
        ProjectContext.responseOperator = new ResponseReaderProvider();

        EasyApiGlobalResponseConfiguration globalResponse = all.getGlobalResponseConfiguration();
        if(globalResponse.isEnable()){
            if(globalResponse.getReturnType()!=null && !StringUtil.isEmpty(globalResponse.getFieldName())){
                //保存全局统一返回类型
                ProjectContext.globalResponseParams = getGlobalParams(globalResponse.getReturnType(),globalResponse.getIgnoreFields());
            }
        }

        EasyApiGlobalRequestConfiguration globalRequest = all.getGlobalRequestConfiguration();
        if(globalRequest.isEnable()){
            if(globalRequest.getRequestType()!=null && !StringUtil.isEmpty(globalRequest.getFieldName())){
                //保存全局统一返回类型
                ProjectContext.globalRequestParams = getGlobalParams(globalRequest.getRequestType(),globalRequest.getIgnoreFields());
            }
        }
    }

    private List<MethodParam> getGlobalParams(Class<?> type,Set<String> ignoreFields){
        List<MethodParam> globalParams = new ArrayList<>();
        Field[] fields = type.getDeclaredFields();
        for (Field field : fields) {
            boolean match = ignoreFields.stream().anyMatch(ignore -> ignore.equals(field.getName()));
            if(match){
                continue;
            }
            MethodParam param = new MethodParam();
            param.setParamType(field.getGenericType());
            param.setParamName(field.getName());
            globalParams.add(param);
        }
        return globalParams;
    }

    private void initDataSource(AllConfiguration all){
        EasyApiDataConfiguration dataConfiguration = all.getDataConfiguration();
        //处理数据库
        if(StringUtil.isEmpty(dataConfiguration.getDriverClassName())){
            dataConfiguration.setDriverClassName("org.h2.Driver");
        }
        if(StringUtil.isEmpty(dataConfiguration.getUrl())){
            String dbFilePath = all.getDataConfiguration().getDbFilePath();
            if(dbFilePath.endsWith(File.separator)){
                dbFilePath = dbFilePath.substring(0,dbFilePath.length()-1);
            }
            dataConfiguration.setUrl("jdbc:h2:"+dbFilePath+"-easyapi;AUTO_SERVER=TRUE");
        }
        if(StringUtil.isEmpty(dataConfiguration.getUserName())){
            dataConfiguration.setUserName("easyapi");
        }
        if(StringUtil.isEmpty(dataConfiguration.getPassword())){
            dataConfiguration.setPassword("123456");
        }
        List<DBTables> mappersClasses = new ArrayList<>(Arrays.asList(DBTables.values()));
        mappersClasses.removeIf(table -> table.getMapper() == null);
        List<Class<?>> collect = mappersClasses.stream().map(DBTables::getMapper).collect(Collectors.toList());
        Class<?>[] mappers = new Class[collect.size()];
        collect.toArray(mappers);
        if(dataConfiguration.getDataSource() != null){
            //使用用户提供的连接池
            MybatisUtil.init(dataConfiguration.getDataSource(),all.getConfiguration().isShowSql(),mappers);
        }else{
            //创建连接池
            DataSource dataSource = dataSource(dataConfiguration.getUrl()
                    , dataConfiguration.getDriverClassName()
                    , dataConfiguration.getUserName()
                    , dataConfiguration.getPassword());
            MybatisUtil.init(dataSource,all.getConfiguration().isShowSql(), mappers);
        }
    }

    private void createDB(){
        SqlExecMapper mapper = MybatisUtil.getMapper(SqlExecMapper.class);
        //重新初始化数据库表，如果不存在则创建
        for (DBTables table : DBTables.values()) {
            mapper.execUpdate(table.tableDDL());
            table.replenishField();
        }
    }

    //清理db数据
    private void clearDB(){
        EasyApiBaseConfiguration configuration = all.getConfiguration();
        SqlExecMapper mapper = MybatisUtil.getMapper(SqlExecMapper.class);
        //如果配置了删除构建，则删除相关表
        if(configuration.isDropAll()){
            if(ProjectContext.currentProject.getDefaultStatus() == 1){
                for (DBTables table : DBTables.values()) {
                    mapper.execUpdate(table.dropSql());
                }
            }else{
                //只清理该项目的东西
                controllerService.remove(Wrappers.lambdaQuery(DBModuleControllerEntity.class).eq(DBModuleControllerEntity::getProjectId,ProjectContext.currentProjectId));
                interfaceService.remove(Wrappers.lambdaQuery(DBModuleInterfaceEntity.class).eq(DBModuleInterfaceEntity::getProjectId,ProjectContext.currentProjectId));
                interfaceParamService.remove(Wrappers.lambdaQuery(DBInterfaceParamEntity.class).eq(DBInterfaceParamEntity::getProjectId,ProjectContext.currentProjectId));
                simpleUnitService.remove(Wrappers.lambdaQuery(DBSimpleUnitEntity.class).eq(DBSimpleUnitEntity::getProjectId,ProjectContext.currentProjectId));
            }
        }
        if(configuration.isDropUsers()){
            if(ProjectContext.currentProject.getDefaultStatus() == 1){
                for (DBTables table : DBTables.getByClassify(DBTableClassify.user)) {
                    mapper.execUpdate(table.clearSql());
                }
            }
        }
        if(configuration.isDropGlobalSetting()){
            if(ProjectContext.currentProject.getDefaultStatus() == 1){
                for (DBTables table : DBTables.getByClassify(DBTableClassify.global)) {
                    mapper.execUpdate(table.clearSql());
                }
            }
        }
        if(configuration.isDropDoc()){
            if(ProjectContext.currentProject.getDefaultStatus() == 1){
                for (DBTables table : DBTables.getByClassify(DBTableClassify.doc)) {
                    mapper.execUpdate(table.clearSql());
                }
            }else{
                controllerService.remove(Wrappers.lambdaQuery(DBModuleControllerEntity.class).eq(DBModuleControllerEntity::getProjectId,ProjectContext.currentProjectId));
                interfaceService.remove(Wrappers.lambdaQuery(DBModuleInterfaceEntity.class).eq(DBModuleInterfaceEntity::getProjectId,ProjectContext.currentProjectId));
                interfaceParamService.remove(Wrappers.lambdaQuery(DBInterfaceParamEntity.class).eq(DBInterfaceParamEntity::getProjectId,ProjectContext.currentProjectId));
                simpleUnitService.remove(Wrappers.lambdaQuery(DBSimpleUnitEntity.class).eq(DBSimpleUnitEntity::getProjectId,ProjectContext.currentProjectId));
            }
        }
    }

    private DataSource dataSource(String url,String driverClassName,String username, String password){
        // 连接数据库
//        String URL = "jdbc:h2:file:"+ProjectContext.dbPath+ProjectContext.dbFileName+"-"+projectName;
//        String USER = "easyapi";
//        String PASSWORD = "123456";

        DruidDataSource druidDataSource = new DruidDataSource();
        druidDataSource.setUrl(url);
        druidDataSource.setUsername(username);
        druidDataSource.setPassword(password);
        druidDataSource.setDriverClassName(driverClassName);
        druidDataSource.setInitialSize(0);
        druidDataSource.setMaxActive(100);
        druidDataSource.setMaxWait(10000);
        druidDataSource.setMinIdle(20);
        druidDataSource.setValidationQuery("Select  'x' from DUAL");
        druidDataSource.setTestOnBorrow(false);
        druidDataSource.setTestOnReturn(false);
        druidDataSource.setTestWhileIdle(true);
        druidDataSource.setTimeBetweenEvictionRunsMillis(60000);
        druidDataSource.setMinEvictableIdleTimeMillis(25200000);
        druidDataSource.setRemoveAbandoned(true);
        druidDataSource.setRemoveAbandonedTimeout(1800);
        druidDataSource.setLogAbandoned(true);
        return druidDataSource;
    }

    /**
     * 初始化项目
     */
    private DBProjectEntity initProject(){
        DBProjectEntity query = projectService.getOne(Wrappers.lambdaQuery(DBProjectEntity.class).eq(DBProjectEntity::getProjectUnique, all.getConfiguration().getUnique()));
        if(query != null){
            //已经有项目了，不需要创建
            ProjectContext.currentProjectId = query.getId();
            ProjectContext.currentProjectName = query.getProjectName();
            ProjectContext.currentProject = query;
            return query;
        }
        int count = projectService.count();
        //创建项目
        DBProjectEntity project = addDefaultProject(count>0?0:1);
        //创建模块
        DBModuleEntity module = addDefaultModule(project);
        DBUserEntity admin = userService.getOne(
                Wrappers.lambdaQuery(DBUserEntity.class)
                        .eq(DBUserEntity::getAccount, "admin"));
        if(admin == null){
            addDefaultUsers(module);
        }else{
            admin.setAccount(all.getUserConfiguration().getAccount());
            admin.setPassword(StringUtil.toMD5(all.getUserConfiguration().getPassword()));
            userService.updateById(admin);
        }
        ProjectContext.currentProjectId = project.getId();
        ProjectContext.currentModuleId = module.getId();
        ProjectContext.currentProjectName = project.getProjectName();
        ProjectContext.currentModuleName = module.getName();
        ProjectContext.currentProject = project;

        DBDictEntity syncEnable = new DBDictEntity();
        syncEnable.setDictKey(DBDictEntity.syncStatusKey());
        syncEnable = dictService.getOne(Wrappers.lambdaQuery(syncEnable),false);
        if(syncEnable == null){
            syncEnable = new DBDictEntity();
            syncEnable.setDictKey(DBDictEntity.syncStatusKey());
            syncEnable.setDictVal("0");
            dictService.save(syncEnable);
        }

        DBDictEntity syncInterfaces = new DBDictEntity();
        syncInterfaces.setDictKey(DBDictEntity.syncInterfacesKey());
        syncInterfaces = dictService.getOne(Wrappers.lambdaQuery(syncInterfaces),false);
        if(syncInterfaces == null){
            syncInterfaces = new DBDictEntity();
            syncInterfaces.setDictKey(DBDictEntity.syncInterfacesKey());
            syncInterfaces.setDictVal("[]");
            dictService.save(syncInterfaces);
        }

        DBDictEntity syncInfo = new DBDictEntity();
        syncInfo.setDictKey(DBDictEntity.syncInfoKey());
        syncInfo = dictService.getOne(Wrappers.lambdaQuery(syncInfo),false);
        if(syncInfo == null){
            syncInfo = new DBDictEntity();
            syncInfo.setDictKey(DBDictEntity.syncInfoKey());
            syncInfo.setDictVal(JsonUtil.beanToJson(new OpenSyncDto()));
            dictService.save(syncInfo);
        }

        DBDictEntity syncSecret = new DBDictEntity();
        syncSecret.setDictKey(DBDictEntity.syncSecretKey());
        syncSecret = dictService.getOne(Wrappers.lambdaQuery(syncSecret),false);
        if(syncSecret == null){
            syncSecret = new DBDictEntity();
            syncSecret.setDictKey(DBDictEntity.syncSecretKey());
            syncSecret.setDictVal(all.getConfiguration().getSyncSecret());
            dictService.save(syncSecret);
        }

        all.getConfiguration().setSyncSecret(syncSecret.getDictVal());
        all.getConfiguration().setUnique(project.getProjectUnique());

        return project;
    }

    /**
     * 初始化项目
     */
    private DBProjectEntity addDefaultProject(Integer defaultStatus) {
        //初始化项目
        DBProjectEntity project = new DBProjectEntity();
        project.setDefaultStatus(defaultStatus);
        project.setTitle(all.getConfiguration().getProjectName());
        project.setProjectName(all.getConfiguration().getProjectName());
        project.setProjectUnique(all.getConfiguration().getUnique());
        project.setAppSecret(all.getConfiguration().getSyncSecret());
        projectService.save(project);
        return project;
    }

    /**
     * 添加默认用户
     */
    private void addDefaultUsers(DBModuleEntity module) {

        //添加管理员账户
        DBUserEntity user = new DBUserEntity();
        user.setAccount(all.getUserConfiguration().getAccount());
        user.setPassword(StringUtil.toMD5(all.getUserConfiguration().getPassword()));
        user.setDisable(0);
        user.setDescription("超级管理员");
        user.setNickName("超级管理员");
        user.setSuperAdminStatus(1);
        user.setRoleId(0L);
        user.setHidden(1);
        user.setProjectId(module.getProjectId());
        user.setCreateTime(System.currentTimeMillis());
        user.setUpdateTime(System.currentTimeMillis());
        userService.save(user);
        userTemporaryAuthService.binds(user, Stream.of(AuthModuleButtonsEnum.values()).map(AuthModuleButtonsEnum::getCode).collect(Collectors.toList()));

        //添加特殊超级管理员，该人员不可登陆，仅用作远程通讯
        DBUserEntity hiddenUser = new DBUserEntity();
        hiddenUser.setAccount("remote");
        hiddenUser.setPassword(StringUtil.toMD5("remote"));
        hiddenUser.setSuperAdminStatus(1);
        hiddenUser.setRoleId(0L);
        hiddenUser.setHidden(1);
        hiddenUser.setProjectId(module.getProjectId());
        hiddenUser.setCreateTime(System.currentTimeMillis());
        hiddenUser.setUpdateTime(System.currentTimeMillis());
        userService.save(hiddenUser);

        //添加测试账号角色
        DBRoleEntity testRole = new DBRoleEntity();
        testRole.setRoleName("测试人员");
        testRole.setDescription("仅提供测试相关的权限");
        testRole.setSuperAdminStatus(0);
        testRole.setProjectId(module.getProjectId());
        testRole.setCreateTime(System.currentTimeMillis());
        testRole.setUpdateTime(System.currentTimeMillis());
        roleService.save(testRole);

        //添加角色对应的项目
        DBRoleProjectEntity roleProject = new DBRoleProjectEntity();
        roleProject.setRoleId(testRole.getId());
        roleProject.setProjectId(module.getProjectId());
        roleProject.setCreateTime(System.currentTimeMillis());
        roleProject.setUpdateTime(System.currentTimeMillis());
        roleProjectService.save(roleProject);

        //添加角色对应的可操作功能
        List<Integer> defaultAuthCodes = AuthModule.getDefaultAuthCodes();
        List<DBRoleAuthEntity> auths = new ArrayList<>();
        for (Integer code : defaultAuthCodes) {
            DBRoleAuthEntity entity = new DBRoleAuthEntity();
            entity.setRoleId(testRole.getId());
            entity.setAuthCode(code);
            entity.setModuleId(module.getId());
            entity.setProjectId(module.getProjectId());
            entity.setCreateTime(System.currentTimeMillis());
            entity.setUpdateTime(System.currentTimeMillis());
            auths.add(entity);
        }
        roleAuthService.saveBatch(auths);

        //添加测试人员
        DBUserEntity test = new DBUserEntity();
        test.setAccount("test");
        test.setPassword(StringUtil.toMD5("test"));
        test.setSuperAdminStatus(0);
        test.setProjectId(module.getProjectId());
        test.setRoleId(testRole.getId());
        test.setNickName("测试账号");
        test.setDescription("测试人员账号");
        test.setDisable(0);
        test.setBindProjectIds(module.getProjectId().toString());
        userService.save(test);
        userTemporaryAuthService.binds(test,Stream.of(AuthModuleButtonsEnum.values()).filter(item -> item.getDefaultStatus()==1).map(AuthModuleButtonsEnum::getCode).collect(Collectors.toList()));
    }

    /**
     * 初始化项目
     */
    private DBModuleEntity addDefaultModule(DBProjectEntity project){
        //初始化项目模块
        DBModuleEntity module = new DBModuleEntity();
        module.setProjectId(project.getId());
        module.setName("默认模块");
        module.setDescription("初始化项目模块");
        module.setOutPackgeStatus(0);
        module.setDefaultStatus(project.getDefaultStatus());
        moduleService.save(module);
        if(project.getDefaultStatus() == 0){
            return module;
        }

        //初始化项目请求环境
        DBModuleHostEntity mock = new DBModuleHostEntity();
        mock.setModuleId(module.getId());
        mock.setProjectId(project.getId());
        mock.setName("mock");
        mock.setDescription("mock请求,返回mock结果");
        mock.setDefaultStatus(project.getDefaultStatus());
        mock.setDisable(0);
        mock.setCanDelete(0);
        String port = appContext.getEnvironment().getProperty("server.port");
        port = port==null?"8080":port;
        mock.setHost("mock");
        hostService.save(mock);

        //添加本地环境
        DBModuleHostEntity local = new DBModuleHostEntity();
        local.setProjectId(project.getId());
        local.setModuleId(module.getId());
        local.setName("本地环境");
        local.setDescription("本机环境");
        local.setDefaultStatus(0);
        local.setDisable(0);
        mock.setCanDelete(1);
        String address = "localhost";
        try {
            address = InetAddress.getLocalHost().getHostAddress();
        } catch (UnknownHostException ignored) {
        }
        local.setHost("http://"+address+":"+port);
        hostService.save(local);
        return module;
    }
}
