package cn.easyutil.easyapi.interview.controller;

import cn.easyutil.easyapi.content.ProjectContext;
import cn.easyutil.easyapi.entity.common.AccessAuth;
import cn.easyutil.easyapi.entity.common.ApidocComment;
import cn.easyutil.easyapi.entity.db.auth.DBDictEntity;
import cn.easyutil.easyapi.entity.db.auth.DBProjectEntity;
import cn.easyutil.easyapi.entity.db.doc.DBArticleEntity;
import cn.easyutil.easyapi.entity.db.doc.DBInterfaceParamEntity;
import cn.easyutil.easyapi.entity.db.doc.DBModuleControllerEntity;
import cn.easyutil.easyapi.entity.db.doc.DBModuleInterfaceEntity;
import cn.easyutil.easyapi.entity.db.unit.DBSimpleUnitEntity;
import cn.easyutil.easyapi.exception.ApidocException;
import cn.easyutil.easyapi.interview.dto.IdDto;
import cn.easyutil.easyapi.interview.dto.OpenSyncDto;
import cn.easyutil.easyapi.interview.dto.TextDto;
import cn.easyutil.easyapi.interview.entity.ResponseBody;
import cn.easyutil.easyapi.interview.entity.UnifiedAccessDto;
import cn.easyutil.easyapi.interview.vo.*;
import cn.easyutil.easyapi.logic.run.DocInterfaceBean;
import cn.easyutil.easyapi.mybatis.service.*;
import cn.easyutil.easyapi.task.SyncBody;
import cn.easyutil.easyapi.task.SyncTask;
import cn.easyutil.easyapi.util.AssertUtil;
import cn.easyutil.easyapi.util.JsonUtil;
import cn.easyutil.easyapi.util.StringUtil;
import cn.easyutil.easyapi.util.http.*;
import com.baomidou.mybatisplus.core.toolkit.Wrappers;
import org.springframework.beans.BeanUtils;
import org.springframework.boot.autoconfigure.condition.ConditionalOnMissingBean;
import org.springframework.util.CollectionUtils;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RestController;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.function.Function;
import java.util.stream.Collectors;

@RestController
@ConditionalOnMissingBean(name = "easyapiProjectController")
@RequestMapping("/easyapi/doc/project")
public class ProjectController {

    private final ProjectService projectService = new ProjectService();

    private final ArticleService articleService = new ArticleService();

    private final DictService dictService = new DictService();

    private final ControllerService controllerService = new ControllerService();

    private final InterfaceService interfaceService = new InterfaceService();

    private final InterfaceParamService interfaceParamService = new InterfaceParamService();

    private final SimpleUnitService simpleUnitService = new SimpleUnitService();

    @ApidocComment("修改同步密钥")
    @PostMapping("/updateSecret")
    @AccessAuth(code = 1000)
    public void updateSecret(@RequestBody TextDto dto){
        AssertUtil.isEmpty(dto.getText(),"密钥不能为空");
        DBDictEntity entity = new DBDictEntity();
        entity.setDictKey(DBDictEntity.syncSecretKey());
        entity = dictService.getOne(Wrappers.lambdaQuery(entity),false);
        entity.setDictVal(dto.getText());
        dictService.updateById(entity);
        ProjectContext.allConfiguration.getConfiguration().setSyncSecret(dto.getText());
    }

    @ApidocComment("获取同步信息")
    @PostMapping("/getSyncInfo")
    @AccessAuth(code = 1000)
    public GetSyncInfoVo getSyncInfo(){
        String secret = ProjectContext.allConfiguration.getConfiguration().getSyncSecret();
        DBDictEntity status = dictService.getByKey(DBDictEntity.syncStatusKey());
        DBDictEntity interfaces = dictService.getByKey(DBDictEntity.syncInterfacesKey());
        DBDictEntity info = dictService.getByKey(DBDictEntity.syncInfoKey());
        GetSyncInfoVo vo = new GetSyncInfoVo();
        vo.setSecret(secret);
        vo.setStatus(Integer.parseInt(status.getDictVal()));
        vo.setSize(JsonUtil.jsonToList(interfaces.getDictVal(),Long.class).size());
        vo.setInfo(JsonUtil.jsonToBean(info.getDictVal(),OpenSyncDto.class));
        //获取默认项目
        DBProjectEntity project = new DBProjectEntity();
        project.setDefaultStatus(1);
        project = projectService.getOne(Wrappers.lambdaQuery(project),false);
        vo.setProjectName(project.getProjectName());
        vo.setProjectSecret(project.getAppSecret());
        vo.setProjectUnique(project.getProjectUnique());
        return vo;
    }

    @ApidocComment("获取项目列表")
    @PostMapping("/getProjects")
    @AccessAuth(code = 1000)
    public List<GetProjectsVo> getProjects(){
        List<GetProjectsVo> vos = new ArrayList<>();
        List<DBProjectEntity> projects = projectService.list();
        for (DBProjectEntity project : projects) {
            GetProjectsVo vo = new GetProjectsVo();
            BeanUtils.copyProperties(project,vo);
            int size = interfaceService.count(
                    Wrappers.lambdaQuery(DBModuleInterfaceEntity.class)
                            .eq(DBModuleInterfaceEntity::getProjectId, project.getId())
            );
            vo.setInterfaceCount(size);
            vos.add(vo);
        }
        return vos;
    }

    @ApidocComment("修改项目信息")
    @PostMapping("/updateProjects")
    @AccessAuth(code = 1000)
    public void updateProjects(@RequestBody DBProjectEntity entity){
        AssertUtil.isNull(entity.getId(), "项目id不能为空");
        if(entity.getDefaultStatus()!=null && entity.getDefaultStatus()==1){
            DBProjectEntity query = new DBProjectEntity();
            query.setDefaultStatus(entity.getDefaultStatus());
            query = projectService.getOne(Wrappers.lambdaQuery(query));
            AssertUtil.isTrue(query!=null, "只能有一个默认项目");
        }
        entity.setTitle(entity.getProjectName());
        projectService.updateById(entity);
    }

    @ApidocComment("新增项目")
    @PostMapping("/addProject")
    @AccessAuth(code = 1000)
    public void addProject(@RequestBody DBProjectEntity entity){
        if(entity.getDefaultStatus()!=null && entity.getDefaultStatus()==1){
            DBProjectEntity query = new DBProjectEntity();
            query.setDefaultStatus(entity.getDefaultStatus());
            query = projectService.getOne(Wrappers.lambdaQuery(query));
            AssertUtil.isTrue(query!=null, "只能有一个默认项目");
        }
        projectService.save(entity);
    }

    @ApidocComment("保存文章列表")
    @PostMapping("/saveArticles")
    @AccessAuth(code = 1000)
    public void saveArticles(@RequestBody List<DBArticleEntity> articles){
        articleService.remove(Wrappers.lambdaQuery(DBArticleEntity.class).ge(DBArticleEntity::getId,1));
        List<DBArticleEntity> list = JsonUtil.jsonToList(JsonUtil.beanToJson(articles), DBArticleEntity.class);
        articleService.saveBatch(list);
    }

    @ApidocComment("删除项目")
    @PostMapping("/delProject")
    @AccessAuth(code = 1000)
    public void delProject(@RequestBody IdDto dto){
        projectService.removeById(dto.getId());
        //只清理该项目的东西
        controllerService.remove(Wrappers.lambdaQuery(DBModuleControllerEntity.class).eq(DBModuleControllerEntity::getProjectId,ProjectContext.currentProjectId));
        interfaceService.remove(Wrappers.lambdaQuery(DBModuleInterfaceEntity.class).eq(DBModuleInterfaceEntity::getProjectId,ProjectContext.currentProjectId));
        interfaceParamService.remove(Wrappers.lambdaQuery(DBInterfaceParamEntity.class).eq(DBInterfaceParamEntity::getProjectId,ProjectContext.currentProjectId));
        simpleUnitService.remove(Wrappers.lambdaQuery(DBSimpleUnitEntity.class).eq(DBSimpleUnitEntity::getProjectId,ProjectContext.currentProjectId));
    }

    @ApidocComment("获取默认项目信息")
    @PostMapping("/getInfo")
    @AccessAuth(code = 1000)
    public DBProjectEntity getInfo(){
        return projectService.getOne(
                Wrappers.lambdaQuery(DBProjectEntity.class)
                        .eq(DBProjectEntity::getDefaultStatus,1));
    }

    @ApidocComment("导入项目信息")
    @PostMapping("/importProject")
    @AccessAuth(code = 1000)
    public void importProject(@RequestBody ProjectExportData data){
        if(data == null){
            return ;
        }
        DBProjectEntity project = data.getProject();
        if(project == null){
            throw new ApidocException("缺少项目名称或标识");
        }
        if(StringUtil.isEmpty(project.getProjectName()) || StringUtil.isEmpty(project.getProjectUnique())){
            throw new ApidocException("缺少项目名称或标识");
        }
        DBProjectEntity query = projectService.getOne(Wrappers.lambdaQuery(DBProjectEntity.class).eq(DBProjectEntity::getProjectUnique, project.getProjectUnique()), false);
        if(query != null){
            project = query;
        }else{
            project.setId(null);
            project.setProjectId(null);
            projectService.save(project);
        }

        //开始更新文档
        List<ControllerExportData> controllers = data.getControllers();
        if(CollectionUtils.isEmpty(controllers)){
            return ;
        }
        for (ControllerExportData controller : controllers) {
            DBModuleControllerEntity c = controller.getController();
            if(c==null || StringUtil.isEmpty(c.getClassName())){
                continue;
            }
            DBModuleControllerEntity queryController = controllerService.getOne(Wrappers.lambdaQuery(DBModuleControllerEntity.class)
                    .eq(DBModuleControllerEntity::getProjectId, project.getId())
                    .eq(DBModuleControllerEntity::getClassName, c.getClassName())
            );
            if(queryController == null){
                c.setProjectId(project.getId());
                c.setId(null);
                controllerService.save(c);
            }else{
                c = queryController;
            }
            //同时保存接口
            List<InterfaceExportData> interfaces = controller.getInterfaces();
            if(CollectionUtils.isEmpty(interfaces)){
                continue;
            }
            for (InterfaceExportData anInterface : interfaces) {
                DBModuleInterfaceEntity interfaceBean = anInterface.getInterfaceBean();
                if(interfaceBean==null || StringUtil.isEmpty(interfaceBean.getInterfaceUnique())){
                    continue;
                }
                DBModuleInterfaceEntity queryInterface = interfaceService.getOne(Wrappers.lambdaQuery(DBModuleInterfaceEntity.class)
                        .eq(DBModuleInterfaceEntity::getControllerId, c.getId())
                        .eq(DBModuleInterfaceEntity::getInterfaceUnique, interfaceBean.getInterfaceUnique())
                );
                if(queryInterface == null){
                    interfaceBean.setControllerId(c.getId());
                    interfaceBean.setId(null);
                    interfaceBean.setProjectId(project.getId());
                    interfaceService.save(interfaceBean);
                    DBInterfaceParamEntity param = anInterface.getRequestParam();
                    if(param != null){
                        param.setId(null);
                        param.setProjectId(project.getId());
                        param.setInterfaceId(interfaceBean.getId());
                        interfaceParamService.save(param);
                    }
                    DBSimpleUnitEntity unit = anInterface.getUnit();
                    if(unit != null){
                        unit.setId(null);
                        unit.setInterfaceUnique(interfaceBean.getInterfaceUnique());
                        unit.setProjectId(project.getId());
                        simpleUnitService.save(unit);
                    }
                }
            }
        }

    }

    @ApidocComment("获取项目导出数据")
    @PostMapping("/getProjectExportData")
    @AccessAuth(code = 0)
    public ProjectExportData getProjectExportData(@RequestBody IdDto dto){
        DBProjectEntity project = projectService.getById(dto.getId());
        if(project == null){
            return null;
        }
        ProjectExportData result = new ProjectExportData();
        result.setProject(project);
        //查询全部controller
        List<DBModuleControllerEntity> controllers = controllerService.list(Wrappers.lambdaQuery(DBModuleControllerEntity.class).eq(DBModuleControllerEntity::getProjectId, dto.getId()));
        if(CollectionUtils.isEmpty(controllers)){
            return result;
        }
        List<ControllerExportData> controllerData = new ArrayList<>();
        result.setControllers(controllerData);
        for (DBModuleControllerEntity controller : controllers) {
            ControllerExportData data = new ControllerExportData();
            data.setController(controller);
            controllerData.add(data);
            //查询controller下的全部接口
            List<DBModuleInterfaceEntity> interfaceEntities = interfaceService.list(Wrappers.lambdaQuery(DBModuleInterfaceEntity.class).eq(DBModuleInterfaceEntity::getControllerId, controller.getId()));
            if(CollectionUtils.isEmpty(interfaceEntities)){
                continue;
            }
            //查询全部接口参数
            Map<Long, DBInterfaceParamEntity> params = interfaceParamService.list(Wrappers.lambdaQuery(DBInterfaceParamEntity.class)
                    .in(DBInterfaceParamEntity::getInterfaceId, interfaceEntities.stream().map(DBModuleInterfaceEntity::getId).collect(Collectors.toList()))
            ).stream().collect(Collectors.toMap(DBInterfaceParamEntity::getInterfaceId, Function.identity()));

            //查询全部接口最近测试历史
            Map<String, DBSimpleUnitEntity> units = simpleUnitService.list(Wrappers.lambdaQuery(DBSimpleUnitEntity.class)
                    .in(DBSimpleUnitEntity::getInterfaceUnique, interfaceEntities.stream().map(DBModuleInterfaceEntity::getInterfaceUnique).collect(Collectors.toList()))
                    .eq(DBSimpleUnitEntity::getType,0)
            ).stream().collect(Collectors.toMap(DBSimpleUnitEntity::getInterfaceUnique, Function.identity()));

            List<InterfaceExportData> interfaces = new ArrayList<>();
            for (DBModuleInterfaceEntity interfaceEntity : interfaceEntities) {
                InterfaceExportData bean = new InterfaceExportData();
                bean.setInterfaceBean(interfaceEntity);
                bean.setRequestParam(params.get(interfaceEntity.getId()));
                bean.setUnit(units.get(interfaceEntity.getInterfaceUnique()));
                interfaces.add(bean);
            }
            data.setInterfaces(interfaces);
        }

        return result;
    }

    @ApidocComment("测试是否可以进行同步")
    @PostMapping("/syncTest")
    @AccessAuth(code = 0)
    public void syncTest(OpenSyncDto dto){
        AssertUtil.isEmpty(dto.getSecret(),"缺少密钥信息");
        AssertUtil.isEmpty(dto.getProjectUnique(),"缺少项目标识");
        AssertUtil.isEmpty(dto.getProjectName(),"缺少项目名称");
        AssertUtil.isTrue(!dto.getSecret().equals(ProjectContext.allConfiguration.getConfiguration().getSyncSecret()),"密钥错误");
    }

    @ApidocComment("开启文档同步")
    @PostMapping("/openSync")
    @AccessAuth(code = 1000)
    public void openSync(@RequestBody OpenSyncDto dto){
        AssertUtil.isEmpty(dto.getSecret(),"缺少密钥信息");
        AssertUtil.isEmpty(dto.getAddress(),"缺少目标服务器地址");
        UnifiedAccessDto uDto = new UnifiedAccessDto();
        uDto.setApi("project::syncTest");
        uDto.setBody(JsonUtil.beanToJson(dto));
        HttpOperation operation = HttpUtilFactory.create(HttpImplEnum.httpClient);
        operation.getHttpReq().setUrl(dto.getAddress()+"/easyapi/doc/unified/post");
        operation.parameter(JsonUtil.beanToJson(uDto));
        operation.method(HttpMethod.POST);
        HttpRes httpRes = operation.doUrl();
        if(!httpRes.getResponseCode().equals(200) || StringUtil.isEmpty(httpRes.getResponseMsg())){
            throw new ApidocException("目标服务器连接异常");
        }
        ResponseBody body = JsonUtil.jsonToBean(httpRes.getResponseMsg(),ResponseBody.class);
        if(!body.getCode().equals(200)){
            throw new ApidocException(body.getRemark());
        }
        DBProjectEntity project = projectService.getOne(
                Wrappers.lambdaQuery(DBProjectEntity.class)
                        .eq(DBProjectEntity::getDefaultStatus, 1));
        int size = interfaceService.count(
                Wrappers.lambdaQuery(DBModuleInterfaceEntity.class)
                        .eq(DBModuleInterfaceEntity::getProjectId, project.getId())
        );
        dto.setTotalCount(size);
        dto.setAddress(dto.getAddress());
        dictService.updateByKey(DBDictEntity.syncStatusKey(),"1");
        dictService.updateByKey(DBDictEntity.syncInterfacesKey(),"[]");
        dictService.updateByKey(DBDictEntity.syncInfoKey(),JsonUtil.beanToJson(dto));
        SyncTask.start();
    }

    @ApidocComment("接收远程来的同步")
    @PostMapping("/receiveSync")
    @AccessAuth(code = 0)
    public void receiveSync(@RequestBody SyncBody body){
        DBModuleControllerEntity controller = body.getController();
        DBModuleInterfaceEntity interfaces = body.getInterfaces();
        DBInterfaceParamEntity params = body.getParams();
        String secret = body.getSecret();
        String unique = body.getUnique();
        String name = body.getName();
        AssertUtil.isTrue(!secret.equals(ProjectContext.allConfiguration.getConfiguration().getSyncSecret()),"密钥错误");
        //先判断项目是否存在，不存在则创建项目
        List<DBProjectEntity> projects = projectService.list();
        DBProjectEntity project = null;
        for (DBProjectEntity p : projects) {
            if(p.getProjectUnique().equals(unique)){
                project = p;
                break;
            }
        }
        if(project == null){
            project = new DBProjectEntity();
            project.setProjectUnique(unique);
            project.setProjectName(name);
            project.setTitle(name);
            projectService.save(project);
        }

        //判断controller是否存在，根据className作为唯一标识
        DBModuleControllerEntity queryController = new DBModuleControllerEntity();
        queryController.setProjectId(project.getId());
        queryController.setClassName(controller.getClassName());
        queryController = controllerService.getOne(Wrappers.lambdaQuery(queryController));
        if(queryController == null){
            queryController = new DBModuleControllerEntity();
            BeanUtils.copyProperties(controller,queryController);
            queryController.setId(null);
            queryController.setProjectId(project.getId());
            controllerService.save(queryController);
        }

        //判断接口是否存在
        DBModuleInterfaceEntity queryInterfaceEntity = new DBModuleInterfaceEntity();
        queryInterfaceEntity.setProjectId(project.getId());
        queryInterfaceEntity.setControllerId(queryController.getId());
        queryInterfaceEntity.setInterfaceUnique(interfaces.getInterfaceUnique());
        queryInterfaceEntity = interfaceService.getOne(Wrappers.lambdaQuery(queryInterfaceEntity));
        if(queryInterfaceEntity == null){
            queryInterfaceEntity = new DBModuleInterfaceEntity();
            BeanUtils.copyProperties(interfaces,queryInterfaceEntity);
            queryInterfaceEntity.setId(null);
            queryInterfaceEntity.setProjectId(project.getId());
            queryInterfaceEntity.setControllerId(queryController.getId());
            interfaceService.save(queryInterfaceEntity);
            //最后更新接口入参和返回参数
            params.setId(null);
            params.setInterfaceId(queryInterfaceEntity.getId());
            interfaceParamService.save(params);
        }
    }
}
