package cn.easyutil.easyapi.util;

import org.springframework.web.bind.annotation.*;

import java.lang.annotation.Annotation;
import java.lang.annotation.Documented;
import java.lang.annotation.Retention;
import java.lang.annotation.Target;
import java.lang.reflect.Field;
import java.lang.reflect.InvocationHandler;
import java.lang.reflect.Method;
import java.lang.reflect.Proxy;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;

/**
 * @Description: SpringBoot 相关工具类
 * @Author: peng.liu
 * @CreateDate: 2018/7/25 1:17
 */
public class SpringUtil {

    public static String getRequestPath(Class clazz){
        Annotation[] annotations = clazz.getAnnotations();
        return getRequestPath(annotations);
    }

    public static String getRequestPath(Method method){
        Annotation[] annotations = method.getAnnotations();
        return getRequestPath(annotations);
    }

    public static String getRequestPath(Annotation[] annotations){
        if(annotations==null || annotations.length==0){
            return "";
        }
        for (Annotation annotation : annotations) {
            if(isAssignableFromAnnotation(annotation,RequestMapping.class)){
                String[] path = getAnnotationValue(annotation, "path", String[].class);
                if(path!=null && path.length > 0){
                    return path[0];
                }
                String name = getAnnotationValue(annotation, "name", String.class);
                if(!StringUtil.isEmpty(name)){
                    return name;
                }
                String[] value = getAnnotationValue(annotation, "value", String[].class);
                if(value!=null && value.length > 0){
                    return value[0];
                }
            }
        }
        return "";
    }

    /**
     * 获取注解的值
     * @param annotation    注解
     * @param key   注解属性(属性方法名)
     * @param parse 需要转换的类型
     * @return
     */
    public static <T>T getAnnotationValue(Annotation annotation, String key, Class<T> parse){
        InvocationHandler invocationHandler = Proxy.getInvocationHandler(annotation);
        try {
            Field field = invocationHandler.getClass().getDeclaredField("memberValues");
            field.setAccessible(true);
            Map<String, Object> memberValues = (Map<String, Object>) field.get(invocationHandler);
            if(memberValues==null || memberValues.isEmpty()){
                return null;
            }
            field.setAccessible(false);
            return (T) memberValues.get(key);
        } catch (Exception e) {
            return null;
        }
    }

    /**
     * 获取包含的另外一个注解
     * @param source    源注解
     * @param target    判断是否包含的注解
     * @return
     */
    public static Annotation getAssignableFromAnnotation(Annotation source,Class target){
        List<Annotation> exsit = new ArrayList<>();
        isAssignableFromAnnotation(source,target,exsit);
        if(exsit.size() > 0){
            return exsit.get(0);
        }
        return null;
    }

    /**
     * 判断一个注解是否包含另外一个注解
     * @param source    源注解
     * @param target    判断是否包含的注解
     * @return  true:包含
     */
    public static boolean isAssignableFromAnnotation(Annotation source,Class target){
        List<Annotation> exsit = new ArrayList<>();
        isAssignableFromAnnotation(source,target,exsit);
        if(exsit.size() > 0){
            return true;
        }
        return false;
    }

    private static void isAssignableFromAnnotation(Annotation source,Class target, List<Annotation> exsit){
        if((source instanceof Target) || (source instanceof Documented) || (source instanceof Retention)){
            return ;
        }
        if(source.annotationType().isAssignableFrom(target)){
            exsit.add(source);
        }
        Annotation[] ans = source.annotationType().getAnnotations();
        if(ans==null || ans.length==0){
            return ;
        }
        for (Annotation an : ans) {
            isAssignableFromAnnotation(an,target,exsit);
        }
    }

    /**
     * 获取请求方式
     * 请求方式 0-全部  1-get  2-post 3-put  4-delete
     *
     * @param method 类
     * @return String 请求方式 eg: get 或者 get,put,post 或者 all(所有请求方式)
     * @see RequestMethod
     */
    public static RequestMethod getRequestMethod(Method method) {
        Annotation[] annotations = method.getAnnotations();
        //获取请求方法类型
        RequestMethod requestMethod = RequestMethod.POST;
        for (Annotation annotation : annotations){
            Annotation an = getAssignableFromAnnotation(annotation, RequestMapping.class);
            if(an != null){
                RequestMapping mapping = (RequestMapping) an;
                RequestMethod[] methods = mapping.method();
                if(methods==null || methods.length==0){
                    break;
                }
                requestMethod = methods[0];
                break;
            }
        }
        return requestMethod;
    }
}
