package cn.easyutil.easyapi.filter.model;

import cn.easyutil.easyapi.entity.common.ApidocComment;
import cn.easyutil.easyapi.entity.common.BodyType;
import cn.easyutil.easyapi.entity.common.RenewType;
import cn.easyutil.easyapi.entity.db.doc.DBModuleInterfaceEntity;
import cn.easyutil.easyapi.filter.ApiExtra;
import cn.easyutil.easyapi.filter.ReadInterfaceApiFilter;
import cn.easyutil.easyapi.util.AnnotationUtil;
import cn.easyutil.easyapi.util.SpringUtil;
import cn.easyutil.easyapi.util.StringUtil;
import org.springframework.web.bind.annotation.*;
import org.springframework.web.multipart.MultipartFile;

import java.lang.annotation.Annotation;
import java.lang.reflect.Method;
import java.lang.reflect.Parameter;
import java.util.ArrayList;
import java.util.List;
import java.util.Optional;

/**
 * 读取接口方法相关的api
 */
public class DefaultReadInterfaceApi extends ReadInterfaceApiFilter {

    @Override
    public List<Method> readAllInterfaces(Class controller, ApiExtra extra) {
        List<Method> result = new ArrayList<Method>();
        Method[] methods = controller.getDeclaredMethods();
        for (Method method : methods) {
            Annotation[] ans = method.getDeclaredAnnotations();
            for (Annotation an : ans) {
                if (an instanceof RequestMapping) {
                    result.add(method);
                } else if (an instanceof GetMapping) {
                    result.add(method);
                } else if (an instanceof PostMapping) {
                    result.add(method);
                } else if (an instanceof PutMapping) {
                    result.add(method);
                } else if (an instanceof DeleteMapping) {
                    result.add(method);
                }
            }
        }
        return result;
    }

    @Override
    public boolean ignore(Method method, ApiExtra extra) {
        ApidocComment apidocComment = method.getDeclaredAnnotation(ApidocComment.class);
        if(apidocComment != null){
            boolean hidden = apidocComment.hidden();
            if(hidden){
                return true;
            }
            return apidocComment.ignore();
        }
        Annotation annotation = AnnotationUtil.getByAnnotationName("io.swagger.annotations.ApiOperation", method.getAnnotations());
        if(annotation != null){
            return (boolean) AnnotationUtil.getAnnotationValue(annotation, "hidden");
        }
        return false;
    }

    @Override
    public boolean show(Method method, ApiExtra extra) {
        return true;
    }

    @Override
    public RenewType renewType(Method method, ApiExtra extra) {
        return RenewType.increment;
    }

    @Override
    public BodyType bodyType(Method method, ApiExtra extra) {
        Parameter[] parameters = method.getParameters();
        if (parameters == null || parameters.length == 0) {
            // 无入参
            return BodyType.FORM;
        }
        for (Parameter parameter : parameters) {
            RequestBody annotation = parameter.getAnnotation(RequestBody.class);
            if (annotation != null) {
                return BodyType.JSON;
            }
            // 文件上传
            if (parameter.getType().getCanonicalName().equals(MultipartFile.class.getCanonicalName())
                    || parameter.getType().getCanonicalName().equals(MultipartFile[].class.getCanonicalName())) {
                return BodyType.UPLOAD;
            }
        }
        return BodyType.FORM;
    }

    @Override
    public String requestUrl(Method method, ApiExtra extra) {
        String baseUrl = Optional.ofNullable(extra.getDbController().getApiPath()).orElse("");
        baseUrl = parseUrl(baseUrl);
        String mapping = Optional.ofNullable(SpringUtil.getRequestPath(method)).orElse("");
        if (!StringUtil.isEmpty(mapping)) {
            mapping =  parseUrl(mapping);
        }
        return baseUrl+mapping;
    }

    @Override
    public List<String> customSearch(Method method, ApiExtra extra) {
        return null;
    }

    @Override
    public RequestMethod requestMethod(Method method, ApiExtra extra) {
        return SpringUtil.getRequestMethod(method);
    }

    @Override
    public String showName(Method method, ApiExtra extra) {
        Annotation methodApi = method.getDeclaredAnnotation(ApidocComment.class);
        if(methodApi != null){
            ApidocComment api = (ApidocComment) methodApi;
            if(!StringUtil.isEmpty( api.value())){
                return api.value();
            }
        }
        Annotation annotation = AnnotationUtil.getByAnnotationName("io.swagger.annotations.ApiOperation", method.getAnnotations());
        if(annotation != null){
            return (String) AnnotationUtil.getAnnotationValue(annotation, "value");
        }
        return "";
    }

    @Override
    public String description(Method method, ApiExtra extra) {
        ApidocComment apidocComment = method.getDeclaredAnnotation(ApidocComment.class);
        if(!StringUtil.isEmpty(apidocComment) && !StringUtil.isEmpty(apidocComment.requestNotes())){
            return apidocComment.requestNotes();
        }
        Annotation annotation = AnnotationUtil.getByAnnotationName("io.swagger.annotations.ApiParam", method.getAnnotations());
        if(annotation == null){
            return "";
        }
        Object value = AnnotationUtil.getAnnotationValue(annotation, "value");
        if(!StringUtil.isEmpty(value)){
            return (String) value;
        }
        Annotation apiOperation = AnnotationUtil.getByAnnotationName("io.swagger.annotations.ApiOperation", method.getAnnotations());
        if(apiOperation == null){
            return "";
        }
        Object notes = AnnotationUtil.getAnnotationValue(apiOperation, "notes");
        if(!StringUtil.isEmpty(notes)){
            return (String) notes;
        }
        Object tags = AnnotationUtil.getAnnotationValue(apiOperation, "tags");
        if(tags == null){
            return "";
        }
        String[] tgs = (String[]) tags;
        StringBuffer sb = new StringBuffer();
        for (int i = 0; i < tgs.length; i++) {
            sb.append(i+1+":"+tgs[i]+".");
        }
        return sb.toString();
    }

    @Override
    public String unique(Method method, ApiExtra extra) {
        return StringUtil.toMD5(method.toGenericString());
    }

    @Override
    public boolean verifyMockTag(String tag, DBModuleInterfaceEntity entity) {
        if(StringUtil.isEmpty(tag)){
            return true;
        }
        return tag.equalsIgnoreCase(entity.getUnique());
    }

    @Override
    public boolean enableReqPackage(Method method, ApiExtra extra) {
        return true;
    }

    @Override
    public boolean enableResPackage(Method method, ApiExtra extra) {
        return true;
    }

    /**
     * 完善请求路径
     *
     * @param requestUrl
     * @return
     */
    private String parseUrl(String requestUrl) {
        if (!requestUrl.startsWith("/")) {
            requestUrl = "/" + requestUrl;
        }
        if (requestUrl.endsWith("/")) {
            requestUrl = requestUrl.substring(0, requestUrl.length() - 1);
        }
        return requestUrl;
    }
}
