package cn.easyutil.easyapi.logic.unit;

import cn.easyutil.easyapi.configuration.EasyapiUnitConfiguration;
import cn.easyutil.easyapi.entity.common.BodyType;
import cn.easyutil.easyapi.entity.db.unit.DBComplexTestInfoEntity;
import cn.easyutil.easyapi.exception.ApidocException;
import cn.easyutil.easyapi.interview.entity.HttpEntity;
import cn.easyutil.easyapi.interview.entity.WebHttpFileReq;
import cn.easyutil.easyapi.interview.session.CurrentSession;
import cn.easyutil.easyapi.util.AssertUtil;
import cn.easyutil.easyapi.util.IOUtil;
import cn.easyutil.easyapi.util.JsonUtil;
import cn.easyutil.easyapi.util.StringUtil;
import cn.easyutil.easyapi.util.http.HttpFileReq;
import cn.easyutil.easyapi.util.http.HttpOperation;
import cn.easyutil.easyapi.util.http.HttpRes;
import cn.easyutil.easyapi.util.http.HttpUtilFactory;
import org.apache.http.entity.ContentType;

import java.io.IOException;
import java.io.OutputStream;
import java.util.*;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;

/**
 * 复杂单元测试
 */
public class ComplexTestProcess {

    //session保持标识
    private static final String sessionKeepKey = "sessionKeepKey";

    //执行并发的线程池
    private static ExecutorService executorService;


    /**
     * 根据请求结果组装测试详情数据
     * @param httpRes   请求结果
     * @param decodes   解密参数
     * @param successKey    判断依据的key
     * @param successOper   判断方式
     * @param successVal    判断依据的val
     * @return
     */
    public static DBComplexTestInfoEntity buildInfo(HttpRes httpRes, List<GroupDecode> decodes, String successKey, String successOper, String successVal){
        DBComplexTestInfoEntity entity = new DBComplexTestInfoEntity();
        entity.setSuccessKey(successKey);
        entity.setSuccessOper(successOper);
        entity.setSuccessVal(successVal);
        entity.setSuccessStatus(0);
        //是否使用http状态码判断成功与失败
        boolean judgeHttpStatus = false;
        if(StringUtil.isEmpty(successKey) || StringUtil.isEmpty(successOper) || StringUtil.isEmpty(successVal)){
            judgeHttpStatus = true;
        }
        if(httpRes == null){
            return entity;
        }
        entity.setHttpStatus(httpRes.getResponseCode());
        entity.setResponseHeaders(httpRes.getResponseHeaders()==null?"":JsonUtil.beanToJson(httpRes.getResponseHeaders()));
        String data = httpRes.getResponseMsg();
        entity.setResponseResult(data);
        if(judgeHttpStatus){
            entity.setSuccessStatus(booleanToInt(httpRes.getResponseCode() == 200));
            return entity;
        }
        if(StringUtil.isEmpty(data)){
            return entity;
        }
        Map<String, Object> map = JsonUtil.jsonToMap(data);
        Object successValue = map.get(successKey);
        if(StringUtil.isEmpty(successValue)){
            return entity;
        }
        //解密响应结果
        successValue = responseDecode(successValue.toString(),decodes);
        entity.setRealVal(successValue.toString());
        int compare = successValue.toString().compareTo(successVal);
        switch (OperEnum.of(successOper)){
            case lte:{
                entity.setSuccessStatus(booleanToInt(compare >= 0));
                return entity;
            }
            case gte:{
                entity.setSuccessStatus(booleanToInt(compare <= 0));
                return entity;
            }
            case ne:{
                entity.setSuccessStatus(booleanToInt(compare != 0));
                return entity;
            }
            case eq:{
                entity.setSuccessStatus(booleanToInt(compare == 0));
                return entity;
            }
        }
        return entity;
    }

    /**
     * 执行单元测试
     * @param entities
     * @return
     */
    public static List<UnitTestProcesser> startTest(List<HttpEntity> entities, EasyapiUnitConfiguration configuration,boolean sessionkeep){
        if(executorService == null){
            executorService = Executors.newWorkStealingPool(configuration.getMaxThreadCount() > 0 ? configuration.getMaxThreadCount() : 10);
        }
        if(entities==null || entities.isEmpty()){
            return new ArrayList<>();
        }
        CountDownLatch latch = new CountDownLatch(entities.size());
        List<UnitTestProcesser> result = Collections.synchronizedList(new ArrayList<>());
        //进行并发测试
        for (HttpEntity entity : entities) {
            executorService.submit(() -> {
                UnitTestProcesser processer = new UnitTestProcesser();
                processer.setDecodes(entity.getDecodes());
                processer.setEnty(entity);
                try {
                    processer.setHttpRes(doUrl(processer.getEnty(),sessionkeep));
                    processer.setSuccess(true);
                }catch (Exception e) {
                    processer.setSuccess(false);
                }
                result.add(processer);
                latch.countDown();
            });
        }
        try {
            latch.await();
        } catch (InterruptedException e) {
            throw new RuntimeException(e);
        }
        return result;
    }

    /**
     * 进行网络请求
     * @param entity
     * @return
     */
    public static HttpRes doUrl(HttpEntity entity,boolean sessionKeep){
        AssertUtil.isNull(entity, "参数缺失");
        AssertUtil.isNull(!StringUtil.isUrlPath(entity.getUrl()), "请求地址格式错误");
        AssertUtil.isNull(StringUtil.isEmpty(entity.getMethod()), "请求方式不能为空");
        AssertUtil.isNull(StringUtil.isEmpty(entity.getBodyType()), "请求体类型不能为空");
        HttpOperation operation = HttpUtilFactory.fastCreate(entity.getUrl(), entity.getMethod());
        BodyType bodyType = BodyType.parse(entity.getBodyType());
        AssertUtil.isTrue(bodyType == BodyType.UNKNOW, "请求体类型不存在");
        if(!StringUtil.isEmpty(entity.getJsonHeaders())){
            operation.headers(JsonUtil.jsonToMap(entity.getJsonHeaders()));
        }
        if(sessionKeep){
            //session保持
            if(CurrentSession.getRequest().getSession().getAttribute(sessionKeepKey) != null){
                operation.setTokenVal(CurrentSession.getRequest().getSession().getAttribute(sessionKeepKey).toString());
            }
        }
        operation.contentType(bodyType.getContentType());
        operation.parameter(entity.getJsonParams());
        switch (bodyType){
            case FORM:
            case UPLOAD:{
                operation.parameter(HttpOperation.jsonToForm(entity.getJsonParams()));
                break;
            }
        }
        if(entity.getFiles() != null){
            List<WebHttpFileReq> files = entity.getFiles();
            List<HttpFileReq> fileReqs = new ArrayList<>();
            for (WebHttpFileReq file : files) {
                HttpFileReq req = new HttpFileReq();
                req.setFileName(file.getFileName());
                req.setParamName(file.getParamName());
                req.setContent(StringUtil.base64ToByte(file.getBase64Content()));
                fileReqs.add(req);
            }
            operation.getHttpReq().setFileParams(fileReqs);
        }
        HttpRes httpRes = operation.doUrl();
        if(bodyType == BodyType.DOWNLOAD){
            try(OutputStream out = CurrentSession.getResponse().getOutputStream()){
                out.write(IOUtil.inputToByte(httpRes.getResponseInputStream()));
                out.flush();
                return httpRes;
            }catch (IOException e) {
                throw new ApidocException(e);
            }
        }
        httpRes.getResponseMsg();
        if(sessionKeep){
            //session保持
            if(!StringUtil.isEmpty(operation.getTokenVal())){
                CurrentSession.getRequest().getSession().setAttribute(sessionKeepKey, operation.getTokenVal());
            }
        }
        return httpRes;
    }


    /**
     * 对响应参数进行解密
     * @param data
     * @param decodes
     * @return
     */
    public static String responseDecode(String data,List<GroupDecode> decodes){
        if(StringUtil.isEmpty(data)){
            return data;
        }
        if(decodes == null){
            return data;
        }
        for (GroupDecode decode : decodes) {
            if(decode.getDecode()==null){
                continue;
            }
            switch (DecodeEnum.of(decode.getDecode())){
                case base64:{
                    data = StringUtil.base64Decode(data);
                    break;
                }
                case aes:{
                    if(decode.getKey()==null){
                        break;
                    }
                    data = StringUtil.AESDecode(data, decode.getKey());
                    break;
                }
                case des:{
                    if(decode.getKey()==null){
                        break;
                    }
                    data = StringUtil.DESDecode(data, decode.getKey());
                    break;
                }
            }
        }
        return data;
    }

    /**
     * boolean转int
     * @param boo
     * @return
     */
    public static Integer booleanToInt(boolean boo){
        return boo?1:0;
    }
}
