package cn.easyutil.easyapi.interview.controller;

import cn.easyutil.easyapi.content.ProjectContext;
import cn.easyutil.easyapi.datasource.bean.EasyApiBindSqlResult;
import cn.easyutil.easyapi.datasource.bean.EasyapiBindSQLExecuter;
import cn.easyutil.easyapi.entity.common.AccessAuth;
import cn.easyutil.easyapi.entity.common.ApidocComment;
import cn.easyutil.easyapi.entity.db.auth.DBRoleAuthEntity;
import cn.easyutil.easyapi.entity.db.auth.DBRoleEntity;
import cn.easyutil.easyapi.entity.db.auth.DBRoleProjectEntity;
import cn.easyutil.easyapi.entity.db.auth.DBUserEntity;
import cn.easyutil.easyapi.interview.dto.IdDto;
import cn.easyutil.easyapi.interview.dto.LoginDto;
import cn.easyutil.easyapi.interview.dto.SelectUsersDto;
import cn.easyutil.easyapi.interview.dto.UpdatePasswordDto;
import cn.easyutil.easyapi.interview.session.CurrentSession;
import cn.easyutil.easyapi.interview.session.SessionUser;
import cn.easyutil.easyapi.service.RoleAuthService;
import cn.easyutil.easyapi.service.RoleProjectService;
import cn.easyutil.easyapi.service.RoleService;
import cn.easyutil.easyapi.service.UserService;
import cn.easyutil.easyapi.util.AssertUtil;
import cn.easyutil.easyapi.util.StringUtil;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.autoconfigure.condition.ConditionalOnMissingBean;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RestController;

import javax.annotation.Resource;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

@RestController
@ConditionalOnMissingBean(name = "easyapiUserController")
@RequestMapping("/easyapi/doc/user")
public class UserController {

    @Resource(name = "easyapiUserService")
    private UserService userService;

    @Resource(name = "easyapiRoleService")
    private RoleService roleService;

    @Resource(name = "easyapiRoleAuthService")
    private RoleAuthService authService;

    @Resource(name = "easyapiRoleProjectService")
    private RoleProjectService roleProjectService;

    @ApidocComment("登陆接口")
    @PostMapping("/login")
    @AccessAuth(code = 1000)
    public DBUserEntity login(@RequestBody LoginDto dto){
        AssertUtil.isNull(StringUtil.isEmpty(dto.getAccount()), "用户名不能为空");
        AssertUtil.isNull(StringUtil.isEmpty(dto.getPassword()), "密码不能为空");
        DBUserEntity user = userService.getByAccount(dto.getAccount());
        AssertUtil.isNull(user, "用户名或密码错误");
        AssertUtil.isTrue(!user.getPassword().equals(StringUtil.toMD5(dto.getPassword())), "用户名或密码错误");
        AssertUtil.isTrue(user.getDisable()==1, "您已被禁用");
        //查询用户的角色
        SessionUser sessionUser = new SessionUser();
        BeanUtils.copyProperties(user, sessionUser);
        CurrentSession.loginSuccess(sessionUser);
        if(user.getSuperAdminStatus() == 1){
            return user;
        }
        //查询角色对应的项目列表
        DBRoleProjectEntity rp = new DBRoleProjectEntity();
        rp.setRoleId(user.getRoleId());
        List<DBRoleProjectEntity> pList = roleProjectService.list(rp);
        if(pList == null){
            return user;
        }
        Map<Long,List<Integer>> auths = new HashMap<>();
        for (DBRoleProjectEntity item : pList) {
            DBRoleAuthEntity ra = new DBRoleAuthEntity();
            ra.setRoleId(item.getRoleId());
            ra.setProjectId(item.getProjectId());
            List<DBRoleAuthEntity> list = authService.list(ra);
            if(list == null){
                continue;
            }
            auths.put(item.getProjectId(), list.stream().map(auth -> auth.getAuthCode()).collect(Collectors.toList()));
        }
        sessionUser.setProjectIds(auths);
        return user;
    }

    @ApidocComment("修改个人密码")
    @PostMapping("/updatePassword")
    @AccessAuth(code = 1000)
    public void updatePassword(@RequestBody UpdatePasswordDto dto){
        SessionUser user = CurrentSession.getCurrentUser();
        AssertUtil.isTrue(!user.getPassword().equals(StringUtil.toMD5(dto.getOldPassword())), "旧密码输入不正确");
        user.setPassword(dto.getNewPassword());
        DBUserEntity userEntity = new DBUserEntity();
        userEntity.setId(user.getId());
        userEntity.setPassword(StringUtil.toMD5(dto.getNewPassword()));
        userService.update(userEntity);
    }

    @ApidocComment("条件查询用户列表")
    @PostMapping("/selectUsers")
    @AccessAuth(code = 1000)
    public EasyApiBindSqlResult<DBUserEntity> selectUsers(@RequestBody SelectUsersDto dto){
        SessionUser currentUser = CurrentSession.getCurrentUser();
        if(currentUser.getSuperAdminStatus() == 0){
            Long currentProjectId = CurrentSession.getCurrentProjectId();
            dto.setProjectId(currentProjectId);
        }
        DBUserEntity query = new DBUserEntity();
        BeanUtils.copyProperties(dto, query);
        query.setHidden(0);
        EasyApiBindSqlResult page = userService.page(query, dto.getCurrentPage(), dto.getShowCount());
        return page;
    }

    @ApidocComment("修改用户信息")
    @PostMapping("/updateUser")
    @AccessAuth(code = 1000)
    public void updateUser(@RequestBody DBUserEntity user){
        AssertUtil.isNull(user.getId(), "要修改的用户id不能为空");
        DBUserEntity query = userService.getById(user.getId());
        AssertUtil.isNull(query, "用户不存在");
        if(!StringUtil.isEmpty(user.getAccount()) && !query.getAccount().equals(user.getAccount())){
            DBUserEntity byAccount = userService.getByAccount(user.getAccount());
            AssertUtil.isTrue(byAccount != null, "账号已存在");
        }
        if(!StringUtil.isEmpty(user.getPassword()) && !user.getPassword().equals(query.getPassword())){
            user.setPassword(StringUtil.toMD5(user.getPassword()));
        }
        userService.update(user);
    }

    @ApidocComment("删除用户信息")
    @PostMapping("/deleteUser")
    @AccessAuth(code = 1000)
    public void deleteUser(@RequestBody IdDto dto){
        userService.deleteById(dto.getId());
    }

    @ApidocComment("添加用户")
    @PostMapping("/addUser")
    @AccessAuth(code = 1000)
    public void addUser(@RequestBody DBUserEntity entity){
        AssertUtil.isNull(StringUtil.isEmpty(entity.getAccount()), "账号不能为空");
        DBUserEntity byAccount = userService.getByAccount(entity.getAccount());
        AssertUtil.isTrue(byAccount != null, "账号已存在");
        entity.setId(null);
        if(entity.getRoleId() == null){
            DBRoleEntity defaultRole = roleService.get(EasyapiBindSQLExecuter.build(new DBRoleEntity()).eq(DBRoleEntity::getDefaultRole, 1));
            entity.setRoleId(defaultRole.getId());
        }
        if(StringUtil.isEmpty(entity.getPassword())){
           entity.setPassword(StringUtil.toMD5("123456"));
        }else{
            entity.setPassword(StringUtil.toMD5(entity.getPassword()));
        }
        if(entity.getProjectId() == null){
            entity.setProjectId(ProjectContext.currentProjectId);
        }
        userService.insert(entity);
    }

}
