package cn.easyutil.easyapi.javadoc.reader;

import cn.easyutil.easyapi.util.StringUtil;
import com.github.javaparser.JavaParser;
import com.github.javaparser.ParseResult;
import com.github.javaparser.TokenRange;
import com.github.javaparser.ast.CompilationUnit;
import com.github.javaparser.ast.Modifier;
import com.github.javaparser.ast.NodeList;
import com.github.javaparser.ast.body.*;
import com.github.javaparser.ast.expr.*;
import com.github.javaparser.ast.type.ClassOrInterfaceType;

import java.io.File;
import java.io.FileInputStream;
import java.io.InputStream;
import java.util.*;
import java.util.stream.Collectors;

public class JavaSourceReader {

    //最终解析完的信息
    private ClassComment classComment = new ClassComment();

    public static JavaSourceReader builder(File file){
        try {
            if(!file.exists()){
                return new JavaSourceReader();
            }
            return builder(new FileInputStream(file));
        }catch (Exception e){
            return new JavaSourceReader();
        }
    }

    public static JavaSourceReader builder(InputStream in){
        JavaSourceReader reader = new JavaSourceReader();
        JavaParser parser = new JavaParser();
        ParseResult<CompilationUnit> parse = parser.parse(in);
        if(!parse.getCommentsCollection().isPresent()){
            return reader;
        }
        reader.parse(parse.getResult().get());
        return reader;
    }

    public void parse(CompilationUnit unit){
        classComment.setSource(unit.getTokenRange().get().toString());
        for (TypeDeclaration typeDec : unit.getTypes()) {
            if(typeDec instanceof ClassOrInterfaceDeclaration){
                parseClass((ClassOrInterfaceDeclaration) typeDec);
            }
            List<BodyDeclaration> members = typeDec.getMembers();
            if (members != null) {
                for (BodyDeclaration member : members) {
                    parseMember(member);
                }
            }
        }
    }

    private void parseClass(ClassOrInterfaceDeclaration declaration){
        classComment.setName(declaration.getNameAsString());
        classComment.setComment(declaration.getComment().isPresent()?declaration.getComment().get().getContent():"");
        classComment.setModifyers((declaration.getModifiers()==null || declaration.getModifiers().size() == 0)?"":declaration.getModifiers().get(0).toString());
        classComment.setExtendsClassName((declaration.getExtendedTypes()==null || declaration.getExtendedTypes().size() == 0)?"Object":declaration.getExtendedTypes().get(0).getName().asString());
        NodeList<ClassOrInterfaceType> implementedTypes = declaration.getImplementedTypes();
        if(implementedTypes!=null && implementedTypes.size() > 0){
            for (ClassOrInterfaceType implementedType : implementedTypes) {
                classComment.addImplClassNames(implementedType.getName().asString());
            }
        }
        classComment.addAllAnnotations(readAnnotations(declaration));
    }

    private void parseMember(BodyDeclaration member){
        //正式接收解析数据
        if(member instanceof MethodDeclaration){
            classComment.addMethod(readMethodComment((MethodDeclaration) member));
        }
        if(member instanceof FieldDeclaration){
            classComment.addField(readFieldComment((FieldDeclaration) member));
        }
    }

    private FieldComment readFieldComment(FieldDeclaration declaration){
        FieldComment comment = new FieldComment();
        comment.setSource(getSourceCode(declaration.getTokenRange()));
        comment.setComment(declaration.getComment().isPresent()?declaration.getComment().get().getContent():"");
        NodeList<Modifier> modifiers = declaration.getModifiers();
        if(modifiers!=null && modifiers.size() > 0){
            comment.setModifyers(modifiers.get(0).toString());
        }
        NodeList<VariableDeclarator> variables = declaration.getVariables();
        if(variables!=null && variables.size() > 0){
            comment.setName(variables.get(0).getName().asString());
        }
        comment.addAllAnnotations(readAnnotations(declaration));
        return comment;
    }

    private List<AnnotationComment> readAnnotations(NodeList<AnnotationExpr> annotations){
        List<AnnotationComment> comments = new ArrayList<>();
        for (AnnotationExpr annotation : annotations) {
            AnnotationComment comment = new AnnotationComment();
            comment.setAnnotationName(annotation.getNameAsString());
            comment.setAnnotationComment(annotation.getComment().isPresent()?annotation.getComment().get().getContent():"");
            comment.setAnnotationSource(annotation.getTokenRange().isPresent()?annotation.getTokenRange().get().toString():"");
            List<AnnotationPair> pairs = new ArrayList<>();
            if(annotation instanceof MarkerAnnotationExpr){
                AnnotationPair pair = new AnnotationPair();
                pair.setName("value");
                pair.setValue("");
                pairs.add(pair);
            }
            if(annotation instanceof SingleMemberAnnotationExpr){
                AnnotationPair pair = new AnnotationPair();
                pair.setName("value");
                Optional<TokenRange> tokenRange = ((SingleMemberAnnotationExpr) annotation).getMemberValue().getTokenRange();
                pair.setValue(getSourceCode(tokenRange));
                pairs.add(pair);
            }
            if(annotation instanceof NormalAnnotationExpr){
                NodeList<MemberValuePair> pairNodeList = ((NormalAnnotationExpr) annotation).getPairs();
                for (MemberValuePair valuePair : pairNodeList) {
                    AnnotationPair pair = new AnnotationPair();
                    pair.setName(valuePair.getNameAsString());
                    pair.setValue(valuePair.getValue().toString());
                    pairs.add(pair);
                }
            }
            comment.setPairs(pairs);
            comments.add(comment);
        }
        return comments;
    }

    private List<AnnotationComment> readAnnotations(BodyDeclaration declaration){
        NodeList<AnnotationExpr> annotations = declaration.getAnnotations();
        if(annotations==null || annotations.size()==0){
            return null;
        }
        return readAnnotations(annotations);
    }

    /**
     * 读取方法注释
     * @param declaration
     * @return
     */
    private MethodComment readMethodComment(MethodDeclaration declaration){
        MethodComment comment = new MethodComment();
        comment.setSource(getSourceCode(declaration.getTokenRange()));
        String methodComment = declaration.getComment().isPresent()?declaration.getComment().get().getContent():"";
        comment.setComment(methodComment);
        comment.setReturnType(declaration.getType().asString());
        NodeList<Parameter> parameters = declaration.getParameters();
        if(!StringUtil.isEmpty(methodComment) && methodComment.contains("@return")){
            comment.setReturnComment(methodComment.substring(methodComment.indexOf("@return")+7).trim());
        }
        if(parameters!=null && parameters.size() > 0){
            Map<String, String> fieldCommentByMethod = getFieldCommentByMethod2(methodComment);
            for (Parameter parameter : parameters) {
                SourceParameterComment spc = new SourceParameterComment();
                spc.setName(parameter.getName().asString());
                spc.setComment(parameter.getComment().isPresent()?parameter.getComment().get().getContent():"");
                if(StringUtil.isEmpty(spc.getComment())){
                    spc.setComment(fieldCommentByMethod.get(spc.getName()));
                }
                spc.setModifyers((parameter.getModifiers()==null || parameter.getModifiers().isEmpty())?"":parameter.getModifiers().get(0).toString());
                spc.addAllAnnotations(readAnnotations(parameter.getAnnotations()));
                comment.addParameter(spc);
            }
        }
        NodeList<Modifier> modifiers = declaration.getModifiers();
        if(modifiers!=null && modifiers.size() > 0){
            comment.setModifyers(modifiers.get(0).toString());
        }
        comment.setName(declaration.getName().asString());
        comment.addAllAnnotations(readAnnotations(declaration));
        return comment;
    }

    private Map<String,String> getFieldCommentByMethod2(String methodComment){
        Map<String,String> result = new HashMap<>();
        if(StringUtil.isEmpty(methodComment)){
            return result;
        }
        String splitComment = methodComment;
        while (splitComment.contains("@param")){
            String val = splitComment.substring(splitComment.indexOf("@param")+6).trim();
            String key = null;
            if(val.contains("@")){
                key = val.substring(0,val.indexOf("@"));
                splitComment = val.substring(key.length());
            }else{
                splitComment = "";
                key = val;
            }
            if(!key.contains(" ")){
                continue;
            }
            String fieldName = null;
            List<String> fieldComment = new ArrayList<>();
            key = key.replaceAll("\\r\\n","")
                    .replaceAll("\\n", "")
                    .replaceAll("\\*","")
                    .trim();
            String[] s1 = key.split(" ");
            StringBuffer sb = new StringBuffer();
            for (String s2 : s1) {
                if(StringUtil.isEmpty(s2)){
                    continue;
                }
                if(StringUtil.isEmpty(fieldName)){
                    fieldName = s2;
                    continue;
                }
                fieldComment.add(s2);
            }
            if(fieldComment.size() > 0){
                result.put(fieldName,fieldComment.stream().collect(Collectors.joining(" ")));
            }
        }
        return result;
    }

    /**
     * 获取源代码
     * @param range
     * @return
     */
    private String getSourceCode(Optional<TokenRange> range){
        if(range == null){
            return "";
        }
        if(range.isPresent()){
            return range.get().toString();
        }
        return "";
    }

    public ClassComment getComment(){
        return classComment;
    }
}
