package cn.easyproject.easymonitor.sender.mail;

import java.io.File;
import java.io.IOException;
import java.util.Arrays;
import java.util.Date;
import java.util.HashMap;
import java.util.Map;
import java.util.Properties;

import javax.mail.MessagingException;
import javax.mail.internet.MimeMessage;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.mail.MailException;
import org.springframework.mail.javamail.JavaMailSenderImpl;
import org.springframework.mail.javamail.MimeMessageHelper;
import org.springframework.ui.freemarker.FreeMarkerTemplateUtils;

import cn.easyproject.easymonitor.configuration.MonitorConfiguration;
import cn.easyproject.easymonitor.sender.Sender;
import freemarker.cache.ClassTemplateLoader;
import freemarker.cache.FileTemplateLoader;
import freemarker.cache.MultiTemplateLoader;
import freemarker.cache.TemplateLoader;
import freemarker.template.Configuration;
import freemarker.template.Template;
import freemarker.template.TemplateException;
import freemarker.template.TemplateExceptionHandler;

/**
 * Mail Sender
 * @author easyproject.cn
 *
 * @since 1.0.0
 */
public class MailSender implements Sender {

	static Logger logger = LoggerFactory.getLogger(MailSender.class);

	static Configuration freemarkerConfiguration;
	
	/**
	 * 加载默认的 Freemarker 配置文件
	 */
	private void loadFreemarkerConfiguration(){
		// 加载默认的 Freemarker 配置文件
		if(freemarkerConfiguration==null){
			freemarkerConfiguration= new Configuration(Configuration.VERSION_2_3_23);
			freemarkerConfiguration.setTemplateExceptionHandler(TemplateExceptionHandler.RETHROW_HANDLER);
			freemarkerConfiguration.setDefaultEncoding("UTF-8");
			try {
				FileTemplateLoader ftl1 = new FileTemplateLoader(new File("template"));
				ClassTemplateLoader ctl = new ClassTemplateLoader(MailSender.class, "/template");
				TemplateLoader[] loaders = new TemplateLoader[] { ftl1,  ctl };
				MultiTemplateLoader mtl = new MultiTemplateLoader(loaders);
				freemarkerConfiguration.setTemplateLoader(mtl);
			} catch (IOException e) {
				e.printStackTrace();
			}
		}
	}


	private JavaMailSenderImpl sender = new JavaMailSenderImpl();

	public void send(MonitorConfiguration monitorConfig) {
		loadFreemarkerConfiguration();
		
		if(monitorConfig.getMailSender()==null||monitorConfig.getMailReceiver()==null||monitorConfig.getMailReceiver().length==0){
			logger.warn("You are not configuration Mail Sender or Receiver. Please Check your mail sender and recevier.");
			return;
		}

		try {

			logger.info(monitorConfig.getType().name() + "-" + monitorConfig.getName()
					+ " Monitor Error, start send mail to [" + Arrays.toString(monitorConfig.getMailReceiver()) + "]");
			sender.setUsername(monitorConfig.getMailSender());
			sender.setPassword(monitorConfig.getMailSenderPassword());
			sender.setHost(monitorConfig.getMailSenderHost());
			sender.setPort(monitorConfig.getMailSenderPort());

			Properties senderProperties = new Properties();
			senderProperties.setProperty("mail.smtp.auth", "true");
			if (monitorConfig.getMailSenderSsl()) {
				senderProperties.setProperty("mail.smtp.socketFactory.class", "javax.net.ssl.SSLSocketFactory");
			}
			sender.setJavaMailProperties(senderProperties);

			// SimpleMailMessage smm=new SimpleMailMessage();
			MimeMessage mm = sender.createMimeMessage();
			MimeMessageHelper smm = new MimeMessageHelper(mm, false, "utf-8");

			smm.setFrom(monitorConfig.getMailSender()); // 发件人，必须和Spring中配置的sender的名字保持一致
			smm.setTo(monitorConfig.getMailReceiver()); // 收件人
			smm.setReplyTo(monitorConfig.getMailSender()); // 回复地址，当点击回复的时候会回复到该地址
			smm.setSubject(monitorConfig.getMailSenderTitle()); // 主题

			Template template = freemarkerConfiguration.getTemplate(monitorConfig.getMailSenderTemplate());

			Map<String,Object> data = new HashMap<String,Object>(); // 通过Map传递动态数据
			// 动态数据的名字和模板标签中指定属性相匹配
			data.put("type", monitorConfig.getType().name());
			data.put("name", monitorConfig.getName());
			data.put("value", monitorConfig.getValue());
			data.put("stoptime", new Date(monitorConfig.getMonitorErrorTime()));
			data.put("monitorConfiguration", monitorConfig);

			// 解析模板并替换动态数据，产生最终的内容
			String text = FreeMarkerTemplateUtils.processTemplateIntoString(template, data);
			smm.setText(text, true); // 内容

			sender.send(mm); //发送
			
			monitorConfig.setMonitorError(true); // 服务器已经出错，等待恢复
			

			logger.info(monitorConfig.getType().name() + "-" + monitorConfig.getName()
					+ " Monitor Error, already send mail to [" + Arrays.toString(monitorConfig.getMailReceiver())
					+ "], errortime is: " + monitorConfig.getMonitorErrorTime()
					+ ", lasttime is: " + monitorConfig.getLastSenderTime());
		} catch (MailException e) {
			logger.error(monitorConfig.getType().name() + "-" + monitorConfig.getName()
					+ " send mail error, please check you sender configuration", e);
		} catch (MessagingException e) {
			logger.error(monitorConfig.getType().name() + "-" + monitorConfig.getName()
					+ " send mail error, please check you sender configuration", e);
		} catch (IOException e) {
			logger.error(monitorConfig.getType().name() + "-" + monitorConfig.getName()
					+ " send mail error, please check you sender configuration", e);
		} catch (TemplateException e) {
			logger.error(monitorConfig.getType().name() + "-" + monitorConfig.getName()
					+ " send mail template error, please check you mail.tpl configuration", e);
		} catch (Exception e) {
			logger.error(monitorConfig.getType().name() + "-" + monitorConfig.getName()
			+ " send mail error, please check you sender configuration", e);
		}

	}
	/**
	 * 设置默认的 freemarker 模板配置文件 
	 * @param freemarkerConfiguration 设置默认的 freemarker 模板配置文件 
	 */
	public static void setFreemarkerConfiguration(Configuration freemarkerConfiguration) {
		MailSender.freemarkerConfiguration = freemarkerConfiguration;
	}
}
