package cn.easyproject.easymonitor.job;

import java.io.IOException;
import org.quartz.DisallowConcurrentExecution;
import org.quartz.JobDataMap;
import org.quartz.JobExecutionContext;
import org.quartz.JobExecutionException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.scheduling.quartz.QuartzJobBean;
import cn.easyproject.easymonitor.configuration.MonitorConfiguration;
import cn.easyproject.easymonitor.monitor.validator.MonitorValidator;
import cn.easyproject.easymonitor.monitor.validator.ValidatorResult;
import cn.easyproject.easymonitor.sender.Sender;
import cn.easyproject.easymonitor.util.SpringUtil;

/**
 * EasyMonitor Job execute
 * @author easyproject.cn
 *
 * @since 1.0.0
 */
@DisallowConcurrentExecution
public class MonitorJob extends QuartzJobBean {
	static Logger logger = LoggerFactory.getLogger(MonitorJob.class);


	@Override
	protected void executeInternal(JobExecutionContext context) throws JobExecutionException {

		// 配置信息
		JobDataMap jobDataMap = context.getJobDetail().getJobDataMap();
		MonitorConfiguration configuration = (MonitorConfiguration) jobDataMap.get("configuration");
		// 必须是有效配置
		if(configuration==null){
			return;
		}

		
		
		// 监控验证器
		MonitorValidator monitorValidator=configuration.getMonitorValidator();
		ValidatorResult res=monitorValidator.validate(configuration);
		
		if(res==ValidatorResult.VALIDATION){
			// 程序恢复，则重置错误标识
			renewMonitor(configuration);
		}else if(res==ValidatorResult.INVALIDATION){
			// 程序验证未通过
			configuration.getNowFailure().incrementAndGet();
		}
		
		
		

		// If now monitor is error, don't sender
		if (configuration.isMonitorError()) {
			logger.info(configuration.getType().name() + "-" + configuration.getName() + " is error status.");
			// If need repeat error notify
			if (configuration.getMailSenderInterval() != null && configuration.getMailSenderInterval() > 0) {
				// Check repeat time
				if ((System.currentTimeMillis() - configuration.getLastSenderTime()) / 1000 < configuration
						.getMailSenderInterval()) {
					return;
				}
				logger.info(
						configuration.getType().name() + "-" + configuration.getName() + " repeate send error notify.");
			} else {
				return;
			}
		}

		logger.info(configuration.getType().name() + "-" + configuration.getName() + " failure count: "
				+ configuration.getNowFailure());

		// 判断是否达到最大错误次数，监控信息是否已经异常
		if (configuration.getNowFailure().get() >= configuration.getMaxfailure()) {
			configuration.getNowFailure().set(0);

			// 记录服务器出错时间
			long time = System.currentTimeMillis();
			if (configuration.getMonitorErrorTime() == 0) {
				configuration.setMonitorErrorTime(time);
			}
			configuration.setLastSenderTime(time); // 记录最后一次发送出错提现时间
			// 任务对象
			JobManager jobManager = SpringUtil.get("jobManager");
			
			// 暂停定时任务
			jobManager.pauseJob(context);

			// 发送失败消息 Senders
			
			for (Sender sender : configuration.getSenderImpl()) {
				sender.send(configuration);
			}
			
//			Sender sender = new MailSender();
//			sender.send(configuration, EasyMonitor.properties);

			// 执行cmd命令
			commandExecute(configuration);
			
			// 恢复任务
			jobManager.resumeJob(context);

		}

	}

	/**
	 * 执行命令
	 * @param configuration 配置对象
	 * @return 是否成功
	 */
	private boolean commandExecute(MonitorConfiguration configuration){
		boolean res=false;
		String[] cmds=configuration.getCmd();
		
		for (String cmd : cmds) {
			try {
				Process proc = Runtime.getRuntime().exec(cmd);
				proc.waitFor();
			} catch (IOException e) {
				logger.error(
						configuration.getType().name() + "-" + configuration.getName()
						+ " execute command ["+cmd+"] error.",
						e);
			} catch (InterruptedException e) {
				logger.error(
						configuration.getType().name() + "-" + configuration.getName()
						+ " execute command ["+cmd+"] error.",
						e);
			}
		}
		
		return res;
	}
	

	
	/**
	 * 程序恢复，则重置错误标识
	 * @param configuration 配置对象
	 */
	private void renewMonitor(MonitorConfiguration configuration){
		configuration.getNowFailure().set(0);
		configuration.setMonitorError(false);
		configuration.setMonitorErrorTime(0);
		configuration.setLastSenderTime(0);
	}

}
