package cn.donting.plugin.spring.boot.starter;

import cn.donting.plugin.spring.boot.starter.exception.PluginInstallException;
import cn.donting.plugin.spring.boot.starter.exception.PluginLoadException;
import cn.donting.plugin.spring.boot.starter.exception.PluginUpdateException;
import org.springframework.beans.factory.InitializingBean;
import org.springframework.context.event.ContextClosedEvent;
import java.io.IOException;
import java.util.List;

/**
 *  插件管理器
 * @author donting
 *  2020-05-03 下午3:59
 */
public interface PluginManger extends InitializingBean {
    /**
     * 安装插件
     *
     * @param bytes 插件byte
     * @return PluginWrapper
     * @throws IOException IOException
     * @throws PluginLoadException 插件加载异常
     * @throws PluginInstallException 插件安装异常
     */
    PluginWrapper install(byte[] bytes) throws IOException, PluginLoadException, PluginInstallException;

    /**
     * 更新插件
     *
     * @param bytes 插件byte
     * @param id    插件id
     * @return 插件处于running，返回PluginWrapper。停止状态返回null
     * @throws IOException IOException
     * @throws PluginLoadException 插件加载异常
     * @throws PluginUpdateException 插件更新异常
     */
    PluginWrapper update(byte[] bytes, String id) throws IOException, PluginLoadException, PluginUpdateException;

    /**
     * 获取插件安装信息
     *
     * @return 插件安装信息
     * @throws IOException IOException
     */
    List<PluginDbInfo> getInstallInfo() throws IOException;

    /**
     * 获取运行的插件
     *
     * @param pluginId 插件Id
     * @return PluginWrapper
     */
    PluginWrapper getRunningPlug(String pluginId);

    /**
     * 插件是否安装
     *
     * @param pluginId 插件Id
     * @return 是/否
     * @throws IOException IOException
     */
    boolean containsPlugInstall(String pluginId) throws IOException;

    /**
     * 是否运行
     *
     * @param pluginId 插件Id
     * @return 是/否
     */
    boolean containsPlugRun(String pluginId);

    /**
     * 获取正在运行中的插件
     *
     * @return 运行中的插件
     */
    List<PluginWrapper> getPluginRunning();



    /**
     * 主程序 启动时是否自动加载以安装的插件
     * @throws IOException IOException
     * @throws PluginLoadException 插件加载异常
     */
    void autoLoadInstallPlug() throws IOException, PluginLoadException;



    /**
     * 启动插件
     *
     * @param pluginId 插件Id
     * @throws IOException IOException
     * @throws PluginLoadException PluginLoadException
     */
    void start(String pluginId) throws IOException, PluginLoadException;

    /**
     * 停止插件
     *
     * @param pluginId 插件Id
     * @throws IOException IOException
     */
    void stop(String pluginId) throws IOException;

    /**
     * 卸载插件
     *
     * @param pluginId 插件Id
     * @throws IOException IOException
     */
    void uninstall(String pluginId) throws IOException;

    /**
     * 系统停止
     * @param contextClosedEvent ContextClosedEvent
     */
    void sysStop(ContextClosedEvent contextClosedEvent);
}
