package cn.dong144.mpgen.generator.impl;

import cn.dong144.mpgen.common.Column;
import cn.dong144.mpgen.common.Package;
import cn.dong144.mpgen.common.Table;
import cn.dong144.mpgen.generator.Generator;
import cn.dong144.mpgen.util.NameUtil;
import cn.hutool.core.collection.CollectionUtil;
import cn.hutool.db.Entity;
import cn.hutool.db.ds.simple.SimpleDataSource;
import cn.hutool.db.handler.EntityListHandler;
import cn.hutool.db.sql.SqlExecutor;
import cn.hutool.extra.template.Template;
import cn.hutool.extra.template.TemplateConfig;
import cn.hutool.extra.template.TemplateEngine;
import cn.hutool.extra.template.TemplateUtil;
import cn.hutool.json.JSONUtil;
import cn.hutool.setting.Setting;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import javax.sql.DataSource;
import java.io.File;
import java.sql.Connection;
import java.sql.SQLException;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;

/**
 * mybatis-plus代码生成器
 *
 * @author haidong
 */
public class MybatisPlusGenerator implements Generator {
    private static final Logger logger = LoggerFactory.getLogger(MybatisPlusGenerator.class);

    private Setting setting;
    private DataSource ds;
    private TemplateEngine engine;
    private String projectPath;

    public MybatisPlusGenerator(Setting setting){
        this.projectPath = setting.get("projectPath");
        this.setting = setting;
//        this.setting.autoLoad(true);
        this.ds = new SimpleDataSource(setting.get("dbUrl"), setting.get("username"), setting.get("password"));
        this.engine = TemplateUtil.createEngine(new TemplateConfig("templates", TemplateConfig.ResourceMode.CLASSPATH));
    }

    @Override
    public void execute() {
        String genTables = setting.get("genTables");
        List<String> genTableNames =  JSONUtil.parseArray(genTables).toList(String.class);
        if(CollectionUtil.isNotEmpty(genTableNames)){
            queryColumns(genTableNames);
        } else {
            genAllTable();
        }
    }

    private void genAllTable(){
        Connection conn = null;
        try {
            conn = ds.getConnection();
            List<Entity> tables = SqlExecutor.query(conn, "show tables", new EntityListHandler());
            List<String> tableNames = tables.stream().map(entity -> {
                Set<String> names = entity.getFieldNames();
                String[] nameList = names.toArray(new String[names.size()]);
                return entity.getStr(nameList[0]);
            }).collect(Collectors.toList());
            logger.info("tables====={}", JSONUtil.toJsonStr(tables));
            queryColumns(tableNames);
        } catch (SQLException e) {
            logger.error(e.getMessage(), e);
        }finally {
            if(conn != null){
                try {
                    conn.close();
                } catch (SQLException e) {
                    logger.error(e.getMessage(), e);
                }
            }
        }
    }

    private void queryColumns(List<String> tables){
        String database = setting.get("database");
        Connection conn = null;
        try {
            conn = ds.getConnection();
            for (String table : tables) {
                //select * from information_schema.COLUMNS where table_name = 'core_area' and TABLE_SCHEMA = 'bolepuhui';
                List<Entity> columns = SqlExecutor.query(conn,
                        "select COLUMN_NAME, DATA_TYPE from information_schema.COLUMNS where table_name = ? and TABLE_SCHEMA = ?",
                        new EntityListHandler(),
                        table, database);
                logger.info("table={}===columns====={}", table, JSONUtil.toJsonStr(columns));
                List<Column> colStrList = columns.stream().map(entity -> {
                    Column column = new Column();
                    String dbName = entity.getStr("COLUMN_NAME");
                    String dbType = entity.getStr("DATA_TYPE");
                    String javaTypeName = NameUtil.dbTypeToJavaType(dbName).getName();
                    String javaTypeSimpleName = NameUtil.dbTypeToJavaType(dbName).getSimpleName();
                    column.setDbName(dbName);
                    column.setJavaName(NameUtil.underlineToHump(dbName, false));
                    column.setDbTypeName(dbType);
                    column.setTypeName(javaTypeName);
                    column.setTypeSimpleName(javaTypeSimpleName);
                    return column;
                }).collect(Collectors.toList());
                parseTable(table,colStrList);
            }
            logger.info("执行完成===========================");
//            System.exit(0);
        } catch (SQLException e) {
            logger.error(e.getMessage(), e);
        }finally {
            if(conn != null){
                try {
                    conn.close();
                } catch (SQLException e) {
                    logger.error(e.getMessage(), e);
                }
            }
        }

    }

    private void parseTable(String tableName, List<Column> columns){
        Table table = new Table();
        table.setTableName(tableName);
        table.setEntityName(NameUtil.underlineToHump(tableName, true));
        if(CollectionUtil.isNotEmpty(columns)){
            List<Column> list = columns.stream().filter(column -> !"id".equals(column.getJavaName())).collect(Collectors.toList());
            table.setColumns(list);
        }
        genFile(table);
    }

    private void genFile(Table table){
        String codePath = setting.get("codePath");
        String xmlPath = setting.get("xmlPath");
        Package pack = new Package(setting.get("packageName"));
        Map context = new HashMap();
        context.putAll(setting);
        context.put("table", table);
        File xmlPathFile = new File(projectPath, xmlPath);
        File codePathFile = new File(projectPath, codePath);
        File packageFile = new File(codePathFile, pack.getPath());
        File entityFile = new File(packageFile, "entity");
        File daoFile = new File(packageFile, "mapper");
        File xmlFile = new File(xmlPathFile, "mapper");

        Template entityTemplate = engine.getTemplate("entity.ftl");
        entityTemplate.render(context, new File(entityFile, table.getEntityName() + ".java"));
        logger.info("生成" + table.getEntityName() + ".java文件===========");
        Template xmlTemplate = engine.getTemplate("mapper.ftl");
        xmlTemplate.render(context, new File(xmlFile, table.getEntityName() + "Mapper.xml"));
        logger.info("生成" + table.getEntityName() + "Mapper.xml文件===========");

        Template mapperTemplate = engine.getTemplate("dao.ftl");
        mapperTemplate.render(context, new File(daoFile, table.getEntityName() + "Mapper.java"));
        logger.info("生成" + table.getEntityName() + "Mapper.java文件===========");
    }
}
