/*
 * Copyright (c) 2025. 深圳市德为信息技术有限公司, 深圳市诠云科技有限公司 保留所有权利。
 *
 * 根据《中华人民共和国著作权法》、《计算机软件保护条例》及相关法律法规，
 * 本软件著作权归属于深圳市德为信息技术有限公司与深圳市诠云科技有限公司共同所有，
 * 任何单位或个人未经书面授权不得复制、修改、分发或用于商业用途。
 * （本声明适用于本项目所有源代码、资源配置文件及文档资料）
 */

package cn.devtech.dmp.musicservices.tidal.api.di

import cn.devtech.dmp.musicservices.tidal.api.AuthInterceptor
import cn.devtech.dmp.musicservices.tidal.api.service.TidalApi
import cn.devtech.dmp.musicservices.tidal.data.model.IncludedItem
import cn.devtech.dmp.musicservices.tidal.utils.IncludedItemDeserializer
import com.google.gson.GsonBuilder
import com.google.gson.Strictness
import dagger.Module
import dagger.Provides
import dagger.hilt.InstallIn
import dagger.hilt.components.SingletonComponent
import okhttp3.OkHttpClient
import okhttp3.logging.HttpLoggingInterceptor
import retrofit2.Retrofit
import retrofit2.converter.gson.GsonConverterFactory
import java.util.concurrent.TimeUnit
import javax.inject.Qualifier
import javax.inject.Singleton


/**
 * TIDAL 网络模块
 *
 * @author:袁培根
 * @date:  2025/9/18 15:18
 */
@Module
@InstallIn(SingletonComponent::class)
object NetworkModule {

    @Qualifier
    @Retention(AnnotationRetention.BINARY)
    annotation class TidalNetwork

    /**
     * 提供 OkHttpClient
     */
    @Provides
    @Singleton
    @TidalNetwork
    fun provideOkHttpClient(
        authInterceptor: AuthInterceptor,
    ): OkHttpClient {
        val loggingInterceptor = HttpLoggingInterceptor().apply {
            level = HttpLoggingInterceptor.Level.BODY
        }
        return OkHttpClient.Builder()
            .addInterceptor(authInterceptor)
            .addInterceptor(loggingInterceptor)
            .connectTimeout(30, TimeUnit.SECONDS)
            .readTimeout(30, TimeUnit.SECONDS)
            .writeTimeout(30, TimeUnit.SECONDS)
            .build()
    }

    /**
     * 提供 Retrofit 实例
     */
    @Provides
    @Singleton
    @TidalNetwork
    fun provideRetrofit(
        @TidalNetwork okHttpClient: OkHttpClient,
    ): Retrofit {
        return Retrofit.Builder()
            .baseUrl("https://openapi.tidal.com/v2/")
            .client(okHttpClient)
            .addConverterFactory(
                GsonConverterFactory.create(
                    GsonBuilder().registerTypeAdapter(
                        IncludedItem::class.java,
                        IncludedItemDeserializer()
                    ).setStrictness(Strictness.LENIENT).create()
                )
            )
            .build()
    }

    /**
     * 提供 TidalApi 实例
     */
    @Provides
    @Singleton
    fun provideTidalApi(@TidalNetwork retrofit: Retrofit): TidalApi {
        return retrofit.create(TidalApi::class.java)
    }

}