/*
 * Copyright (c) 2025. 深圳市德为信息技术有限公司, 深圳市诠云科技有限公司 保留所有权利
 *
 * 根据《中华人民共和国著作权法》、《计算机软件保护条例》及相关法律法规，
 * 本软件著作权归属于深圳市德为信息技术有限公司与深圳市诠云科技有限公司共同所有，
 * 任何单位或个人未经书面授权不得复制、修改、分发或用于商业用途
 * （本声明适用于本项目所有源代码、资源配置文件及文档资料）
 */

package cn.devtech.dmp.musicservices.tidal.api

import cn.devtech.dmp.common.utils.toTimeFormat
import cn.devtech.dmp.musicservices.tidal.api.service.TidalApi
import cn.devtech.dmp.musicservices.tidal.data.Album
import cn.devtech.dmp.musicservices.tidal.data.AlbumDetail
import cn.devtech.dmp.musicservices.tidal.data.AlbumList
import cn.devtech.dmp.musicservices.tidal.data.Artist
import cn.devtech.dmp.musicservices.tidal.data.ArtistDetail
import cn.devtech.dmp.musicservices.tidal.data.ArtistList
import cn.devtech.dmp.musicservices.tidal.data.Pagination
import cn.devtech.dmp.musicservices.tidal.data.Playlist
import cn.devtech.dmp.musicservices.tidal.data.PlaylistList
import cn.devtech.dmp.musicservices.tidal.data.PlaylistTracks
import cn.devtech.dmp.musicservices.tidal.data.ResourceType
import cn.devtech.dmp.musicservices.tidal.data.TrackDetail
import cn.devtech.dmp.musicservices.tidal.data.UserInfo
import cn.devtech.dmp.musicservices.tidal.data.model.AddTracksToPlaylistData
import cn.devtech.dmp.musicservices.tidal.data.model.AddTracksToPlaylistRequest
import cn.devtech.dmp.musicservices.tidal.data.model.AlbumResource
import cn.devtech.dmp.musicservices.tidal.data.model.ArtistResource
import cn.devtech.dmp.musicservices.tidal.data.model.ArtworkResource
import cn.devtech.dmp.musicservices.tidal.data.model.CollectionRequest
import cn.devtech.dmp.musicservices.tidal.data.model.CreatePlaylistAttributes
import cn.devtech.dmp.musicservices.tidal.data.model.CreatePlaylistData
import cn.devtech.dmp.musicservices.tidal.data.model.CreatePlaylistRequest
import cn.devtech.dmp.musicservices.tidal.data.model.MinimalistResources
import cn.devtech.dmp.musicservices.tidal.data.model.PlaylistItemMeta
import cn.devtech.dmp.musicservices.tidal.data.model.PlaylistResource
import cn.devtech.dmp.musicservices.tidal.data.model.RemoveTracksFromPlaylistData
import cn.devtech.dmp.musicservices.tidal.data.model.RemoveTracksFromPlaylistRequest
import cn.devtech.dmp.musicservices.tidal.data.model.TrackResource
import cn.devtech.dmp.musicservices.tidal.data.model.toArtistList
import cn.devtech.dmp.musicservices.tidal.data.model.toTrackDetailList
import cn.devtech.dmp.musicservices.tidal.data.model.toTrackList
import cn.devtech.dmp.musicservices.tidal.data.toCoverArtList
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.flow.Flow
import kotlinx.coroutines.flow.flow
import kotlinx.coroutines.flow.flowOn
import kotlinx.coroutines.flow.map
import javax.inject.Inject
import javax.inject.Singleton


/**
 * TIDAL 仓库
 *
 * @author:袁培根
 * @date:  2025/9/18 15:36
 */
@Singleton
class TidalRepository @Inject constructor(
    private val api: TidalApi,
) {

    /**
     * 获取专辑详情
     */
    fun getAlbumDetail(albumId: String): Flow<AlbumDetail> = flow {
        emit(api.getAlbum(albumId))
    }.map { response ->
        val albumData = response.data.attributes
        val includedData = response.included
        val artists = includedData.filterIsInstance<ArtistResource>().toArtistList()
        val coverArts = includedData.filterIsInstance<ArtworkResource>().toCoverArtList()
        val tracks = includedData.filterIsInstance<TrackResource>().toTrackList()
        val albumInfo = Album(
            id = response.data.id,
            title = albumData.title,
            barcodeId = albumData.barcodeId,
            numberOfItems = albumData.numberOfItems,
            duration = albumData.duration.toTimeFormat(),
            releaseDate = albumData.releaseDate,
            coverArts = coverArts,
            artists = artists
        )
        AlbumDetail(
            albumInfo = albumInfo,
            tracks = tracks
        )
    }.flowOn(Dispatchers.IO)

    /**
     * 获取歌单详情
     */
    fun getPlaylistDetail(playlistId: String): Flow<Playlist> = flow {
        emit(api.getPlaylist(playlistId))
    }.map { response ->
        val playlistData = response.data.attributes
        val includedData = response.included
        val coverArts = includedData.filterIsInstance<ArtworkResource>().toCoverArtList()
        Playlist(
            id = response.data.id,
            name = playlistData.name,
            description = playlistData.description,
            createdAt = playlistData.createdAt,
            lastModifiedAt = playlistData.lastModifiedAt,
            coverArts = coverArts
        )
    }.flowOn(Dispatchers.IO)

    /**
     * 获取用户推荐-myMixes
     */
    fun getMyMixes(userId: String): Flow<List<Playlist>> = flow {
        emit(api.getMyMixes(userId))
    }.map { response ->
        val includedData = response.included
        val artworks = includedData.filterIsInstance<ArtworkResource>()
        includedData.filterIsInstance<PlaylistResource>().map { playlist ->
            Playlist(
                id = playlist.id,
                name = playlist.attributes.name,
                description = playlist.attributes.description,
                createdAt = playlist.attributes.createdAt,
                lastModifiedAt = playlist.attributes.lastModifiedAt,
                coverArts = artworks.filter { it.id == playlist.relationships.coverArt.data?.firstOrNull()?.id }
                    .toCoverArtList()
            )
        }
    }.flowOn(Dispatchers.IO)

    /**
     * 获取用户推荐-discoveryMixes
     */
    fun getDiscoveryMixes(userId: String): Flow<List<Playlist>> = flow {
        emit(api.getDiscoveryMixes(userId))
    }.map { response ->
        val includedData = response.included
        val artworks = includedData.filterIsInstance<ArtworkResource>()
        includedData.filterIsInstance<PlaylistResource>().map { playlist ->
            Playlist(
                id = playlist.id,
                name = playlist.attributes.name,
                description = playlist.attributes.description,
                createdAt = playlist.attributes.createdAt,
                lastModifiedAt = playlist.attributes.lastModifiedAt,
                coverArts = artworks.filter { it.id == playlist.relationships.coverArt.data?.firstOrNull()?.id }
                    .toCoverArtList()
            )
        }
    }.flowOn(Dispatchers.IO)

    /**
     * 获取用户推荐-newArrivalMixes
     */
    fun getNewArrivalMixes(userId: String): Flow<List<Playlist>> = flow {
        emit(api.getNewArrivalMixes(userId))
    }.map { response ->
        val includedData = response.included
        val artworks = includedData.filterIsInstance<ArtworkResource>()
        includedData.filterIsInstance<PlaylistResource>().map { playlist ->
            Playlist(
                id = playlist.id,
                name = playlist.attributes.name,
                description = playlist.attributes.description,
                createdAt = playlist.attributes.createdAt,
                lastModifiedAt = playlist.attributes.lastModifiedAt,
                coverArts = artworks.filter { it.id == playlist.relationships.coverArt.data?.firstOrNull()?.id }
                    .toCoverArtList()
            )
        }
    }.flowOn(Dispatchers.IO)

    /**
     * 获取多个单曲
     */
    fun getTracks(trackIds: List<String>): Flow<List<TrackDetail>> = flow {
        emit(api.getTracks(trackIds = trackIds))
    }.map { response ->
        response.toTrackDetailList()
    }.flowOn(Dispatchers.IO)

    /**
     * 获取歌单中的单曲
     */
    fun getPlaylistTracks(playlistId: String, pageCursor: String? = null): Flow<PlaylistTracks> =
        flow {
            val playlistItemsResponse =
                api.getPlaylistItems(playlistId = playlistId, pageCursor = pageCursor)
            val pagination = Pagination(nextCursor = playlistItemsResponse.links.meta?.nextCursor)
            val trackIds = playlistItemsResponse.data.map { it.id }

            val tracksResponse = api.getTracks(trackIds = trackIds)
            val tracks = tracksResponse.toTrackDetailList()

            emit(
                PlaylistTracks(
                    tracks = tracks,
                    pagination = pagination
                )
            )
        }.flowOn(Dispatchers.IO)

    /**
     * 获取收藏的专辑
     */
    fun getCollectionAlbums(userId: String, pageCursor: String? = null): Flow<AlbumList> = flow {
        emit(api.getCollectionAlbums(userId = userId, pageCursor = pageCursor))
    }.map { response ->
        val pagination = Pagination(nextCursor = response.links.meta?.nextCursor)
        val includedData = response.included!!
        val artists = includedData.filterIsInstance<ArtistResource>().toArtistList()
        val artworks = includedData.filterIsInstance<ArtworkResource>()
        val albums = includedData.filterIsInstance<AlbumResource>().map { albumResource ->
            val albumData = albumResource.attributes
            val coverArts =
                artworks.filter { artwork -> albumResource.relationships.coverArt.data?.firstOrNull()?.id == artwork.id }
                    .toCoverArtList()
            val artists =
                artists.filter { artist -> albumResource.relationships.artists.data?.firstOrNull()?.id == artist.id }
            Album(
                id = albumResource.id,
                title = albumData.title,
                barcodeId = albumData.barcodeId,
                numberOfItems = albumData.numberOfItems,
                duration = albumData.duration.toTimeFormat(),
                releaseDate = albumData.releaseDate,
                coverArts = coverArts,
                artists = artists
            )
        }
        AlbumList(
            albums = albums,
            pagination = pagination
        )
    }.flowOn(Dispatchers.IO)

    /**
     * 收藏专辑
     */
    fun addAlbumsToCollection(userId: String, albumIds: List<String>): Flow<Unit> = flow {
        emit(
            api.addAlbumsToCollection(
                userId = userId,
                requestBody = CollectionRequest(
                    albumIds.map { MinimalistResources(id = it, type = ResourceType.ALBUM.type) }
                )
            )
        )
    }.flowOn(Dispatchers.IO)

    /**
     * 检查专辑是否被收藏
     * 该函数会递归地获取所有收藏的专辑，直到所有页面都被加载完毕或者找到匹配的专辑
     *
     * @param userId 用户的ID
     * @param albumId 要检查的专辑ID
     * @return 一个发出 Boolean 值的 Flow，如果找到则为 true，否则为 false
     */
    fun checkAlbumIsCollected(userId: String, albumId: String): Flow<Boolean> = flow {
        var nextCursor: String? = null
        var isCollected = false
        do {
            try {
                val collectionAlbumsResponse = api.getCollectionAlbums(
                    userId = userId,
                    include = null,
                    pageCursor = nextCursor
                )
                val pageContainsAlbum = collectionAlbumsResponse.data.any { it.id == albumId }
                if (pageContainsAlbum) {
                    isCollected = true
                    break
                }
                nextCursor = collectionAlbumsResponse.links.meta?.nextCursor
            } catch (_: Exception) {
                isCollected = false
                break
            }
        } while (nextCursor != null)
        emit(isCollected)
    }.flowOn(Dispatchers.IO)

    /**
     * 取消专辑收藏
     */
    fun removeAlbumsFromCollection(userId: String, albumIds: List<String>): Flow<Unit> = flow {
        emit(
            api.removeAlbumsFromCollection(
                userId = userId,
                requestBody = CollectionRequest(
                    albumIds.map { MinimalistResources(id = it, type = ResourceType.ALBUM.type) }
                )
            )
        )
    }.flowOn(Dispatchers.IO)

    /**
     * 获取收藏的歌单
     */
    fun getCollectionPlaylists(userId: String, pageCursor: String? = null): Flow<PlaylistList> =
        flow {
            emit(api.getCollectionPlaylists(userId = userId, pageCursor = pageCursor))
        }.map { response ->
            val pagination = Pagination(nextCursor = response.links.meta?.nextCursor)
            val includedData = response.included!!
            val artworks = includedData.filterIsInstance<ArtworkResource>()
            val playlists = includedData.filterIsInstance<PlaylistResource>().map {
                Playlist(
                    id = it.id,
                    name = it.attributes.name,
                    description = it.attributes.description,
                    createdAt = it.attributes.createdAt,
                    lastModifiedAt = it.attributes.lastModifiedAt,
                    coverArts = artworks.filter { artwork -> it.relationships.coverArt.data?.firstOrNull()?.id == artwork.id }
                        .toCoverArtList()
                )
            }
            PlaylistList(
                playlists = playlists,
                pagination = pagination
            )
        }.flowOn(Dispatchers.IO)

    /**
     * 收藏歌单
     */
    fun addPlaylistsToCollection(userId: String, playlistIds: List<String>): Flow<Unit> = flow {
        emit(
            api.addPlaylistsToCollection(
                userId = userId,
                requestBody = CollectionRequest(
                    playlistIds.map {
                        MinimalistResources(
                            id = it,
                            type = ResourceType.PLAYLIST.type
                        )
                    }
                )
            )
        )
    }.flowOn(Dispatchers.IO)

    /**
     * 检查歌单是否被收藏
     * 该函数会递归地获取所有收藏的歌单，直到所有页面都被加载完毕或者找到匹配的歌单
     *
     * @param userId 用户的ID
     * @param playlistId 要检查的歌单ID
     * @return 一个发出 Boolean 值的 Flow，如果找到则为 true，否则为 false
     */
    fun checkPlaylistIsCollected(userId: String, playlistId: String): Flow<Boolean> = flow {
        var nextCursor: String? = null
        var isCollected = false
        do {
            try {
                val collectionPlaylistsResponse = api.getCollectionPlaylists(
                    userId = userId,
                    include = null,
                    pageCursor = nextCursor
                )
                val pageContainsPlaylist =
                    collectionPlaylistsResponse.data.any { it.id == playlistId }
                if (pageContainsPlaylist) {
                    isCollected = true
                    break
                }
                nextCursor = collectionPlaylistsResponse.links.meta?.nextCursor
            } catch (_: Exception) {
                isCollected = false
                break
            }
        } while (nextCursor != null)
        emit(isCollected)
    }.flowOn(Dispatchers.IO)

    /**
     * 取消歌单收藏
     */
    fun removePlaylistsFromCollection(userId: String, playlistIds: List<String>): Flow<Unit> =
        flow {
            emit(
                api.removePlaylistsFromCollection(
                    userId = userId,
                    requestBody = CollectionRequest(
                        playlistIds.map {
                            MinimalistResources(
                                id = it,
                                type = ResourceType.PLAYLIST.type
                            )
                        }
                    )
                )
            )
        }.flowOn(Dispatchers.IO)

    /**
     * 获取收藏的单曲
     */
    fun getCollectionTracks(userId: String, pageCursor: String? = null): Flow<PlaylistTracks> =
        flow {
            val collectionResponse =
                api.getCollectionTracks(userId = userId, pageCursor = pageCursor)
            val pagination = Pagination(nextCursor = collectionResponse.links.meta?.nextCursor)
            val trackIds = collectionResponse.data.map { it.id }

            val tracksResponse = api.getTracks(trackIds = trackIds)
            val tracks = tracksResponse.toTrackDetailList()

            emit(
                PlaylistTracks(
                    tracks = tracks,
                    pagination = pagination
                )
            )
        }.flowOn(Dispatchers.IO)

    /**
     * 收藏单曲
     */
    fun addTracksToCollection(userId: String, trackIds: List<String>): Flow<Unit> = flow {
        emit(
            api.addTracksToCollection(
                userId = userId,
                requestBody = CollectionRequest(
                    trackIds.map { MinimalistResources(id = it, type = ResourceType.TRACK.type) }
                )
            )
        )
    }.flowOn(Dispatchers.IO)

    /**
     * 检查单曲是否被收藏
     * 该函数会递归地获取所有收藏的单曲，直到所有页面都被加载完毕或者找到匹配的单曲
     *
     * @param userId 用户的ID
     * @param trackId 要检查的单曲ID
     * @return 一个发出 Boolean 值的 Flow，如果找到则为 true，否则为 false
     */
    fun checkTrackIsCollected(userId: String, trackId: String): Flow<Boolean> = flow {
        var nextCursor: String? = null
        var isCollected = false
        do {
            try {
                val collectionTracksResponse = api.getCollectionTracks(
                    userId = userId,
                    include = null,
                    pageCursor = nextCursor
                )
                val pageContainsTrack = collectionTracksResponse.data.any { it.id == trackId }
                if (pageContainsTrack) {
                    isCollected = true
                    break
                }
                nextCursor = collectionTracksResponse.links.meta?.nextCursor
            } catch (_: Exception) {
                isCollected = false
                break
            }
        } while (nextCursor != null)
        emit(isCollected)
    }.flowOn(Dispatchers.IO)

    /**
     * 取消单曲收藏
     */
    fun removeTracksFromCollection(userId: String, trackIds: List<String>): Flow<Unit> = flow {
        emit(
            api.removeTracksFromCollection(
                userId = userId,
                requestBody = CollectionRequest(
                    trackIds.map { MinimalistResources(id = it, type = ResourceType.TRACK.type) }
                )
            )
        )
    }.flowOn(Dispatchers.IO)

    /**
     * 获取收藏的艺术家
     */
    fun getCollectionArtists(userId: String, pageCursor: String? = null): Flow<ArtistList> = flow {
        emit(api.getCollectionArtists(userId = userId, pageCursor = pageCursor))
    }.map { response ->
        val pagination = Pagination(nextCursor = response.links.meta?.nextCursor)
        val includedData = response.included!!
        val artworks = includedData.filterIsInstance<ArtworkResource>()
        val artists = includedData.filterIsInstance<ArtistResource>().map {
            ArtistDetail(
                artistInfo = Artist(
                    id = it.id,
                    name = it.attributes.name
                ),
                profileArts = artworks.filter { artwork -> it.relationships.profileArt.data?.firstOrNull()?.id == artwork.id }
                    .toCoverArtList()
            )
        }
        ArtistList(
            artists = artists,
            pagination = pagination
        )
    }.flowOn(Dispatchers.IO)

    /**
     * 收藏艺术家
     */
    fun addArtistsToCollection(userId: String, artistIds: List<String>): Flow<Unit> = flow {
        emit(
            api.addArtistsToCollection(
                userId = userId,
                requestBody = CollectionRequest(
                    artistIds.map { MinimalistResources(id = it, type = ResourceType.ARTIST.type) }
                )
            )
        )
    }.flowOn(Dispatchers.IO)

    /**
     * 检查艺术家是否被收藏
     * 该函数会递归地获取所有收藏的艺术家，直到所有页面都被加载完毕或者找到匹配的艺术家
     *
     * @param userId 用户的ID
     * @param artistId 要检查的艺术家ID
     * @return 一个发出 Boolean 值的 Flow，如果找到则为 true，否则为 false
     */
    fun checkArtistIsCollected(userId: String, artistId: String): Flow<Boolean> = flow {
        var nextCursor: String? = null
        var isCollected = false
        do {
            try {
                val collectionArtistsResponse = api.getCollectionArtists(
                    userId = userId,
                    include = null,
                    pageCursor = nextCursor
                )
                val pageContainsArtist = collectionArtistsResponse.data.any { it.id == artistId }
                if (pageContainsArtist) {
                    isCollected = true
                    break
                }
                nextCursor = collectionArtistsResponse.links.meta?.nextCursor
            } catch (_: Exception) {
                isCollected = false
                break
            }
        } while (nextCursor != null)
        emit(isCollected)
    }.flowOn(Dispatchers.IO)

    /**
     * 取消艺术家收藏
     */
    fun removeArtistsFromCollection(userId: String, artistIds: List<String>): Flow<Unit> = flow {
        emit(
            api.removeArtistsFromCollection(
                userId = userId,
                requestBody = CollectionRequest(
                    artistIds.map { MinimalistResources(id = it, type = ResourceType.ARTIST.type) }
                )
            )
        )
    }.flowOn(Dispatchers.IO)

    /**
     * 搜索专辑
     */
    fun searchAlbums(query: String, pageCursor: String? = null): Flow<AlbumList> = flow {
        emit(api.searchAlbums(query = query, pageCursor = pageCursor))
    }.map { response ->
        val pagination = Pagination(nextCursor = response.links.meta?.nextCursor)
        val includedData = response.included!!
        val artists = includedData.filterIsInstance<ArtistResource>().toArtistList()
        val artworks = includedData.filterIsInstance<ArtworkResource>()
        val albums = includedData.filterIsInstance<AlbumResource>().map { albumResource ->
            val albumData = albumResource.attributes
            val coverArts =
                artworks.filter { artwork -> albumResource.relationships.coverArt.data?.firstOrNull()?.id == artwork.id }
                    .toCoverArtList()
            val artists =
                artists.filter { artist -> albumResource.relationships.artists.data?.firstOrNull()?.id == artist.id }
            Album(
                id = albumResource.id,
                title = albumData.title,
                barcodeId = albumData.barcodeId,
                numberOfItems = albumData.numberOfItems,
                duration = albumData.duration.toTimeFormat(),
                releaseDate = albumData.releaseDate,
                coverArts = coverArts,
                artists = artists
            )
        }
        AlbumList(
            albums = albums,
            pagination = pagination
        )
    }.flowOn(Dispatchers.IO)

    /**
     * 搜索艺术家
     */
    fun searchArtists(query: String, pageCursor: String? = null): Flow<ArtistList> = flow {
        emit(api.searchArtists(query = query, pageCursor = pageCursor))
    }.map { response ->
        val pagination = Pagination(nextCursor = response.links.meta?.nextCursor)
        val includedData = response.included!!
        val artworks = includedData.filterIsInstance<ArtworkResource>()
        val artists = includedData.filterIsInstance<ArtistResource>().map {
            ArtistDetail(
                artistInfo = Artist(
                    id = it.id,
                    name = it.attributes.name
                ),
                profileArts = artworks.filter { artwork -> it.relationships.profileArt.data?.firstOrNull()?.id == artwork.id }
                    .toCoverArtList()
            )
        }
        ArtistList(
            artists = artists,
            pagination = pagination
        )
    }.flowOn(Dispatchers.IO)

    /**
     * 搜索歌单
     */
    fun searchPlaylists(query: String, pageCursor: String? = null): Flow<PlaylistList> =
        flow {
            emit(api.searchPlaylists(query = query, pageCursor = pageCursor))
        }.map { response ->
            val pagination = Pagination(nextCursor = response.links.meta?.nextCursor)
            val includedData = response.included!!
            val artworks = includedData.filterIsInstance<ArtworkResource>()
            val playlists = includedData.filterIsInstance<PlaylistResource>().map {
                Playlist(
                    id = it.id,
                    name = it.attributes.name,
                    description = it.attributes.description,
                    createdAt = it.attributes.createdAt,
                    lastModifiedAt = it.attributes.lastModifiedAt,
                    coverArts = artworks.filter { artwork -> it.relationships.coverArt.data?.firstOrNull()?.id == artwork.id }
                        .toCoverArtList()
                )
            }
            PlaylistList(
                playlists = playlists,
                pagination = pagination
            )
        }.flowOn(Dispatchers.IO)

    /**
     * 搜索单曲
     */
    fun searchTracks(query: String, pageCursor: String? = null): Flow<PlaylistTracks> =
        flow {
            val searchResponse = api.searchTracks(query = query, pageCursor = pageCursor)
            val pagination = Pagination(nextCursor = searchResponse.links.meta?.nextCursor)
            val trackIds = searchResponse.data.map { it.id }

            val tracksResponse = api.getTracks(trackIds = trackIds)
            val tracks = tracksResponse.toTrackDetailList()

            emit(
                PlaylistTracks(
                    tracks = tracks,
                    pagination = pagination
                )
            )
        }.flowOn(Dispatchers.IO)

    /**
     * 获取用户的歌单
     */
    fun getUserPlaylists(userId: String, pageCursor: String? = null): Flow<PlaylistList> = flow {
        emit(api.getUserPlaylists(userId = userId, pageCursor = pageCursor))
    }.map { response ->
        val pagination = Pagination(nextCursor = response.links.meta?.nextCursor)
        val includedData = response.included
        val artworks = includedData.filterIsInstance<ArtworkResource>()
        val playlists = response.data.map {
            Playlist(
                id = it.id,
                name = it.attributes.name,
                description = it.attributes.description,
                createdAt = it.attributes.createdAt,
                lastModifiedAt = it.attributes.lastModifiedAt,
                coverArts = artworks.filter { artwork -> it.relationships.coverArt.data?.firstOrNull()?.id == artwork.id }
                    .toCoverArtList()
            )
        }
        PlaylistList(
            playlists = playlists,
            pagination = pagination
        )
    }.flowOn(Dispatchers.IO)

    /**
     * 创建歌单
     */
    fun createPlaylist(
        name: String,
        description: String = "",
    ): Flow<Any> = flow {
        emit(
            api.createPlaylist(
                requestBody = CreatePlaylistRequest(
                    CreatePlaylistData(
                        type = ResourceType.PLAYLIST.type,
                        attributes = CreatePlaylistAttributes(
                            name = name,
                            description = description
                        )
                    )
                )
            )
        )
    }.flowOn(Dispatchers.IO)

    /**
     * 删除歌单
     */
    fun deletePlaylist(
        playlistId: String,
    ): Flow<Unit> = flow {
        emit(api.deletePlaylist(playlistId = playlistId))
    }.flowOn(Dispatchers.IO)

    /**
     * 添加单曲到歌单
     */
    fun addTracksToPlaylist(
        playlistId: String,
        trackIds: List<String>,
    ): Flow<Unit> = flow {
        emit(
            api.addTracksToPlaylist(
                playlistId = playlistId,
                requestBody = AddTracksToPlaylistRequest(
                    data = trackIds.map {
                        AddTracksToPlaylistData(
                            type = ResourceType.TRACK.type,
                            id = it
                        )
                    }
                )
            )
        )
    }.flowOn(Dispatchers.IO)

    /**
     * 从歌单中删除单曲
     */
    fun removeTracksFromPlaylist(
        playlistId: String,
        trackIds: List<String>,
    ): Flow<Unit> = flow {
        emit(
            api.removeTracksFromPlaylist(
                playlistId = playlistId,
                requestBody = RemoveTracksFromPlaylistRequest(
                    data = trackIds.map {
                        RemoveTracksFromPlaylistData(
                            type = ResourceType.TRACK.type,
                            id = it,
                            meta = PlaylistItemMeta(
                                itemId = playlistId
                            )
                        )
                    }
                )
            )
        )
    }.flowOn(Dispatchers.IO)

    /**
     * 获取艺术家的专辑
     */
    fun getArtistAlbums(artistId: String, pageCursor: String? = null): Flow<AlbumList> = flow {
        emit(api.getArtistAlbums(artistId = artistId, pageCursor = pageCursor))
    }.map { response ->
        val pagination = Pagination(nextCursor = response.links.meta?.nextCursor)
        val includedData = response.included!!
        val artists = includedData.filterIsInstance<ArtistResource>().toArtistList()
        val artworks = includedData.filterIsInstance<ArtworkResource>()
        val albums = includedData.filterIsInstance<AlbumResource>().map { albumResource ->
            val albumData = albumResource.attributes
            val coverArts =
                artworks.filter { artwork -> albumResource.relationships.coverArt.data?.firstOrNull()?.id == artwork.id }
                    .toCoverArtList()
            val artists =
                artists.filter { artist -> albumResource.relationships.artists.data?.firstOrNull()?.id == artist.id }
            Album(
                id = albumResource.id,
                title = albumData.title,
                barcodeId = albumData.barcodeId,
                numberOfItems = albumData.numberOfItems,
                duration = albumData.duration.toTimeFormat(),
                releaseDate = albumData.releaseDate,
                coverArts = coverArts,
                artists = artists
            )
        }
        AlbumList(
            albums = albums,
            pagination = pagination
        )
    }.flowOn(Dispatchers.IO)

    /**
     * 获取用户信息
     */
    fun getUser(): Flow<UserInfo> = flow {
        emit(api.getUser())
    }.map { response ->
        val userData = response.attributes
        UserInfo(
            id = response.id,
            userName = userData.username,
            email = userData.email,
            firstName = userData.firstName,
            lastName = userData.lastName
        )
    }
}