/*
 * Copyright (c) 2025. 深圳市德为信息技术有限公司, 深圳市诠云科技有限公司 保留所有权利
 *
 * 根据《中华人民共和国著作权法》、《计算机软件保护条例》及相关法律法规，
 * 本软件著作权归属于深圳市德为信息技术有限公司与深圳市诠云科技有限公司共同所有，
 * 任何单位或个人未经书面授权不得复制、修改、分发或用于商业用途
 * （本声明适用于本项目所有源代码、资源配置文件及文档资料）
 */

package cn.devtech.dmp.musicservices.tidal.auth

import cn.devtech.dmp.musicservices.tidal.TidalService
import cn.devtech.dmp.musicservices.tidal.cache.MemoryCacheManager
import cn.devtech.dmp.musicservices.tidal.data.UiState
import com.tidal.sdk.auth.model.AuthorizationError
import com.tidal.sdk.auth.model.CredentialsUpdatedMessage
import com.tidal.sdk.auth.model.LoginConfig
import com.tidal.sdk.auth.model.QueryParameter
import com.tidal.sdk.auth.model.TokenResponseError
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.Job
import kotlinx.coroutines.flow.MutableStateFlow
import kotlinx.coroutines.flow.StateFlow
import kotlinx.coroutines.flow.asStateFlow
import kotlinx.coroutines.flow.collectLatest
import kotlinx.coroutines.launch
import javax.inject.Inject
import javax.inject.Singleton


/**
 * TIDAL 授权管理器 `负责自动刷新和管理用户 Token，并向上层提供统一的授权状态和接口`
 *
 * @author:袁培根
 * @date:  2025/9/15 9:18
 */
@Singleton
class TidalAuth @Inject constructor(
    private val memoryCache: MemoryCacheManager,
) {
    private val authScope = CoroutineScope(Dispatchers.IO + Job())

    private val _accessTokenFlow = MutableStateFlow<String?>(null)
    val accessTokenFlow: StateFlow<String?> = _accessTokenFlow.asStateFlow()

    private val _userIdFlow = MutableStateFlow<String?>(null)
    val userIdFlow: StateFlow<String?> = _userIdFlow.asStateFlow()

    private val _getTokenState = MutableStateFlow<UiState>(UiState.Initial)
    val getTokenState: StateFlow<UiState> = _getTokenState.asStateFlow()

    private val _isLoggedInFlow = MutableStateFlow(false)
    val isLoggedInFlow: StateFlow<Boolean> = _isLoggedInFlow.asStateFlow()

    private val _qrCodeFlow = MutableStateFlow<String?>(null)
    val qrCodeFlow: StateFlow<String?> = _qrCodeFlow.asStateFlow()

    init {
        saveToken()
        onCredentialsUpdates()
    }

    /**
     * 获取授权URL
     */
    fun getAuthorizeUrl(): String {
        val loginConfig = LoginConfig(
            customParams = setOf(
                QueryParameter(
                    key = "appMode",
                    value = "android",
                )
            )
        )
        return TidalService.tidalAuth?.auth?.initializeLogin(
            redirectUri = TidalService.authConfig?.redirectUri ?: "",
            loginConfig = loginConfig
        ).toString()
    }

    /**
     * 处理URI登录回调
     */
    fun finalizeLogin(loginResponseUri: String) {
        authScope.launch {
            try {
                _getTokenState.value = UiState.Loading
                val result = TidalService.tidalAuth?.auth?.finalizeLogin(loginResponseUri)
                if (result?.isSuccess == true) {
                    _getTokenState.value = UiState.Success(null)
                    saveToken()
                } else {
                    _getTokenState.value = UiState.Error("Login failed")
                }
            } catch (e: Exception) {
                e.printStackTrace()
                _getTokenState.value = UiState.Error("Login failed")
            }
        }
    }

    /**
     * 启动二维码登录
     */
    fun startQrCodeLogin() {
        authScope.launch {
            val result = TidalService.tidalAuth?.auth?.initializeDeviceLogin()
            if (result?.isSuccess == true) {
                _qrCodeFlow.value = result.successData?.verificationUri
                val deviceLoginResult = TidalService.tidalAuth?.auth?.finalizeDeviceLogin(
                    result.successData?.deviceCode ?: ""
                )
                if (deviceLoginResult?.isSuccess == true) {
                    saveToken()
                }
            }
        }
    }

    /**
     * 保存Token
     */
    private fun saveToken() {
        authScope.launch {
            _accessTokenFlow.value = TidalService.credentialsProvider?.getCredentials()?.successData?.token
            memoryCache.setAccessToken(_accessTokenFlow.value ?: "")
            if (_accessTokenFlow.value != null) {
                _isLoggedInFlow.value = TidalService.credentialsProvider?.isUserLoggedIn() == true
            }
            _userIdFlow.value = TidalService.credentialsProvider?.getCredentials()?.successData?.userId
        }
    }

    /**
     * 登出并清除所有授权信息
     */
    fun logout() {
        authScope.launch {
            TidalService.tidalAuth?.auth?.logout()
            memoryCache.clearAccessToken()
            _accessTokenFlow.value = null
            _isLoggedInFlow.value = false
        }
    }

    /**
     * 监听TIDAL Token更新
     */
    private fun onCredentialsUpdates() {
        authScope.launch {
            TidalService.credentialsProvider?.bus?.collectLatest {
                when (it) {
                    is AuthorizationError -> {
                        logout()
                    }

                    is TokenResponseError -> {
                        logout()
                    }

                    is CredentialsUpdatedMessage -> {
                        saveToken()
                    }
                }
            }
        }
    }
}