/*
 * Copyright (c) 2025. 深圳市德为信息技术有限公司, 深圳市诠云科技有限公司 保留所有权利
 *
 * 根据《中华人民共和国著作权法》、《计算机软件保护条例》及相关法律法规，
 * 本软件著作权归属于深圳市德为信息技术有限公司与深圳市诠云科技有限公司共同所有，
 * 任何单位或个人未经书面授权不得复制、修改、分发或用于商业用途
 * （本声明适用于本项目所有源代码、资源配置文件及文档资料）
 */

package cn.devtech.dmp.musicservices.netdisk115.auth

import android.graphics.Bitmap
import cn.devtech.dmp.common.utils.QrcodeUtil
import cn.devtech.dmp.common.utils.formatTimestamp
import cn.devtech.dmp.musicservices.netdisk115.api.Netdisk115Repository
import cn.devtech.dmp.musicservices.netdisk115.cache.MemoryCacheManager
import cn.devtech.dmp.musicservices.netdisk115.cache.Netdisk115Preferences
import cn.devtech.dmp.musicservices.netdisk115.data.model.AccessToken
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.Deferred
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.Job
import kotlinx.coroutines.async
import kotlinx.coroutines.cancel
import kotlinx.coroutines.delay
import kotlinx.coroutines.flow.MutableStateFlow
import kotlinx.coroutines.flow.StateFlow
import kotlinx.coroutines.flow.asStateFlow
import kotlinx.coroutines.flow.catch
import kotlinx.coroutines.flow.first
import kotlinx.coroutines.isActive
import kotlinx.coroutines.launch
import kotlinx.coroutines.withTimeoutOrNull
import timber.log.Timber
import javax.inject.Inject
import javax.inject.Singleton


/**
 * 115 网盘授权管理器 `负责自动刷新和管理用户 Token，并向上层提供统一的授权状态和接口`
 *
 * @author:袁培根
 * @date:  2025/6/27 9:18
 */
@Singleton
class Netdisk115Auth @Inject constructor(
    private val preferences: Netdisk115Preferences,
    private val repository: Netdisk115Repository,
    private val memoryCacheManager: MemoryCacheManager,
) {
    private val authScope = CoroutineScope(Dispatchers.IO + Job())
    private var refreshJob: Job? = null
    private var loginJob: Deferred<Boolean>? = null

    private val _accessTokenFlow = MutableStateFlow<String?>(null)
    val accessTokenFlow: StateFlow<String?> = _accessTokenFlow.asStateFlow()

    private val _isLoggedInFlow = MutableStateFlow(false)
    val isLoggedInFlow: StateFlow<Boolean> = _isLoggedInFlow.asStateFlow()

    private val _qrCodeFlow = MutableStateFlow<Bitmap?>(null)
    val qrCodeFlow: StateFlow<Bitmap?> = _qrCodeFlow.asStateFlow()

    private val _isRefreshFinishedFlow = MutableStateFlow(false)
    val isRefreshFinishedFlow: StateFlow<Boolean> = _isRefreshFinishedFlow.asStateFlow()

    private companion object {
        private const val REFRESH_TIME_BUFFER_MS = 5 * 60 * 1000L
        private const val LOGIN_TIMEOUT_MS = 3 * 60 * 1000L
        private const val POLLING_INTERVAL_MS = 3000L
        private const val TAG = "AuthManager"
        private const val MIN_REFRESH_INTERVAL_MS = 5 * 60 * 1000L // Token 最小刷新间隔：5分钟
    }

    init {
        performInitialRefresh()
        observeAuthChanges()
    }

    /**
     * 发起二维码登录流程
     * 1. 调用 getDeviceCode 获取设备码和二维码信息
     * 2. 通过回调函数将二维码字符串传递给 UI 层进行展示
     * 3. 启动长轮询，调用 getQrcodeStatus 监控二维码状态
     * 4. 当状态码为 2 (已授权) 时，调用 getToken 获取最终的访问令牌并结束流程
     * @return 如果登录成功返回 `true`，如果用户取消、超时或发生错误则返回 `false`
     */
    suspend fun login(): Boolean {
        if (loginJob?.isActive == true) {
            Timber.tag(TAG).d("Login process is already active. Skipping redundant call.")
            return false
        }
        loginJob = authScope.async {
            // 获取设备码和二维码
            val deviceCodeResponse = repository.getDeviceCode().first()
            val deviceCode = deviceCodeResponse.data ?: run {
                Timber.tag(TAG).e("Failed to get device code. Response was null.")
                return@async false
            }
            val qrcodeBitmap = QrcodeUtil.generateQRCode(deviceCode.qrcode)
            _qrCodeFlow.value = qrcodeBitmap
            Timber.tag(TAG).d("QR Code received. Polling for user authorization...")

            // 在指定的超时时间内，轮询二维码状态
            val isAuthorized = withTimeoutOrNull(LOGIN_TIMEOUT_MS) {
                var authorized = false
                // 循环直到协程被取消或用户成功授权
                while (isActive && !authorized) {
                    repository.getQrcodeStatus(
                        uid = deviceCode.uid,
                        time = deviceCode.time,
                        sign = deviceCode.sign
                    ).catch { }.collect { statusResponse ->
                        val status = statusResponse.data
                        if (status == null) {
                            Timber.tag(TAG)
                                .w("Polling for QR status returned null data. Continuing.")
                            return@collect
                        }
                        when (status.status) {
                            2 -> { // 用户已授权
                                Timber.tag(TAG).d("User authorized successfully.")
                                authorized = true
                            }

                            1 -> Timber.tag(TAG)
                                .d("QR Code status: Scanned, waiting for confirmation.")

                            else -> { // 已过期、已取消等
                                Timber.tag(TAG)
                                    .e("QR Code expired or canceled. Status: ${status.status}")
                                cancel()
                            }
                        }
                    }
                    // 如果尚未授权，且协程未被取消，则等待一段时间再进行下一次轮询
                    if (isActive && !authorized) {
                        delay(POLLING_INTERVAL_MS)
                    }
                }
                authorized
            }

            // 如果 isAuthorized 不为 true，说明流程超时或被取消
            if (isAuthorized != true) {
                Timber.tag(TAG).e("Login failed due to timeout or cancellation.")
                return@async false
            }

            // 获取并保存 Access Token
            var tokenFetchSuccess = false
            repository.getToken(uid = deviceCode.uid).collect { tokenResponse ->
                val accessToken = tokenResponse.data
                if (accessToken != null) {
                    saveToken(accessToken)
                    Timber.tag(TAG).d("Token fetch and save successful. Login complete.")
                    tokenFetchSuccess = true
                } else {
                    Timber.tag(TAG).e("Failed to fetch access token after authorization.")
                    tokenFetchSuccess = false
                }
            }
            tokenFetchSuccess
        }
        return loginJob?.await() == true
    }

    /**
     * 保存Token
     */
    fun saveToken(accessToken: AccessToken) {
        _accessTokenFlow.value = accessToken.accessToken
        _isLoggedInFlow.value = true
        memoryCacheManager.setAccessToken(accessToken.accessToken)
        authScope.launch {
            preferences.updateAuthInfo(accessToken)
        }
    }

    /**
     * 登出并清除所有授权信息
     */
    suspend fun logout() {
        authScope.launch {
            // 取消任何已安排的刷新任务
            refreshJob?.cancel()
            refreshJob = null
            loginJob?.cancel()
            loginJob = null
            // 清除本地存储的授权和用户信息
            preferences.clearAuthInfo()
            Timber.tag(TAG).d("User logged out, all auth info cleared.")
        }.join()
    }

    /**
     * 在程序初始化时执行一次检查
     * 如果存在刷新令牌，则立即尝试刷新，以确保令牌的有效性
     */
    private fun performInitialRefresh() {
        authScope.launch {
            val initialAuthInfo = preferences.getAuthInfo().first()
            if (initialAuthInfo != null) {
                val lastRefreshTime = initialAuthInfo.authTime
                val currentTime = System.currentTimeMillis()
                if (currentTime - lastRefreshTime >= MIN_REFRESH_INTERVAL_MS) {
                    Timber.tag(TAG).d("Initial refresh: Interval check passed. Refreshing token.")
                    refreshToken(initialAuthInfo.accessToken.refreshToken)
                } else {
                    Timber.tag(TAG).d("Initial refresh: Skipped due to refresh interval limit. Last refresh was at ${lastRefreshTime.formatTimestamp()}")
                    _accessTokenFlow.value = initialAuthInfo.accessToken.accessToken
                    memoryCacheManager.setAccessToken(initialAuthInfo.accessToken.accessToken)
                    _isLoggedInFlow.value = true
                    _isRefreshFinishedFlow.value = true
                }
            } else {
                _isRefreshFinishedFlow.value = true
            }
        }
    }

    /**
     * 监听授权信息的变化 `每当授权信息（如：登录、刷新令牌后）发生变化时，此方法会重新计算过期时间并安排下一次的自动刷新任务`
     */
    private fun observeAuthChanges() {
        authScope.launch {
            preferences.getAuthInfo().collect { authInfo ->
                // 每当 authInfo 变化时，先取消上一个刷新任务
                refreshJob?.cancel()
                if (authInfo == null) {
                    Timber.tag(TAG).d("Auth info is null. No refresh scheduled.")
                    _isLoggedInFlow.value = false
                    return@collect
                }
                // 计算下一次需要刷新的时间点
                val expiresInMs = authInfo.accessToken.expiresIn * 1000
                val refreshDelayMs = (authInfo.authTime + expiresInMs) - System.currentTimeMillis() - REFRESH_TIME_BUFFER_MS

                if (refreshDelayMs > 0) {
                    Timber.tag(TAG).d("Scheduling token refresh in ${refreshDelayMs / 1000} seconds.")
                    _accessTokenFlow.value = authInfo.accessToken.accessToken
                    _isRefreshFinishedFlow.value = true
                    _isLoggedInFlow.value = true
                    memoryCacheManager.setAccessToken(authInfo.accessToken.accessToken)
                    refreshJob = launch {
                        delay(refreshDelayMs)
                        Timber.tag(TAG).d("Scheduled refresh time reached. Refreshing token now.")
                        refreshToken(authInfo.accessToken.refreshToken)
                    }
                } else {
                    // 如果 token 已经过期或即将过期（在缓冲区内），立即刷新
                    Timber.tag(TAG)
                        .d("Token is expired or within the refresh buffer. Refreshing immediately.")
                    refreshToken(authInfo.accessToken.refreshToken)
                }
            }
        }
    }

    /**
     * 调用仓库层执行刷新令牌的网络请求，并处理返回结果
     *
     * @param refreshToken 用于刷新访问令牌的 refresh token
     */
    private suspend fun refreshToken(refreshToken: String) {
        // 防止在已有刷新任务执行时重复调用
        if (refreshJob?.isActive == true && refreshJob?.isCompleted == false) {
            Timber.tag(TAG).d("Refresh job is already active. Skipping.")
            return
        }

        val lastRefreshTime = preferences.getAuthInfo().first()?.authTime ?: 0L
        val currentTime = System.currentTimeMillis()

        if (currentTime - lastRefreshTime < MIN_REFRESH_INTERVAL_MS) {
            Timber.tag(TAG).d("Token refresh skipped due to interval limit. Last refresh was at ${lastRefreshTime.formatTimestamp()}")
            if (!_isRefreshFinishedFlow.value && preferences.getAuthInfo().first() != null) {
                _isLoggedInFlow.value = true
                _isRefreshFinishedFlow.value = true
            }
            return
        }

        repository.refreshToken(refreshToken)
            .catch { e ->
                Timber.tag(TAG).e(e, "Error occurred during token refresh network call.")
                _isLoggedInFlow.value = false
                logout()
                _isRefreshFinishedFlow.value = true
            }
            .collect { response ->
                val newAccessToken = response.data
                if (response.state != 0 && newAccessToken is AccessToken) {
                    // 刷新成功，更新本地存储
                    saveToken(newAccessToken)
                    Timber.tag(TAG).d("Token refresh successful.")
                } else {
                    // 刷新失败，这通常意味着 refreshToken 已失效，需要用户重新登录
                    Timber.tag(TAG)
                        .e("Token refresh failed. Clearing all auth info to force re-login.")
                    logout()
                }
                _isRefreshFinishedFlow.value = true
            }
    }
}