/*
 * Copyright (c) 2025. 深圳市德为信息技术有限公司, 深圳市诠云科技有限公司 保留所有权利。
 *
 * 根据《中华人民共和国著作权法》、《计算机软件保护条例》及相关法律法规，
 * 本软件著作权归属于深圳市德为信息技术有限公司与深圳市诠云科技有限公司共同所有，
 * 任何单位或个人未经书面授权不得复制、修改、分发或用于商业用途。
 * （本声明适用于本项目所有源代码、资源配置文件及文档资料）
 */

package cn.devtech.dmp.musicservices.netdisk115.api

import cn.devtech.dmp.common.utils.PKCEUtil
import cn.devtech.dmp.musicservices.netdisk115.api.service.AuthApi
import cn.devtech.dmp.musicservices.netdisk115.api.service.NetdiskApi
import cn.devtech.dmp.musicservices.netdisk115.api.service.QrCodeApi
import cn.devtech.dmp.musicservices.netdisk115.Netdisk115Service
import cn.devtech.dmp.musicservices.netdisk115.data.OrderBy
import cn.devtech.dmp.musicservices.netdisk115.data.model.AccessToken
import cn.devtech.dmp.musicservices.netdisk115.data.model.BaseResponse
import cn.devtech.dmp.musicservices.netdisk115.data.model.DeviceCode
import cn.devtech.dmp.musicservices.netdisk115.data.model.FileInfoResponse
import cn.devtech.dmp.musicservices.netdisk115.data.model.FileListResponse
import cn.devtech.dmp.musicservices.netdisk115.data.model.OperationResponse
import cn.devtech.dmp.musicservices.netdisk115.data.model.PlayUrlResponse
import cn.devtech.dmp.musicservices.netdisk115.data.model.QrcodeStatus
import cn.devtech.dmp.musicservices.netdisk115.data.model.SearchFileResponse
import cn.devtech.dmp.musicservices.netdisk115.data.model.UpdateFileResult
import cn.devtech.dmp.musicservices.netdisk115.data.model.UserInfo
import com.google.gson.JsonObject
import kotlinx.coroutines.flow.Flow
import kotlinx.coroutines.flow.flow
import javax.inject.Inject
import javax.inject.Singleton


/**
 * 115网盘 仓库
 *
 * @author:袁培根
 * @date:  2025/6/26 18:28
 */
@Singleton
class Netdisk115Repository @Inject constructor(
    private val api: NetdiskApi,
    private val authApi: AuthApi,
    private val qrCodeApi: QrCodeApi,
) {
    private var codeVerifier = ""

    /**
     * 获取设备码
     */
    fun getDeviceCode(): Flow<BaseResponse<DeviceCode>> {
        codeVerifier = PKCEUtil.generateCodeVerifier()
        val codeChallenge = PKCEUtil.generateCodeChallenge(codeVerifier)
        return flow {
            emit(
                authApi.getDeviceCode(
                    clientId = Netdisk115Service.authConfig?.clientId ?: "",
                    codeChallenge = codeChallenge,
                    codeChallengeMethod = "sha256"
                )
            )
        }
    }

    /**
     * 获取Token
     */
    fun getToken(
        uid: String,
    ): Flow<BaseResponse<AccessToken>> = flow {
        emit(
            authApi.getToken(
                uid = uid,
                codeVerifier = codeVerifier,
            )
        )
    }

    /**
     * 刷新Token
     */
    fun refreshToken(
        refreshToken: String,
    ): Flow<BaseResponse<AccessToken>> = flow {
        emit(
            authApi.refreshToken(
                refreshToken = refreshToken,
            )
        )
    }

    /**
     * 获取二维码状态
     */
    fun getQrcodeStatus(
        uid: String,
        time: Long,
        sign: String,
    ): Flow<BaseResponse<QrcodeStatus>> = flow {
        emit(
            qrCodeApi.getQrcodeStatus(
                uid = uid,
                time = time,
                sign = sign
            )
        )
    }

    /**
     * 获取用户信息
     */
    fun getUserInfo(
    ): Flow<BaseResponse<UserInfo>> = flow {
        emit(api.getUserInfo())
    }

    /**
     * 获取文件列表
     */
    fun getFileList(
        parentId: String? = null,
        onlyAudioFile: Boolean = false,
        limit: Int,
        offset: Int,
        desc: Boolean = true,
        orderBy: OrderBy = OrderBy.USER_UTIME,
    ): Flow<FileListResponse> = flow {
        emit(
            api.getFileList(
                cid = parentId,
                type = if (onlyAudioFile) "3" else null,
                limit = limit,
                offset = offset,
                asc = if (desc) 0 else 1,
                orderBy = orderBy.name.lowercase(),
                dir = if (onlyAudioFile) 0 else 1,
                showDir = if (onlyAudioFile) 0 else 1,
            )
        )
    }

    /**
     * 获取文件信息
     */
    fun getFileInfo(
        fileId: String,
    ): Flow<FileInfoResponse> = flow {
        emit(api.getFileInfo(fileId))
    }

    /**
     * 更新文件
     */
    fun updateFile(
        fileId: String,
        fileName: String,
        star: String,
    ): Flow<OperationResponse<UpdateFileResult>> = flow {
        emit(api.updateFile(fileId, fileName, star))
    }

    /**
     * 删除文件
     */
    fun deleteFile(
        fileIds: String,
        parentId: String? = null,
    ): Flow<OperationResponse<List<String>>> = flow {
        emit(
            api.deleteFile(fileIds, parentId)
        )
    }

    /**
     * 搜索文件
     */
    fun searchFile(
        keyword: String,
        limit: Int,
        offset: Int,
    ): Flow<SearchFileResponse> = flow {
        emit(api.searchFile(keyword, limit, offset))
    }

    /**
     * 获取在线播放地址
     */
    fun getPlayUrl(
        pickCode: String,
    ): Flow<PlayUrlResponse> = flow {
        emit(api.getPlayUrl(pickCode))
    }

    /**
     * 获取文件下载地址
     */
    fun getDownloadUrl(
        pickCode: String,
    ): Flow<OperationResponse<JsonObject>> = flow {
        emit(
            api.getDownloadUrl(
                userAgent = Netdisk115Service.authConfig?.userAgent ?: "",
                pickCode = pickCode
            )
        )
    }

}