/*
 * Copyright (c) 2025. 深圳市德为信息技术有限公司, 深圳市诠云科技有限公司 保留所有权利。
 *
 * 根据《中华人民共和国著作权法》、《计算机软件保护条例》及相关法律法规，
 * 本软件著作权归属于深圳市德为信息技术有限公司与深圳市诠云科技有限公司共同所有，
 * 任何单位或个人未经书面授权不得复制、修改、分发或用于商业用途。
 * （本声明适用于本项目所有源代码、资源配置文件及文档资料）
 */

package cn.devtech.dmp.musicservices.dropbox.cache

import android.content.Context
import androidx.datastore.core.DataStore
import androidx.datastore.preferences.core.Preferences
import androidx.datastore.preferences.core.edit
import androidx.datastore.preferences.core.stringPreferencesKey
import androidx.datastore.preferences.preferencesDataStore
import cn.devtech.dmp.musicservices.dropbox.data.model.TokenResponse
import com.google.gson.Gson
import com.google.gson.reflect.TypeToken
import dagger.hilt.android.qualifiers.ApplicationContext
import kotlinx.coroutines.flow.Flow
import kotlinx.coroutines.flow.map
import javax.inject.Inject
import javax.inject.Singleton


/**
 * Dropbox 用户信息存储
 *
 * @author:袁培根
 * @date:  2025/7/25 9:06
 */
@Singleton
class DropboxPreferences @Inject constructor(
    @ApplicationContext private val context: Context,
) {
    private val Context.dataStore: DataStore<Preferences> by preferencesDataStore(name = "dropbox_preferences")

    private val gson = Gson()

    data class DropboxSettings(
        val authInfo: AuthInfo?,
        val searchHistory: List<String> = emptyList(),
    )

    data class AuthInfo(
        val tokenInfo: TokenResponse,
        val authTime: Long = System.currentTimeMillis(),
    )

    /**
     * 设置键
     */
    private object PreferencesKeys {
        val AUTH_INFO = stringPreferencesKey("auth_info")
        val SEARCH_HISTORY = stringPreferencesKey("search_history")
    }

    /**
     * 设置流
     */
    val dropboxSettings: Flow<DropboxSettings> =
        context.dataStore.data.map { preferences ->
            DropboxSettings(
                authInfo = preferences[PreferencesKeys.AUTH_INFO]?.let {
                    gson.fromJson(it, AuthInfo::class.java)
                },
                searchHistory = preferences[PreferencesKeys.SEARCH_HISTORY]?.let {
                    val listType = object : TypeToken<List<String>>() {}.type
                    gson.fromJson(it, listType)
                } ?: emptyList(),
            )
        }

    /**
     * 更新授权信息
     */
    suspend fun updateAuthInfo(accessToken: TokenResponse) {
        context.dataStore.edit { preferences ->
            preferences[PreferencesKeys.AUTH_INFO] = gson.toJson(
                AuthInfo(
                    tokenInfo = accessToken,
                    authTime = System.currentTimeMillis(),
                )
            )
        }
    }

    /**
     * 获取 Access Token
     */
    fun getAccessToken(): Flow<String?> = context.dataStore.data.map { preferences ->
        preferences[PreferencesKeys.AUTH_INFO]?.let {
            try {
                gson.fromJson(it, AuthInfo::class.java).tokenInfo.accessToken
            } catch (_: Exception) {
                null
            }
        }
    }

    /**
     * 获取授权信息
     */
    fun getAuthInfo(): Flow<AuthInfo?> = context.dataStore.data.map { preferences ->
        preferences[PreferencesKeys.AUTH_INFO]?.let {
            gson.fromJson(it, AuthInfo::class.java)
        }
    }

    /**
     * 清除授权信息
     */
    suspend fun clearAuthInfo() {
        context.dataStore.edit { preferences ->
            preferences.remove(PreferencesKeys.AUTH_INFO)
        }
    }

    /**
     * 更新搜索历史
     */
    suspend fun updateSearchHistory(searchHistory: List<String>) {
        context.dataStore.edit { preferences ->
            if (searchHistory.isEmpty()) {
                preferences.remove(PreferencesKeys.SEARCH_HISTORY)
            } else {
                preferences[PreferencesKeys.SEARCH_HISTORY] = gson.toJson(searchHistory)
            }
        }
    }
}