/*
 * Copyright (c) 2025. 深圳市德为信息技术有限公司, 深圳市诠云科技有限公司 保留所有权利
 *
 * 根据《中华人民共和国著作权法》、《计算机软件保护条例》及相关法律法规，
 * 本软件著作权归属于深圳市德为信息技术有限公司与深圳市诠云科技有限公司共同所有，
 * 任何单位或个人未经书面授权不得复制、修改、分发或用于商业用途
 * （本声明适用于本项目所有源代码、资源配置文件及文档资料）
 */

package cn.devtech.dmp.musicservices.dropbox.auth

import cn.devtech.dmp.common.utils.isNotNullOrEmptyOrBlank
import cn.devtech.dmp.musicservices.dropbox.api.DropboxRepository
import cn.devtech.dmp.musicservices.dropbox.cache.DropboxPreferences
import cn.devtech.dmp.musicservices.dropbox.cache.MemoryCacheManager
import cn.devtech.dmp.musicservices.dropbox.data.UiState
import cn.devtech.dmp.musicservices.dropbox.data.model.TokenResponse
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.Job
import kotlinx.coroutines.delay
import kotlinx.coroutines.flow.MutableStateFlow
import kotlinx.coroutines.flow.StateFlow
import kotlinx.coroutines.flow.asStateFlow
import kotlinx.coroutines.flow.catch
import kotlinx.coroutines.flow.first
import kotlinx.coroutines.launch
import timber.log.Timber
import javax.inject.Inject
import javax.inject.Singleton


/**
 * Dropbox 授权管理器 `负责自动刷新和管理用户 Token，并向上层提供统一的授权状态和接口`
 *
 * @author:袁培根
 * @date:  2025/7/25 9:18
 */
@Singleton
class DropboxAuth @Inject constructor(
    private val preferences: DropboxPreferences,
    private val repository: DropboxRepository,
    private val memoryCacheManager: MemoryCacheManager,
) {
    private val authScope = CoroutineScope(Dispatchers.IO + Job())
    private var refreshJob: Job? = null

    private val _getTokenState = MutableStateFlow<UiState>(UiState.Initial)
    val getTokenState: StateFlow<UiState> = _getTokenState.asStateFlow()

    private val _accessTokenFlow = MutableStateFlow<String?>(null)
    val accessTokenFlow: StateFlow<String?> = _accessTokenFlow.asStateFlow()

    private val _isLoggedInFlow = MutableStateFlow(false)
    val isLoggedInFlow: StateFlow<Boolean> = _isLoggedInFlow.asStateFlow()

    private val _isRefreshFinishedFlow = MutableStateFlow(false)
    val isRefreshFinishedFlow: StateFlow<Boolean> = _isRefreshFinishedFlow.asStateFlow()

    private companion object {
        private const val REFRESH_TIME_BUFFER_MS = 5 * 60 * 1000L
        private const val TAG = "AuthManager"
    }

    init {
        performInitialRefresh()
        observeAuthChanges()
    }

    /**
     * 获取授权URL
     */
    fun getAuthorizeUrl(): String {
        return repository.getAuthorizeUrl()
    }

    /**
     * 授权码换取Token
     */
    fun getToken(code: String) {
        authScope.launch {
            try {
                _getTokenState.value = UiState.Loading
                repository.getToken(code)
                    .catch { e ->
                        _getTokenState.value = UiState.Error(e.message ?: "Unknown error")
                        e.printStackTrace()
                    }
                    .collect { res ->
                        saveToken(res)
                        _getTokenState.value = UiState.Success(res)
                    }
            } catch (e: Exception) {
                e.printStackTrace()
                _getTokenState.value = UiState.Error(e.message ?: "Unknown error")
            }
        }
    }

    /**
     * 保存Token
     */
    fun saveToken(accessToken: TokenResponse) {
        _accessTokenFlow.value = accessToken.accessToken
        _isLoggedInFlow.value = true
        memoryCacheManager.setAccessToken(accessToken.accessToken)
        authScope.launch {
            preferences.updateAuthInfo(accessToken)
        }
    }

    /**
     * 登出并清除所有授权信息
     */
    suspend fun logout() {
        authScope.launch {
            // 取消任何已安排的刷新任务
            refreshJob?.cancel()
            refreshJob = null
            // 清除本地存储的授权和用户信息
            preferences.clearAuthInfo()
            Timber.tag(TAG).d("User logged out, all auth info cleared.")
        }.join()
    }

    /**
     * 在程序初始化时执行一次检查
     * 如果存在刷新令牌，则立即尝试刷新，以确保令牌的有效性
     */
    private fun performInitialRefresh() {
        authScope.launch {
            val initialAuthInfo = preferences.getAuthInfo().first()
            if (initialAuthInfo != null && initialAuthInfo.tokenInfo.refreshToken.isNotNullOrEmptyOrBlank()) {
                refreshToken(initialAuthInfo.tokenInfo)
            } else {
                _isRefreshFinishedFlow.value = true
            }
        }
    }

    /**
     * 监听授权信息的变化 `每当授权信息（如：登录、刷新令牌后）发生变化时，此方法会重新计算过期时间并安排下一次的自动刷新任务`
     */
    private fun observeAuthChanges() {
        authScope.launch {
            preferences.getAuthInfo().collect { authInfo ->
                // 每当 authInfo 变化时，先取消上一个刷新任务
                refreshJob?.cancel()
                if (authInfo == null) {
                    Timber.tag(TAG).d("Auth info is null. No refresh scheduled.")
                    _isLoggedInFlow.value = false
                    return@collect
                }
                // 计算下一次需要刷新的时间点
                val expiresInMs = authInfo.tokenInfo.expiresIn * 1000
                val refreshDelayMs =
                    (authInfo.authTime + expiresInMs) - System.currentTimeMillis() - REFRESH_TIME_BUFFER_MS

                if (refreshDelayMs > 0) {
                    Timber.tag(TAG)
                        .d("Scheduling token refresh in ${refreshDelayMs / 1000} seconds.")
                    _accessTokenFlow.value = authInfo.tokenInfo.accessToken
                    _isRefreshFinishedFlow.value = true
                    _isLoggedInFlow.value = true
                    memoryCacheManager.setAccessToken(authInfo.tokenInfo.accessToken)
                    refreshJob = launch {
                        delay(refreshDelayMs)
                        Timber.tag(TAG).d("Scheduled refresh time reached. Refreshing token now.")
                        refreshToken(authInfo.tokenInfo)
                    }
                } else {
                    // 如果 token 已经过期或即将过期（在缓冲区内），立即刷新
                    Timber.tag(TAG)
                        .d("Token is expired or within the refresh buffer. Refreshing immediately.")
                    refreshToken(authInfo.tokenInfo)
                }
            }
        }
    }

    /**
     * 调用仓库层执行刷新令牌的网络请求，并处理返回结果
     */
    private suspend fun refreshToken(tokenInfo: TokenResponse) {
        // 防止在已有刷新任务执行时重复调用
        if (refreshJob?.isActive == true && refreshJob?.isCompleted == false) {
            Timber.tag(TAG).d("Refresh job is already active. Skipping.")
            return
        }
        repository.refreshToken(tokenInfo.refreshToken)
            .catch {
                _isRefreshFinishedFlow.value = true
                logout()
            }
            .collect { response ->
                if (response.accessToken.isNotNullOrEmptyOrBlank()) {
                    // 刷新成功，更新本地存储
                    saveToken(
                        tokenInfo.copy(
                            accessToken = response.accessToken,
                            expiresIn = response.expiresIn
                        )
                    )
                    Timber.tag(TAG).d("Token refresh successful.")
                } else {
                    // 刷新失败，这通常意味着 refreshToken 已失效，需要用户重新登录
                    Timber.tag(TAG)
                        .e("Token refresh failed. Clearing all auth info to force re-login.")
                    logout()
                }
                _isRefreshFinishedFlow.value = true
            }
    }
}