/*
 * Copyright (c) 2025. 深圳市德为信息技术有限公司, 深圳市诠云科技有限公司 保留所有权利。
 *
 * 根据《中华人民共和国著作权法》、《计算机软件保护条例》及相关法律法规，
 * 本软件著作权归属于深圳市德为信息技术有限公司与深圳市诠云科技有限公司共同所有，
 * 任何单位或个人未经书面授权不得复制、修改、分发或用于商业用途。
 * （本声明适用于本项目所有源代码、资源配置文件及文档资料）
 */

package cn.devtech.dmp.musicservices.dropbox.api.di

import cn.devtech.dmp.musicservices.dropbox.api.AuthInterceptor
import cn.devtech.dmp.musicservices.dropbox.api.service.AuthApi
import cn.devtech.dmp.musicservices.dropbox.api.service.DropboxApi
import cn.devtech.dmp.musicservices.dropbox.data.constant.ApiUri
import com.google.gson.GsonBuilder
import com.google.gson.Strictness
import dagger.Module
import dagger.Provides
import dagger.hilt.InstallIn
import dagger.hilt.components.SingletonComponent
import okhttp3.OkHttpClient
import okhttp3.logging.HttpLoggingInterceptor
import retrofit2.Retrofit
import retrofit2.converter.gson.GsonConverterFactory
import java.util.concurrent.TimeUnit
import javax.inject.Qualifier
import javax.inject.Singleton


/**
 * Dropbox 网络模块
 *
 * @author:袁培根
 * @date:  2025/7/25 15:36
 */
@Module
@InstallIn(SingletonComponent::class)
object NetworkModule {

    @Qualifier
    @Retention(AnnotationRetention.BINARY)
    annotation class DropboxNetwork

    @Qualifier
    @Retention(AnnotationRetention.BINARY)
    annotation class DropboxNetworkAuth

    /**
     * 提供 OkHttpClient
     */
    @Provides
    @Singleton
    @DropboxNetwork
    fun provideOkHttpClient(
        authInterceptor: AuthInterceptor,
    ): OkHttpClient {
        val loggingInterceptor = HttpLoggingInterceptor().apply {
            level = HttpLoggingInterceptor.Level.BODY
        }
        return OkHttpClient.Builder()
            .addInterceptor(authInterceptor)
            .addInterceptor(loggingInterceptor)
            .connectTimeout(30, TimeUnit.SECONDS)
            .readTimeout(30, TimeUnit.SECONDS)
            .writeTimeout(30, TimeUnit.SECONDS)
            .build()
    }

    /**
     * 提供授权 OkHttpClient
     */
    @Provides
    @Singleton
    @DropboxNetworkAuth
    fun provideAuthOkHttpClient(
    ): OkHttpClient {
        val loggingInterceptor = HttpLoggingInterceptor().apply {
            level = HttpLoggingInterceptor.Level.BODY
        }
        return OkHttpClient.Builder()
            .addInterceptor(loggingInterceptor)
            .connectTimeout(30, TimeUnit.SECONDS)
            .readTimeout(30, TimeUnit.SECONDS)
            .writeTimeout(30, TimeUnit.SECONDS)
            .build()
    }

    /**
     * 提供 Retrofit 实例
     */
    @Provides
    @Singleton
    @DropboxNetwork
    fun provideRetrofit(
        @DropboxNetwork okHttpClient: OkHttpClient,
    ): Retrofit {
        return Retrofit.Builder()
            .baseUrl(ApiUri.BASE_URL)
            .client(okHttpClient)
            .addConverterFactory(
                GsonConverterFactory.create(
                    GsonBuilder().setStrictness(Strictness.LENIENT).create()
                )
            )
            .build()
    }

    /**
     * 提供授权 Retrofit 实例
     */
    @Provides
    @Singleton
    @DropboxNetworkAuth
    fun provideRetrofitWithAuth(
        @DropboxNetworkAuth okHttpClient: OkHttpClient,
    ): Retrofit {
        return Retrofit.Builder()
            .baseUrl(ApiUri.AUTH_URL)
            .client(okHttpClient)
            .addConverterFactory(
                GsonConverterFactory.create(
                    GsonBuilder().setStrictness(Strictness.LENIENT).create()
                )
            )
            .build()
    }

    /**
     * 提供 DropboxApi 实例
     */
    @Provides
    @Singleton
    fun provideDropboxApi(@DropboxNetwork retrofit: Retrofit): DropboxApi {
        return retrofit.create(DropboxApi::class.java)
    }

    /**
     * 提供 AuthApi 实例
     */
    @Provides
    @Singleton
    fun provideAuthApi(@DropboxNetworkAuth retrofit: Retrofit): AuthApi {
        return retrofit.create(AuthApi::class.java)
    }

}