/*
 * Copyright (c) 2025. 深圳市德为信息技术有限公司, 深圳市诠云科技有限公司 保留所有权利。
 *
 * 根据《中华人民共和国著作权法》、《计算机软件保护条例》及相关法律法规，
 * 本软件著作权归属于深圳市德为信息技术有限公司与深圳市诠云科技有限公司共同所有，
 * 任何单位或个人未经书面授权不得复制、修改、分发或用于商业用途。
 * （本声明适用于本项目所有源代码、资源配置文件及文档资料）
 */

package cn.devtech.dmp.musicservices.dropbox.api

import androidx.core.net.toUri
import cn.devtech.dmp.common.utils.PKCEUtil
import cn.devtech.dmp.musicservices.dropbox.DropboxService
import cn.devtech.dmp.musicservices.dropbox.api.service.AuthApi
import cn.devtech.dmp.musicservices.dropbox.api.service.DropboxApi
import cn.devtech.dmp.musicservices.dropbox.data.constant.ApiUri
import cn.devtech.dmp.musicservices.dropbox.data.constant.ApiUri.SCOPE
import cn.devtech.dmp.musicservices.dropbox.data.model.FileListResponse
import cn.devtech.dmp.musicservices.dropbox.data.model.OperationResponse
import cn.devtech.dmp.musicservices.dropbox.data.model.RefreshTokenResponse
import cn.devtech.dmp.musicservices.dropbox.data.model.SearchResultResponse
import cn.devtech.dmp.musicservices.dropbox.data.model.TemporaryLinkResponse
import cn.devtech.dmp.musicservices.dropbox.data.model.TokenResponse
import com.google.gson.JsonObject
import kotlinx.coroutines.flow.Flow
import kotlinx.coroutines.flow.flow
import java.util.UUID
import javax.inject.Inject
import javax.inject.Singleton


/**
 * Dropbox 仓库
 *
 * @author:袁培根
 * @date:  2025/7/25 18:28
 */
@Singleton
class DropboxRepository @Inject constructor(
    private val api: DropboxApi,
    private val authApi: AuthApi,
) {
    private var codeVerifier = ""

    /**
     * 获取授权URL
     */
    fun getAuthorizeUrl(): String {
        codeVerifier = PKCEUtil.generateCodeVerifier()
        val codeChallenge = PKCEUtil.generateCodeChallenge(codeVerifier)
        return ApiUri.AUTHORIZATION_URL.toUri()
            .buildUpon()
            .appendQueryParameter("client_id", DropboxService.authConfig?.clientId ?: "")
            .appendQueryParameter("redirect_uri", DropboxService.authConfig?.redirectUri ?: "")
            .appendQueryParameter("response_type", "code")
            .appendQueryParameter("token_access_type", "offline")
            .appendQueryParameter("code_challenge_method", "S256")
            .appendQueryParameter("code_challenge", codeChallenge)
            .appendQueryParameter("scope", SCOPE)
            .appendQueryParameter("state", UUID.randomUUID().toString())
            .build().toString()
    }

    /**
     * 获取Token
     */
    fun getToken(
        code: String,
    ): Flow<TokenResponse> = flow {
        emit(
            authApi.getToken(
                clientId = DropboxService.authConfig?.clientId ?: "",
                code = code,
                codeVerifier = codeVerifier,
                redirectUri = DropboxService.authConfig?.redirectUri ?: "",
            )
        )
    }

    /**
     * 刷新Token
     */
    fun refreshToken(
        refreshToken: String,
    ): Flow<RefreshTokenResponse> = flow {
        emit(
            authApi.refreshToken(
                clientId = DropboxService.authConfig?.clientId ?: "",
                refreshToken = refreshToken,
            )
        )
    }

    /**
     * 获取文件列表
     */
    fun getFileList(
        path: String = "",
        pageSize: Int,
    ): Flow<FileListResponse> = flow {
        emit(
            api.getFileList(
                JsonObject().apply {
                    addProperty("path", path)
                    addProperty("include_deleted", false)
                    addProperty("recursive", false)
                    addProperty("include_has_explicit_shared_members", false)
                    addProperty("include_mounted_folders", true)
                    addProperty("include_non_downloadable_files", true)
                    addProperty("limit", pageSize)
                }
            )
        )
    }

    /**
     * 获取下一页文件列表
     */
    fun getFileListContinue(
        cursor: String,
    ): Flow<FileListResponse> = flow {
        emit(
            api.getFileListContinue(
                JsonObject().apply {
                    addProperty("cursor", cursor)
                }
            )
        )
    }

    /**
     * 搜索文件
     */
    fun searchFile(
        query: String,
        path: String? = null,
        pageSize: Int = 30,
    ): Flow<SearchResultResponse> = flow {
        emit(
            api.searchFile(
                JsonObject().apply {
                    addProperty("query", query)
                    val optionsObject = JsonObject().apply {
                        path?.let { addProperty("path", it) }
                        addProperty("max_results", pageSize)
                    }
                    add("options", optionsObject)
                }
            )
        )
    }

    /**
     * 搜索文件下一页
     */
    fun searchFileContinue(
        cursor: String,
    ): Flow<SearchResultResponse> = flow {
        emit(
            api.searchFileContinue(
                JsonObject().apply {
                    addProperty("cursor", cursor)
                }
            )
        )
    }

    /**
     * 删除文件
     */
    fun deleteFile(
        path: String,
    ): Flow<OperationResponse> = flow {
        emit(api.deleteFile(JsonObject().apply { addProperty("path", path) }))
    }

    /**
     * 移动文件
     */
    fun moveFile(
        fromPath: String,
        toPath: String,
    ): Flow<OperationResponse> = flow {
        emit(
            api.moveFile(
                JsonObject().apply {
                    addProperty("from_path", fromPath)
                    addProperty("to_path", toPath)
                }
            )
        )
    }

    /**
     * 更新文件名
     */
    fun updateFileName(
        path: String,
        oldFileName: String,
        newFileName: String,
    ): Flow<OperationResponse> = flow {
        emit(
            api.moveFile(
                JsonObject().apply {
                    addProperty("from_path", path)
                    addProperty("to_path", path.replace(oldFileName, newFileName))
                }
            )
        )
    }

    /**
     * 获取文件的临时链接
     */
    fun getTemporaryLink(
        path: String,
    ): Flow<TemporaryLinkResponse> = flow {
        emit(
            api.getTemporaryLink(
                JsonObject().apply {
                    addProperty("path", path)
                }
            )
        )
    }
}