package cn.ctyun.apis;

import static cn.ctyun.apis.Apis.ENDPOINT_NAME;

import cn.ctyun.sdk.*;

import com.fasterxml.jackson.annotation.JsonProperty;

/**
 * 将 HaVip 从 ECS 实例上解绑，由于绑定是异步操作，在第一次请求后，并不会立即返回解绑结果，调用者在获取到解绑状态为 in_progress
 * 时，继续使用相同参数进行请求，获取最新的解绑结果，直到最后的解绑状态为 done 即可停止请求。
 */
public class CtvpcUnbindHavipApi {
    private final CtyunRequestTemplate template;
    private final CtyunClient client;

    public CtvpcUnbindHavipApi(CtyunClient client) {
        this.client = client;
        this.template =
                new CtyunRequestTemplate(
                        ENDPOINT_NAME, "POST", "/v4/vpc/havip/unbind", "application/json");
    }

    /**
     * 发起请求
     *
     * @param credential 用户信息
     * @param request 请求
     * @return 响应
     * @throws CtyunRequestException 异常
     */
    public UnbindHavipResponse send(Credential credential, UnbindHavipRequest request)
            throws CtyunRequestException {
        CtyunRequestBuilder builder = new CtyunRequestBuilder(template);
        builder.withCredential(credential);
        CtyunRequest ctReq = builder.build();
        ctReq.writeJson(request, template.contentType);
        CtyunResponse response = this.client.requestToEndpoint(ctReq);
        return response.parse(UnbindHavipResponse.class);
    }

    public static class UnbindHavipRequest {
        /** 客户端存根，用于保证订单幂等性, 长度 1 - 64 */
        @JsonProperty("clientToken")
        private String clientToken;

        /** 资源池ID */
        @JsonProperty("regionID")
        private String regionID;

        /** 绑定的实例类型，VM 表示虚拟机ECS, PM 表示裸金属, NETWORK 表示弹性 IP */
        @JsonProperty("resourceType")
        private String resourceType;

        /** 高可用虚IP的ID */
        @JsonProperty("haVipID")
        private String haVipID;

        /** 虚拟网卡ID */
        @JsonProperty("networkInterfaceID")
        private String networkInterfaceID;

        /** ECS示例ID，当 resourceType 为 VM / PM 时，必填 */
        @JsonProperty("instanceID")
        private String instanceID;

        /** 弹性IP ID，当 resourceType 为 NETWORK 时，必填 */
        @JsonProperty("floatingID")
        private String floatingID;

        public String getClientToken() {
            return this.clientToken;
        }

        public void setClientToken(String clientToken) {
            this.clientToken = clientToken;
        }

        public String getRegionID() {
            return this.regionID;
        }

        public void setRegionID(String regionID) {
            this.regionID = regionID;
        }

        public String getResourceType() {
            return this.resourceType;
        }

        public void setResourceType(String resourceType) {
            this.resourceType = resourceType;
        }

        public String getHaVipID() {
            return this.haVipID;
        }

        public void setHaVipID(String haVipID) {
            this.haVipID = haVipID;
        }

        public String getNetworkInterfaceID() {
            return this.networkInterfaceID;
        }

        public void setNetworkInterfaceID(String networkInterfaceID) {
            this.networkInterfaceID = networkInterfaceID;
        }

        public String getInstanceID() {
            return this.instanceID;
        }

        public void setInstanceID(String instanceID) {
            this.instanceID = instanceID;
        }

        public String getFloatingID() {
            return this.floatingID;
        }

        public void setFloatingID(String floatingID) {
            this.floatingID = floatingID;
        }
    }

    public static class UnbindHavipResponse {
        /** 返回状态码（800为成功，900为失败） */
        @JsonProperty("statusCode")
        private Integer statusCode;

        /** statusCode为900时的错误信息; statusCode为800时为success, 英文 */
        @JsonProperty("message")
        private String message;

        /** statusCode为900时的错误信息; statusCode为800时为成功, 中文 */
        @JsonProperty("description")
        private String description;

        /** statusCode为900时为业务细分错误码，三段式：product.module.code; statusCode为800时为SUCCESS */
        @JsonProperty("errorCode")
        private String errorCode;

        /** 解绑状态 */
        @JsonProperty("returnObj")
        private UnbindHavipReturnObjResponse returnObj;

        public Integer getStatusCode() {
            return this.statusCode;
        }

        public void setStatusCode(Integer statusCode) {
            this.statusCode = statusCode;
        }

        public String getMessage() {
            return this.message;
        }

        public void setMessage(String message) {
            this.message = message;
        }

        public String getDescription() {
            return this.description;
        }

        public void setDescription(String description) {
            this.description = description;
        }

        public String getErrorCode() {
            return this.errorCode;
        }

        public void setErrorCode(String errorCode) {
            this.errorCode = errorCode;
        }

        public UnbindHavipReturnObjResponse getReturnObj() {
            return this.returnObj;
        }

        public void setReturnObj(UnbindHavipReturnObjResponse returnObj) {
            this.returnObj = returnObj;
        }
    }

    public static class UnbindHavipReturnObjResponse {
        /** 解绑状态，取值 in_progress / done */
        @JsonProperty("status")
        private String status;

        /** 绑定状态提示信息 */
        @JsonProperty("message")
        private String message;

        public String getStatus() {
            return this.status;
        }

        public void setStatus(String status) {
            this.status = status;
        }

        public String getMessage() {
            return this.message;
        }

        public void setMessage(String message) {
            this.message = message;
        }
    }
}
