package cn.ctyun.sdk;

import com.fasterxml.jackson.annotation.JsonInclude;
import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.core.type.TypeReference;
import com.fasterxml.jackson.databind.DeserializationFeature;
import com.fasterxml.jackson.databind.MapperFeature;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.SerializationFeature;

public class JacksonUtils {

    private static final ObjectMapper OBJECT_MAPPER = new ObjectMapper();

    static {
        // 禁用在遇到空的Bean类时抛出异常
        OBJECT_MAPPER.configure(SerializationFeature.FAIL_ON_EMPTY_BEANS, false);

        // 禁用在遇到未知属性时抛出异常
        OBJECT_MAPPER.configure(DeserializationFeature.FAIL_ON_UNKNOWN_PROPERTIES, false);

        // 不序列化transient修饰的字段
        OBJECT_MAPPER.configure(MapperFeature.PROPAGATE_TRANSIENT_MARKER, true);

        // 不序列化null的对象
        OBJECT_MAPPER.setSerializationInclusion(JsonInclude.Include.NON_NULL);
    }

    /**
     * 获取objectMapper
     *
     * @return objectMapper实例
     */
    public static ObjectMapper getObjectMapperInstance() {
        return OBJECT_MAPPER;
    }

    /**
     * 对象转JSON字符串
     *
     * @param obj 任意对象
     * @return JSON字符串
     * @throws JsonProcessingException 转换异常时抛出
     */
    public static String toJson(Object obj) {
        try {
            return OBJECT_MAPPER.writeValueAsString(obj);
        } catch (JsonProcessingException e) {
            throw new RuntimeException(e);
        }
    }

    /**
     * JSON字符串转对象
     *
     * @param json  JSON字符串
     * @param clazz 目标对象类型
     * @return 转换后的对象
     * @throws JsonProcessingException 转换异常时抛出
     */
    public static <T> T fromJson(String json, Class<T> clazz) {
        try {
            return OBJECT_MAPPER.readValue(json, clazz);
        } catch (JsonProcessingException e) {
            throw new RuntimeException(e);
        }
    }

    /**
     * JSON字符串转泛型对象（支持List/Map等复杂类型）
     *
     * @param json          JSON字符串
     * @param typeReference 目标类型（如 new TypeReference<List<User>>() {}）
     * @return 转换后的对象
     * @throws JsonProcessingException 转换异常时抛出
     */
    public static <T> T fromJson(String json, TypeReference<T> typeReference) {
        try {
            return OBJECT_MAPPER.readValue(json, typeReference);
        } catch (JsonProcessingException e) {
            throw new RuntimeException(e);
        }
    }

    /**
     * 对象深度拷贝（通过JSON序列化实现）
     *
     * @param obj   源对象
     * @param clazz 目标类型
     * @return 拷贝后的新对象
     */
    public static <T> T deepCopy(Object obj, Class<T> clazz) {
        String json = toJson(obj);
        return fromJson(json, clazz);
    }
}
