package cn.ctyun.apis;

import static cn.ctyun.apis.Apis.ENDPOINT_NAME;

import cn.ctyun.sdk.*;

import com.fasterxml.jackson.annotation.JsonProperty;

/** 创建一个或者多个分布式缓存服务Redis基础版、增强版、经典版实例。 */
public class Dcs2CreateInstanceApi {
    private final CtyunRequestTemplate template;
    private final CtyunClient client;

    public Dcs2CreateInstanceApi(CtyunClient client) {
        this.client = client;
        this.template =
                new CtyunRequestTemplate(
                        ENDPOINT_NAME,
                        "POST",
                        "/v2/lifeCycleServant/createInstance",
                        "application/json");
    }

    /**
     * 发起请求
     *
     * @param credential 用户信息
     * @param request 请求
     * @return 响应
     * @throws CtyunRequestException 异常
     */
    public CreateInstanceResponse send(Credential credential, CreateInstanceRequest request)
            throws CtyunRequestException {
        CtyunRequestBuilder builder = new CtyunRequestBuilder(template);
        builder.withCredential(credential);
        CtyunRequest ctReq = builder.build();
        if (request.regionId != null) {
            ctReq.addHeader("regionId", request.regionId);
            request.regionId = null;
        }
        if (request.projectID != null) {
            ctReq.addHeader("projectID", request.projectID);
            request.projectID = null;
        }
        ctReq.writeJson(request, template.contentType);
        CtyunResponse response = this.client.requestToEndpoint(ctReq);
        return response.parse(CreateInstanceResponse.class);
    }

    public static class CreateInstanceRequest {
        /**
         * 资源池ID，使用此参数指定要创建实例的资源池。获取方法如下：<br>
         * 方法一：通过查看附录文档<a target="_blank" rel="noopener noreferrer"
         * href="https://www.ctyun.cn/document/10029420/11067697">分布式缓存服务Redis资源池</a>获取资源池ID。<br>
         * 方法二：可调用 <a target="_blank" rel="noopener noreferrer"
         * href="https://eop.ctyun.cn/ebp/ctapiDocument/search?sid=49&api=7830&isNormal=1&vid=270">查询可用的资源池</a>
         * 接口获取resPoolCode字段。
         */
        @JsonProperty("regionId")
        private String regionId;

        /**
         * 企业项目ID(默认值：0)，可以通过 <a target="_blank" rel="noopener noreferrer"
         * href="https://eop.ctyun.cn/ebp/ctapiDocument/search?sid=77&api=7246&data=114&isNormal=1&vid=107">查询企业项目列表</a>
         * 接口，使用recordList表id字段。
         */
        @JsonProperty("projectID")
        private String projectID;

        /**
         * 计费模式，取值：
         * <li>PrePaid：包年包月（需配合period使用）。
         * <li>PostPaid（默认）：按需计费。
         */
        @JsonProperty("chargeType")
        private String chargeType;

        /**
         * 购买时长，单位为月，取值：1~6,12,24,36。<br>
         * 说明：chargeType 取值为 PrePaid 时，本参数才生效且必须传参。
         */
        @JsonProperty("period")
        private Integer period;

        /**
         * 是否自动支付（仅对包周期实例有效），取值：
         * <li>true：自动付费。
         * <li>false（默认）：手动付费。<br>
         *     说明：选择为手动付费时，您需要在控制台的顶部菜单栏进入费用 > 待支付订单页面，找到目标订单进行支付。
         */
        @JsonProperty("autoPay")
        private Boolean autoPay;

        /** 购买数量（默认1），取值范围：1~100。 */
        @JsonProperty("size")
        private Integer size;

        /**
         * 是否启用自动续订，取值：
         * <li>true：开启。
         * <li>false（默认）：关闭 。
         */
        @JsonProperty("autoRenew")
        private Boolean autoRenew;

        /**
         * 自动续期购买时长，单位为月，取值：1~6,12,24,36。<br>
         * 说明：当 autoRenew 取值为 true 时，本参数才生效且必须传参。
         */
        @JsonProperty("autoRenewPeriod")
        private String autoRenewPeriod;

        /**
         * 版本类型，取值：
         * <li>BASIC（默认）：基础版。
         * <li>PLUS：增强版。
         * <li>Classic：经典版(白名单)。
         */
        @JsonProperty("version")
        private String version;

        /**
         * 实例类型，例如 StandardSingle 为单机版实例，详细信息请参见 <a target="_blank" rel="noopener noreferrer"
         * href="https://www.ctyun.cn/document/10029420/11030280" target="_blank">产品规格参数说明 </a>。
         */
        @JsonProperty("edition")
        private String edition;

        /**
         * 版本号，详细信息请参见 <a target="_blank" rel="noopener noreferrer"
         * href="https://www.ctyun.cn/document/10029420/11030280" target="_blank">产品规格参数说明 </a>。<br>
         * 当 version 取值为 BASIC时，版本号取值：
         * <li>5.0
         * <li>6.0（推荐）
         * <li>7.0<br>
         *     当 version 取值为 PLUS，版本号取值：
         * <li>6.0
         * <li>7.0<br>
         *     当 version 取值为 Classic时，版本号取值：
         * <li>2.8
         * <li>4.0
         * <li>5.0
         */
        @JsonProperty("engineVersion")
        private String engineVersion;

        /**
         * 主可用区名称，可以通过 <a target="_blank" rel="noopener noreferrer"
         * href="https://eop.ctyun.cn/ebp/ctapiDocument/search?sid=49&api=17764&isNormal=1&vid=270">查询可用区信息</a>
         * name字段获取，使用此参数指定要创建实例的主可用区。
         */
        @JsonProperty("zoneName")
        private String zoneName;

        /**
         * 备可用区名称（双/多副本建议填写），不填默认与主可用区相同，可以通过 <a target="_blank" rel="noopener noreferrer"
         * href="https://eop.ctyun.cn/ebp/ctapiDocument/search?sid=49&api=17764&isNormal=1&vid=270">查询可用区信息</a>
         * name字段获取。
         */
        @JsonProperty("secondaryZoneName")
        private String secondaryZoneName;

        /**
         * 主机类型，详细信息请参见 <a target="_blank" rel="noopener noreferrer"
         * href="https://www.ctyun.cn/document/10029420/11030280" target="_blank">产品规格参数说明 </a>。<br>
         * X86取值：
         * <li>S：通用型
         * <li>C：计算增强型
         * <li>M：内存型
         * <li>HS：海光通用型
         * <li>HC：海光计算增强型<br>
         *     ARM取值：
         * <li>KS：鲲鹏通用型
         * <li>KC：鲲鹏计算增强型<br>
         *     说明：当 version 取值为 Classic 时不可选，默认为S。
         */
        @JsonProperty("hostType")
        private String hostType;

        /**
         * 分片规格，单位为 GB，详细信息请参见 <a target="_blank" rel="noopener noreferrer"
         * href="https://www.ctyun.cn/document/10029420/11030280" target="_blank">产品规格参数说明 </a>。<br>
         * 当 version 取值为 BASIC，取值：
         * <li>1
         * <li>2
         * <li>4
         * <li>8
         * <li>16
         * <li>32
         * <li>64<br>
         *     当 version 取值为 PLUS时，取值：
         * <li>8
         * <li>16
         * <li>32
         * <li>64<br>
         *     其他版本无需填写此参数。
         */
        @JsonProperty("shardMemSize")
        private String shardMemSize;

        /**
         * 分片数，详细信息请参见 <a target="_blank" rel="noopener noreferrer"
         * href="https://www.ctyun.cn/document/10029420/11030280" target="_blank">产品规格参数说明 </a> ，
         * 取值范围如下：
         * <li>当 edition 取值为 DirectClusterSingle/DirectCluster/ClusterOriginalProxy时: 3~256。
         * <li>当 edition 取其他值时无需填写。
         */
        @JsonProperty("shardCount")
        private Integer shardCount;

        /**
         * 实例的存储容量，单位为 GB，仅当version 取值为 Classic 时填写，详细信息请参见 <a target="_blank" rel="noopener
         * noreferrer" href="https://www.ctyun.cn/document/10029420/11030280"
         * target="_blank">产品规格参数说明 </a>。
         */
        @JsonProperty("capacity")
        private String capacity;

        /**
         * 副本数,默认值：2，取值范围2~10，详细信息请参见 <a target="_blank" rel="noopener noreferrer"
         * href="https://www.ctyun.cn/document/10029420/11030280" target="_blank">产品规格参数说明 </a>。
         * <li>当 edition 取值为 OriginalMultipleReadLvs时必填。
         * <li>当 edition 取值为 StandardDual/DirectCluster/ClusterOriginalProxy时选填。<br>
         *     当 edition 取其他值时无需填写。
         */
        @JsonProperty("copiesCount")
        private Integer copiesCount;

        /**
         * 磁盘类型，详细信息请参见 <a target="_blank" rel="noopener noreferrer"
         * href="https://www.ctyun.cn/document/10029420/11030280" target="_blank">产品规格参数说明 </a>，取值：
         * <li>SSD（默认）：超高IO。
         * <li>SAS：高IO。 <br>
         *     说明：当 version 取值为 Classic时不可选，默认为SAS。
         */
        @JsonProperty("dataDiskType")
        private String dataDiskType;

        /**
         * 虚拟私有云ID。<br>
         * 获取方法如下：
         * <li>方法1：登录虚拟私有云服务的控制台界面，在虚拟私有云的详情页面查找ID。
         * <li>方法2：通过虚拟私有云服务的 <a target="_blank" rel="noopener noreferrer"
         *     href="https://eop.ctyun.cn/ebp/ctapiDocument/search?sid=18&api=8658&isNormal=1&vid=88">新查询VPC列表</a>
         *     API接口获取表 vpcs 的 vpcID 字段。
         */
        @JsonProperty("vpcId")
        private String vpcId;

        /**
         * 所在子网ID。<br>
         * 获取方法如下：
         * <li>方法1：登录虚拟私有云服务的控制台界面，在虚拟私有云的详情页面，点击下方指定的子网进入子网详情页面，查找子网ID。
         * <li>方法2：通过虚拟私有云服务的 <a target="_blank" rel="noopener noreferrer"
         *     href="https://eop.ctyun.cn/ebp/ctapiDocument/search?sid=18&api=8659&isNormal=1&vid=88">新查询子网列表</a>
         *     API接口获取表 subnets 的 subnetID 字段。
         */
        @JsonProperty("subnetId")
        private String subnetId;

        /**
         * 安全组ID。<br>
         * 获取方法如下：
         * <li>方法1：登录虚拟私有云服务的控制台界面，在左侧导航栏，选择“访问控制-安全组”选项，查看指定VPC下的安全组ID。
         * <li>方法2：通过虚拟私有云服务的 <a target="_blank" rel="noopener noreferrer"
         *     href="https://eop.ctyun.cn/ebp/ctapiDocument/search?sid=18&api=8661&isNormal=1&vid=88">新查询用户安全组列表</a>
         *     API接口获取表 securityGroupRuleList 的 securityGroupID字段。
         */
        @JsonProperty("secgroups")
        private String secgroups;

        /**
         * 实例名称。
         * <li>大小写字母开头。
         * <li>只能包含大小写字母、数字及分隔符(-)。
         * <li>大小写字母或数字结尾。
         * <li>长度4~40个字符。
         * <li>实例名称不可重复。
         */
        @JsonProperty("instanceName")
        private String instanceName;

        /**
         * 指定实例端口（默认值：6379），端口的可选范围可通过 <a target="_blank" rel="noopener noreferrer"
         * href="https://eop.ctyun.cn/ebp/ctapiDocument/search?sid=49&api=19894&isNormal=1&vid=270"
         * target="_blank">查询实例端口的可修改范围 </a>接口获取。
         */
        @JsonProperty("cacheServerPort")
        private Integer cacheServerPort;

        /**
         * 实例密码。
         * <li>长度8-26字符。
         * <li>必须同时包含大写字母、小写字母、数字、英文格式特殊符号(@%^*_+!$-=.) 中的三种类型。
         * <li>不能有空格。
         */
        @JsonProperty("password")
        private String password;

        public String getRegionId() {
            return this.regionId;
        }

        public void setRegionId(String regionId) {
            this.regionId = regionId;
        }

        public String getProjectID() {
            return this.projectID;
        }

        public void setProjectID(String projectID) {
            this.projectID = projectID;
        }

        public String getChargeType() {
            return this.chargeType;
        }

        public void setChargeType(String chargeType) {
            this.chargeType = chargeType;
        }

        public Integer getPeriod() {
            return this.period;
        }

        public void setPeriod(Integer period) {
            this.period = period;
        }

        public Boolean getAutoPay() {
            return this.autoPay;
        }

        public void setAutoPay(Boolean autoPay) {
            this.autoPay = autoPay;
        }

        public Integer getSize() {
            return this.size;
        }

        public void setSize(Integer size) {
            this.size = size;
        }

        public Boolean getAutoRenew() {
            return this.autoRenew;
        }

        public void setAutoRenew(Boolean autoRenew) {
            this.autoRenew = autoRenew;
        }

        public String getAutoRenewPeriod() {
            return this.autoRenewPeriod;
        }

        public void setAutoRenewPeriod(String autoRenewPeriod) {
            this.autoRenewPeriod = autoRenewPeriod;
        }

        public String getVersion() {
            return this.version;
        }

        public void setVersion(String version) {
            this.version = version;
        }

        public String getEdition() {
            return this.edition;
        }

        public void setEdition(String edition) {
            this.edition = edition;
        }

        public String getEngineVersion() {
            return this.engineVersion;
        }

        public void setEngineVersion(String engineVersion) {
            this.engineVersion = engineVersion;
        }

        public String getZoneName() {
            return this.zoneName;
        }

        public void setZoneName(String zoneName) {
            this.zoneName = zoneName;
        }

        public String getSecondaryZoneName() {
            return this.secondaryZoneName;
        }

        public void setSecondaryZoneName(String secondaryZoneName) {
            this.secondaryZoneName = secondaryZoneName;
        }

        public String getHostType() {
            return this.hostType;
        }

        public void setHostType(String hostType) {
            this.hostType = hostType;
        }

        public String getShardMemSize() {
            return this.shardMemSize;
        }

        public void setShardMemSize(String shardMemSize) {
            this.shardMemSize = shardMemSize;
        }

        public Integer getShardCount() {
            return this.shardCount;
        }

        public void setShardCount(Integer shardCount) {
            this.shardCount = shardCount;
        }

        public String getCapacity() {
            return this.capacity;
        }

        public void setCapacity(String capacity) {
            this.capacity = capacity;
        }

        public Integer getCopiesCount() {
            return this.copiesCount;
        }

        public void setCopiesCount(Integer copiesCount) {
            this.copiesCount = copiesCount;
        }

        public String getDataDiskType() {
            return this.dataDiskType;
        }

        public void setDataDiskType(String dataDiskType) {
            this.dataDiskType = dataDiskType;
        }

        public String getVpcId() {
            return this.vpcId;
        }

        public void setVpcId(String vpcId) {
            this.vpcId = vpcId;
        }

        public String getSubnetId() {
            return this.subnetId;
        }

        public void setSubnetId(String subnetId) {
            this.subnetId = subnetId;
        }

        public String getSecgroups() {
            return this.secgroups;
        }

        public void setSecgroups(String secgroups) {
            this.secgroups = secgroups;
        }

        public String getInstanceName() {
            return this.instanceName;
        }

        public void setInstanceName(String instanceName) {
            this.instanceName = instanceName;
        }

        public Integer getCacheServerPort() {
            return this.cacheServerPort;
        }

        public void setCacheServerPort(Integer cacheServerPort) {
            this.cacheServerPort = cacheServerPort;
        }

        public String getPassword() {
            return this.password;
        }

        public void setPassword(String password) {
            this.password = password;
        }
    }

    public static class CreateInstanceResponse {
        /** 响应信息。 */
        @JsonProperty("message")
        private String message;

        /**
         * 响应状态码。
         * <li>800：成功。
         * <li>900：失败。
         */
        @JsonProperty("statusCode")
        private Integer statusCode;

        /** 返回对象 */
        @JsonProperty("returnObj")
        private CreateInstanceReturnObjResponse returnObj;

        /** 请求 ID。 */
        @JsonProperty("requestId")
        private String requestId;

        /** 响应码，仅表示请求是否执行。 */
        @JsonProperty("code")
        private String code;

        /** 错误码，参见错误码说明。 */
        @JsonProperty("error")
        private String error;

        public String getMessage() {
            return this.message;
        }

        public void setMessage(String message) {
            this.message = message;
        }

        public Integer getStatusCode() {
            return this.statusCode;
        }

        public void setStatusCode(Integer statusCode) {
            this.statusCode = statusCode;
        }

        public CreateInstanceReturnObjResponse getReturnObj() {
            return this.returnObj;
        }

        public void setReturnObj(CreateInstanceReturnObjResponse returnObj) {
            this.returnObj = returnObj;
        }

        public String getRequestId() {
            return this.requestId;
        }

        public void setRequestId(String requestId) {
            this.requestId = requestId;
        }

        public String getCode() {
            return this.code;
        }

        public void setCode(String code) {
            this.code = code;
        }

        public String getError() {
            return this.error;
        }

        public void setError(String error) {
            this.error = error;
        }
    }

    public static class CreateInstanceReturnObjResponse {
        /** 错误信息。 */
        @JsonProperty("errorMessage")
        private String errorMessage;

        /** 是否成功提交 */
        @JsonProperty("submitted")
        private Boolean submitted;

        /** 订单ID */
        @JsonProperty("newOrderId")
        private String newOrderId;

        /** 订单号 */
        @JsonProperty("newOrderNo")
        private String newOrderNo;

        /** 总价。 */
        @JsonProperty("totalPrice")
        private Double totalPrice;

        public String getErrorMessage() {
            return this.errorMessage;
        }

        public void setErrorMessage(String errorMessage) {
            this.errorMessage = errorMessage;
        }

        public Boolean getSubmitted() {
            return this.submitted;
        }

        public void setSubmitted(Boolean submitted) {
            this.submitted = submitted;
        }

        public String getNewOrderId() {
            return this.newOrderId;
        }

        public void setNewOrderId(String newOrderId) {
            this.newOrderId = newOrderId;
        }

        public String getNewOrderNo() {
            return this.newOrderNo;
        }

        public void setNewOrderNo(String newOrderNo) {
            this.newOrderNo = newOrderNo;
        }

        public Double getTotalPrice() {
            return this.totalPrice;
        }

        public void setTotalPrice(Double totalPrice) {
            this.totalPrice = totalPrice;
        }
    }
}
